#include <akaxiso2/configuration.h>

#ifdef AKAXISO2_USE_EXPAT

#include "dtd_processor.h"
#include "processor.h"
#include "util.h"

#include <akaxiso2/imported/expat_parser.h>
#include <akaxiso2/imported/tricklib/tricklib.h>
#include "serializer.h"
#include "XMLSchema_xiso.h"

#include <iostream>
#include <fstream>

using namespace osx;

/**
 * xs:complexType -> complexTypeModel -> complexTypeModel_s0 -> typeDefParticle -> particle ->
 * explicitGroup -> 
 * particle  -> localElement, explicitGroup
 */

namespace {

  /* attribute */
  struct attdecl {
    attdecl() : required_(false) {}
    std::string elmname_;
    std::string name_;
    std::string type_;
    bool required_;
    std::string default_;
  };


  typedef std::multimap<std::string, attdecl> attdecl_map;
  typedef std::map<std::string, xs::topLevelComplexType> complexTypes;
  typedef std::set<std::string> value_elements;
  struct regex_type {
    xs::topLevelSimpleType simpleType_;
    std::string regex_;
    std::string name_;
  };

  typedef std::map<std::string, regex_type> regex_types;

  struct handler : aka::expat_dtd_handler {
    handler(const registry &reg)  : registry_(reg) { 
      aka::construct_element(topLevelElement_, xs::topLevelElement_leaf());

      lcp_translater_ = babel::manual_translate_engine<std::string, std::string>::
	create(babel::base_encoding::utf8, babel_util::default_encoding_);
    }
    virtual ~handler() { }
    virtual void elementDeclHandler(const XML_Char *name, XML_Content *model);
    virtual void attlistDeclHandler(const XML_Char *elname,
				    const XML_Char *attname,
				    const XML_Char *att_type,
				    const XML_Char *dflt,
				    int isrequired);
    virtual void startDoctypeDeclHandler(const XML_Char *doctypeName,
					 const XML_Char *sysid,
					 const XML_Char *pubid,
					 int has_internal_subset);
    virtual void endDoctypeDeclHandler();

    std::string translate(const std::string &str) {
      lcp_translater_.clear();
      lcp_translater_.translate(str);
      lcp_translater_.flush();
      return lcp_translater_.get_string();
    }

    xs::explicitGroup process_explicitGroup(XML_Content &model);
    void process_complexTypeModel_s0(xs::complexTypeModel_s0 &cdef, XML_Content &model);
    xs::localElement process_localElement(const std::string &name, 
					  const aka::qname &type,
					  XML_Content_Quant quant);

    std::string create_regex_type(const std::string &expression);
    const std::string &find_regex_name(const std::string &expression);


    const registry &registry_;
    xs::topLevelElement topLevelElement_;
    attdecl_map attdecls_;
    complexTypes complexTypes_;
    regex_types regex_types_;
    babel::bbl_translater<std::string, std::string> lcp_translater_;
    value_elements value_elements_;
  };

  void set_occurrence(long &minOccurs, long &maxOccurs, XML_Content_Quant quant) {
    switch (quant) {
    case XML_CQUANT_NONE:
      minOccurs = 1;
      maxOccurs = 1;
      break;
    case XML_CQUANT_OPT:
      minOccurs = 0;
      maxOccurs = 1;
      break;
    case XML_CQUANT_REP:
      minOccurs = 0;
      maxOccurs = aka::unbounded;
      break;
    case XML_CQUANT_PLUS:
      minOccurs = 1;
      maxOccurs = aka::unbounded;
      break;
    }
  }

  xs::localElement handler::process_localElement(const std::string &name, 
						 const aka::qname &type,
						 XML_Content_Quant quant) {
    xs::localElement le;
    aka::construct_element(le, xs::localElement_leaf());
    le.name_ = name;
    le.type_ = type;
    set_occurrence(le.minOccurs_, le.maxOccurs_, quant);
    return le;
  }

  xs::explicitGroup handler::process_explicitGroup(XML_Content &model) {

    xs::explicitGroup eg;
    aka::construct_element(eg, xs::explicitGroup_leaf());
    set_occurrence(eg.minOccurs_, eg.maxOccurs_, model.quant);
 
    xs::nestedParticle_leaf::moc moc(eg.nestedParticle_);
   
    for (size_t index = 0; index < model.numchildren; ++index) {
      XML_Content &child = model.children[index];

      switch (child.type) {
      case XML_CTYPE_EMPTY: {
	xs::localElement le = 
	  process_localElement(translate(child.name),
			       aka::qname(registry_.get_nill_type()),
			       child.quant);
	moc.push_back(le, "xs:element");
	break;
      }
      case XML_CTYPE_ANY: {
	xs::any any;
	aka2::construct_element(any, xs::any_leaf());
	any.minOccurs_ = 0;
	any.maxOccurs_ = aka::unbounded;
	moc.push_back(any, "xs:any");
	break;
      }
      case XML_CTYPE_MIXED: {
	assert(!"Must not reach here.");
	break;
      }
      case XML_CTYPE_NAME: {
	std::string name = translate(child.name);
	xs::localElement le = 
	  process_localElement(name, aka::qname(name), child.quant);
	moc.push_back(le, "xs:element");
	break;
      }
      case XML_CTYPE_CHOICE: {
	xs::explicitGroup choice = process_explicitGroup(child);
	set_occurrence(choice.minOccurs_, choice.maxOccurs_, child.quant);
	moc.push_back(choice, "xs:choice");
	break;
      }
      case XML_CTYPE_SEQ: {
	xs::explicitGroup sequence = process_explicitGroup(child);
	set_occurrence(sequence.minOccurs_, sequence.maxOccurs_, child.quant);
	moc.push_back(sequence, "xs:sequence");
	break;
      }
      }
    }
    return eg;
  }


  void handler::process_complexTypeModel_s0(xs::complexTypeModel_s0 &cdef, 
					    XML_Content &model) {
    xs::typeDefParticle_leaf::moc moc(cdef.typeDefParticle_);

    switch (model.type) {
    case XML_CTYPE_EMPTY: {
      // Empty complexType.  Nothing to define.
      break;
    }
    case XML_CTYPE_ANY: {
      xs::explicitGroup eg;
      aka::construct_element(eg, xs::explicitGroup_leaf());

      xs::any any;
      aka2::construct_element(any, xs::any_leaf());
      any.minOccurs_ = 0;
      any.maxOccurs_ = aka::unbounded;
      xs::nestedParticle_leaf::moc(eg.nestedParticle_).push_back(any, "xs:any");
      moc.push_back(eg, "xs:sequence");
      break;
    }
    case XML_CTYPE_MIXED: {
      assert(model.numchildren != 0);
      XML_Content choice_model = model;
      choice_model.type = XML_CTYPE_CHOICE;
      process_complexTypeModel_s0(cdef, choice_model);
      break;
    }
    case XML_CTYPE_NAME: {
      std::string name = translate(model.name);
      xs::localElement le = process_localElement(name, aka::qname(name), model.quant);
      moc.push_back(le, "xs:element");
      break;
    }
    case XML_CTYPE_CHOICE: {
      xs::explicitGroup choice = process_explicitGroup(model);
      moc.push_back(choice, "xs:choice");
      break;
    }
    case XML_CTYPE_SEQ: {
      xs::explicitGroup sequence = process_explicitGroup(model);
      moc.push_back(sequence, "xs:sequence");
      break;
    }
    }
  }
  
  void handler::elementDeclHandler(const XML_Char *name, XML_Content *model) {
    xs::topLevelComplexType tlc;
    aka::construct_element(tlc, xs::topLevelComplexType_leaf());
    tlc.name_ = translate(name);

    xs::complexTypeModel_s0 cdef;
    aka::construct_element(cdef, xs::complexTypeModel_s0_leaf());
    
    if (model->type == XML_CTYPE_MIXED && model->numchildren == 0) {
      // if it has attributes, assigned to simpleContent.
      // if it does not have attributes, reduced to a simpleType.
      // Check attribute at last.  Here the entry is registered.
      value_elements_.insert(name);
    }
    else {
      if (model->type == XML_CTYPE_MIXED)
	tlc.mixed_ = true;
      process_complexTypeModel_s0(cdef, *model);
      xs::complexTypeModel_leaf::moc(tlc.complexTypeModel_).push_back(cdef, "&s0");
      complexTypes_.insert(complexTypes::value_type(name, tlc));
    }
  }
  
  void handler::attlistDeclHandler(const XML_Char *elname,
				   const XML_Char *attname,
				   const XML_Char *att_type,
				   const XML_Char *dflt,
				   int isrequired) {
    attdecl adecl;
    adecl.elmname_ = translate(elname);
    adecl.name_ = translate(attname);
    adecl.required_ = isrequired != 0;
    if (dflt != 0)
      adecl.default_ = dflt;

    std::string type = translate(att_type);

    if (type == "CDATA")
      adecl.type_ = "xs:string";
    else if (type == "ID")
      adecl.type_ = "xs:ID";
    else if (type == "IDREF")
      adecl.type_ = "xs:IDREF";
    else if (type == "IDREFS")
      adecl.type_ = "xs:IDREFS";
    else if (type == "NMTOKEN")
      adecl.type_ = "xs:NMTOKEN";
    else if (type == "NMTOKENS")
      adecl.type_ = "xs:NMTOKENS";
    else if (type == "ENTITY")
      adecl.type_ = "xs:ENTITY";
    else if (type == "ENTITIES")
      adecl.type_ = "xs:ENTITIES";
    else if (type == "NOTATION")
      adecl.type_ = "xs:NOTATION";
    else
      adecl.type_ = create_regex_type(type);

    attdecls_.insert(attdecl_map::value_type(adecl.elmname_, adecl));

  }


  void handler::startDoctypeDeclHandler(const XML_Char *doctypeName,
					const XML_Char *sysid,
					const XML_Char *pubid,
					int has_internal_subset) {
    aka::construct_element(topLevelElement_, xs::topLevelElement_leaf());
    topLevelElement_.name_ = translate(doctypeName);
    topLevelElement_.type_ = aka::qname(topLevelElement_.name_);
  }
  
  void handler::endDoctypeDeclHandler() {

  }

  std::string handler::create_regex_type(const std::string &expression) {

    regex_types::iterator it = regex_types_.find(expression);
    if (it != regex_types_.end())
      return it->second.name_;

    regex_type regex;

    std::ostringstream ostm;
    ostm << "pattern" << regex_types_.size();
    regex.name_ = ostm.str();

    regex.regex_ = expression;

    // xs:topLevelSimpleType:member -> &simpleDerivation:choice ->
    // xs:restriction::simpleRestrictionModel_ -> &simpleRestrictionModel::facets_ ->
    // &facets:choice -> xs:pattern(noFixedFacet).
    xs::pattern facet;
    aka::construct_element(facet, xs::pattern_leaf());
    facet.value_ = expression;

    xs::restriction restriction;
    aka::construct_element(restriction.simpleRestrictionModel_, 
			   xs::simpleRestrictionModel_leaf());
    restriction.base_ = aka::qname("xs:string");
    xs::facets_leaf::moc(restriction.simpleRestrictionModel_.facets_)
      .push_back(facet, "xs:pattern");
    
    aka::construct_element(regex.simpleType_, xs::topLevelSimpleType_leaf());
    xs::simpleDerivation_leaf::moc(regex.simpleType_.simpleDerivation_).
      push_back(restriction, "xs:restriction");
    
    regex.simpleType_.name_ = regex.name_;
    regex_types_.insert(regex_types::value_type(regex.regex_, regex));
    return regex.name_;
  }
  
  const std::string &handler::find_regex_name(const std::string &expression) {
    regex_types::iterator it = regex_types_.find(expression);
    assert(it != regex_types_.end());
    return it->second.name_;
  }
  
  
  bool create_attributes(attdecl_map::iterator attrbegin, 
			 attdecl_map::iterator attrend,
			 xs::attrDecls_c0_leaf::moc &at_moc) {
    bool has_xml_import = false;

    for (attdecl_map::iterator ait = attrbegin; ait != attrend; ++ait) {
      attdecl &adecl = ait->second;
      
      xs::attribute attr;
      aka::construct_element(attr, xs::attribute_leaf());
      if (adecl.required_) {
	if (!adecl.default_.empty())
	  attr.fixed_ = adecl.default_;
	else
	  attr.use_ = "required";
      }

      if ((adecl.name_ == "xml:lang") ||
	  (adecl.name_ == "xml:space") ||
	  (adecl.name_ == "xml:base")) {
	attr.ref_ = aka::qname(adecl.name_);
	has_xml_import = true;
      }
      else {
	attr.name_ = adecl.name_;
	attr.type_ = aka::qname(adecl.type_);
      }
      /** insert attrDecls -> complexTypeModel_s0; */
      at_moc.push_back(attr, "xs:attribute");
    }
    return has_xml_import;
  }
}


dtd_processor::dtd_processor(registry &reg, std::ostream &ostm, bool verbose, bool do_dump)
  : registry_(reg), ostm_(&ostm), verbose_(verbose), do_dump_(do_dump), schema_(0) { }

int dtd_processor::process(const std::string &filename) { 
  if (schema_ != 0)
    delete schema_;

  bool has_xml_import = false;
  std::string base = get_dirpath(filename);


  aka::expat_parser parser;
  std::ifstream file(filename.c_str());
  handler h(registry_);

  parser.set_dtd_handler(h);
  parser.set_base(base);

  try {
    parser.parse_istream(file, filename);
  }
  catch (const std::exception &e) {
    std::cerr << e.what() << std::endl
	      << "failed to process document." << std::endl;
  }

  /** Insert declaration elements to created root object. */

  /** Construct xs::schema document. */
  schema_ = new xs::schema;
  aka::construct_element(*schema_, xs::schema_leaf());

  /** Insert top level element */
  xs::schema_sequence_s0 top;
  aka::construct_element(top, xs::schema_sequence_s0_leaf());
  xs::schemaTop_leaf::moc(top.schemaTop_).push_back(h.topLevelElement_, "xs:element");
  schema_->array0_.push_back(top);

  /** Insert complexTypes */
  for (complexTypes::iterator it = h.complexTypes_.begin(); 
       it != h.complexTypes_.end(); ++it) {


    // get complexTypeModel_s0 to store xs::complexType.
    xs::topLevelComplexType &tlc = it->second;
    assert(tlc.complexTypeModel_.size() == 1);

    aka::item item = tlc.complexTypeModel_.front();
    xs::complexTypeModel_s0 &cdef = aka::item_cast<xs::complexTypeModel_s0>(item);

    // Insert attribute declarations to attrDecls.
    // construct attrDecls.
    xs::attrDecls_c0_leaf::moc at_moc(cdef.attrDecls_.c0_);

    // get attribute associated with element named it->first.

    attdecl_map::iterator attrbegin = h.attdecls_.lower_bound(it->first);
    attdecl_map::iterator attrend = h.attdecls_.upper_bound(it->first);
    has_xml_import |= create_attributes(attrbegin, attrend, at_moc);

    bool cdef_empty = cdef.typeDefParticle_.empty()
      && cdef.attrDecls_.c0_.empty()
      && (cdef.attrDecls_.anyAttribute_.get() == 0);
    if (cdef_empty)
      tlc.complexTypeModel_.clear();

    xs::redefinable redefinable;
    xs::redefinable_leaf::moc(redefinable).push_back(it->second, "xs:complexType");
    xs::schema_sequence_s0 top;
    xs::schemaTop_leaf::moc(top.schemaTop_).push_back(redefinable, "&redefinable");
    schema_->array0_.push_back(top);
  }


  // value element (simpleContent or simpleType).
  for (value_elements::iterator vit = h.value_elements_.begin();
       vit != h.value_elements_.end(); ++vit) {
    attdecl_map::iterator attrbegin = h.attdecls_.lower_bound(*vit);
    attdecl_map::iterator attrend = h.attdecls_.upper_bound(*vit);

    if (attrbegin == attrend) {
      // value element is xs:simpleType.

      // xs:topLevelSimpleType:member -> &simpleDerivation:choice ->
      // xs:restriction::simpleRestrictionModel_ -> &simpleRestrictionModel::facets_ ->
      // &facets:choice -> xs:pattern(noFixedFacet).
      xs::restriction restriction;
      aka::construct_element(restriction.simpleRestrictionModel_, 
			     xs::simpleRestrictionModel_leaf());
      restriction.base_ = aka::qname("xs:string");

      xs::topLevelSimpleType st;
      aka::construct_element(st, xs::topLevelSimpleType_leaf());
      st.name_ = *vit;

      xs::simpleDerivation_leaf::moc(st.simpleDerivation_).
	push_back(restriction, "xs:restriction");
      xs::redefinable redefinable;
      aka::construct_element(redefinable, xs::redefinable_leaf());
      xs::redefinable_leaf::moc(redefinable).push_back(st, "xs:simpleType");
      xs::schema_sequence_s0 top;
      xs::schemaTop_leaf::moc(top.schemaTop_).push_back(redefinable, "&redefinable");
      schema_->array0_.push_back(top);
    }
    else {
      // value element is xs:simpleContent.

      xs::simpleExtensionType extension;
      aka::construct_element(extension, xs::simpleExtensionType_leaf());
      extension.base_ = aka::qname("xs:string");
      xs::attrDecls_c0_leaf::moc at_moc(extension.attrDecls_.c0_);
      // adding attribute declarations.
      has_xml_import = create_attributes(attrbegin, attrend, at_moc);

      // Add xs:schema/xs:complexType/xs:simpleContent/xs:extension
      xs::simpleContent sc;
      aka::construct_element(sc, xs::simpleContent_leaf());
      xs::simpleContent_choice_leaf::moc(sc.c0_).push_back(extension, "xs:extension");

      xs::topLevelComplexType ct;
      aka::construct_element(ct, xs::topLevelComplexType_leaf());
      ct.name_ = *vit;
      xs::complexTypeModel_leaf::moc(ct.complexTypeModel_).push_back(sc, "xs:simpleContent");

      xs::redefinable redefinable;
      xs::redefinable_leaf::moc(redefinable).push_back(ct, "xs:complexType");
      xs::schema_sequence_s0 top;
      xs::schemaTop_leaf::moc(top.schemaTop_).push_back(redefinable, "&redefinable");
      schema_->array0_.push_back(top);
    }
  }


  for (regex_types::iterator regit = h.regex_types_.begin();
       regit != h.regex_types_.end(); ++regit) {
    xs::redefinable redefinable;
    xs::redefinable_leaf::moc rd_moc(redefinable);
    rd_moc.push_back(regit->second, "xs:simpleType");

    xs::schema_sequence_s0 top;
    aka::construct_element(top, xs::schema_sequence_s0_leaf());
    xs::schemaTop_leaf::moc(top.schemaTop_).push_back(redefinable, "&redefinable");
    schema_->array0_.push_back(top);
  }

  if (has_xml_import) {
    xs::import import;
    aka::construct_element(import, xs::import_leaf());
    import.namespace_ = "http://www.w3.org/XML/1998/namespace";
    import.schemaLocation_ = "xml.xsd";
    xs::schema_sequence_c0_leaf::moc(schema_->c0_).push_back(import, "xs:import");
  }

  if (do_dump_) {
    aka::xml_serializer ser;
    try {
      ser.serialize(*schema_, "xs:schema", std::cout);
    }
    catch (const std::exception &e) {
      *ostm_ << "Serialization failed." << std::endl
	     << e.what() << std::endl;
    }
  }
  return 0;
}

#endif
