#include "sequence_handler.h"

#include "all_handler.h"
#include "choice_handler.h"
#include "simpletype_handler.h"
#include "simplecontent_handler.h"
#include "array_handler.h"
#include "ptrmember_handler.h"
#include "closure_handler.h"
#include "any_handler.h"
#include "fixed_handler.h"
#include "../../util/string_funcs.h"

#include <cassert>

using namespace aka2;

sequence_handler::sequence_handler(bool emptiable, 
				   void *seq, const sequence_op &sop,
				   int depth,
				   const element_props &props,
				   parser_context &context)  
  : handler(emptiable, depth, props, context) , seq_(seq), 
    sop_(sop) {
  current_ = sop_.get_member_types().begin();
  itend_ = sop_.get_member_types().end();
  
}

sequence_handler::sequence_handler(bool emptiable, 
				   int depth, sequence_handler *seqhandler, 
				   const element_props &props) 
  : handler(emptiable, *seqhandler, depth, props), 
    seq_(seqhandler->seq_), sop_(seqhandler->sop_) {
  current_ = seqhandler->current_;
  itend_ = seqhandler->itend_;
}


validation_result sequence_handler::query_element(const qname &tagname, const attribute_values &attrs){

  if (current_ == itend_)    // No more element.
    return skip;

  while (current_ != itend_) {

    if (current_->is_element()) {
      bool err = false;

      if (context_.compare_tagname(current_->get_name(), tagname))
	return parse_element(tagname, attrs, current_);

//       schematype_id id = current_->get_schematype();
//       if ((id != array_id) && (id != ptrmember_id))
// 	err = true;
      if (!can_skip_member()) {
	if (emptiable_)
	  return skip;
	else
	  err = true;
      }
      if (err) {
	context_.report_no_element(current_->get_name(), tagname, 
				   __FILE__, __LINE__);
	return invalid;
      }
    }
    else {
      // particle.
      assert(!current_->is_element());
 
      validation_result res = find_particle(tagname, attrs, current_, emptiable_);
      
      if (res == ok)// (Nested) Particle found.
	return ok;
      else if (res == invalid)  { // Expected element not found. 
	return invalid;
      }
      assert(res == skip);
    }
    ++current_; // res == skip, then go next.
  }

  // Element not found in sequence.
  // Allowing to go to the next particle if sequence is in a model group,
  if (!props_.is_element())
    return skip;

  // otherwise raise error.
  context_.report("sequence could not handle " 
		  + quote(context_.qualified_name(tagname)) + ".",
		  __FILE__, __LINE__);
  return invalid;
}


validation_result sequence_handler::query_next(const qname &tagname, const attribute_values &attrs) {

  member_types::const_iterator it = current_;
  if (it == itend_)    // No more element.
    return skip;

  ++it;

  while (it != itend_) {

    if (it->is_element())
      return context_.compare_tagname(it->get_name(), tagname) ? ok : invalid;

    // particle.
    assert(!it->is_element());
    bool emptiable = emptiable_;
    validation_result res = find_particle(tagname, attrs, it, emptiable);
    if (res == ok)// (Nested) Particle found.
      return ok;
    else if (res == invalid) { // Expected element not found. 
      return invalid;
    }
    assert(res == skip);

    ++it;
  }
  
  // Element not found in particles.
  // Allowing to go to the next particle.
  return skip;
}


validation_result sequence_handler::parse_element(const qname &tagname, 
						  const attribute_values &attrs,
						  member_types::const_iterator &it) {
  
  emptiable_ = false; // Element found.

  const member_type &mtype = *it;  

  schematype_id id = mtype.get_schematype();  
  cache_.set(seq_, mtype.op(), mtype.getter());
  
  // if @default is specified, copy-construct member.
  cache_.prepare(mtype.is_default_specified());
  void *memelm = cache_.value();
  const element_op &memberop = it->op();

  // Branch according to the member type.
  switch (id) {
  case sequence_id:
  case choice_id:
  case all_id: {
    create_particle_handler(false, memelm, memberop, 
			    depth_ + 1, 
			    tagname, *it);
    break;
  }
  case simpletype_id:
  case simplecontent_id: {
    create_simple_handler(memelm, memberop, 
			  depth_ + 1, 
			  tagname, *it);
    break;
  }
  case fixed_id: { // @fixed is always simpletype. (spec. restriction of akaxiso.)
    // entity of fixed array is null for <xs:sequence>.
    create_fixed_handler(0, static_cast<const fixed_op&>(memberop), 
			 depth_ + 1, 
			 tagname, *it);
    break;
  }
  case array_id: {
    handler *h = create_array_handler(false, memelm, 
				      depth_, 
				      tagname, *it);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  }
  case ptrmember_id: {
    handler *h = create_ptrmember_handler(false, memelm, depth_, 
					  tagname, *it);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  }
  case enclose_id: {
    handler *h = new closure_handler(depth_ + 1, this, tagname, *it);
    context_.push(h);
    ++it;
    break;
  }
  case any_id: { // anyType.
    handler *h = create_any_handler(*static_cast<any*>(memelm), depth_ + 1,
				    tagname, mtype);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  }
  case any_array_id: {
    handler *h = 
      create_any_array_handler(false, 
			       *static_cast<any_array*>(memelm), depth_,
			       mtype);
    validation_result res = h->query_element(tagname, attrs);
    assert(res == ok);
    break;
  }
  case disclose_id:  // disclose does not appear during any startElement() call.
  case wildcard_attribute_id:
  case wildcard_id:
    assert(!"Internal error.  Must not reach here.");
  }
  return ok;
}



validation_result sequence_handler::find_particle(const qname &tagname, 
						  const attribute_values &attrs,  
						  member_types::const_iterator &it,
						  bool &emptiable) {

  handler *current = 0;
  const member_type &mtype = *it;

  cache_.set(seq_, mtype.op(), mtype.getter()); 
  cache_.prepare(mtype.is_default_specified());

  void *memelm = cache_.value();
  const element_op &memberop = mtype.op();

  schematype_id id = memberop.get_schematype();

  if ( ((id == any_array_id) || (id == any_id)) 
       && (context_.ignore_any())) // Do not find <xs:any>.
    return invalid;

  validation_result res = invalid;

  // Branch according to the member type.
  switch (id) {
  case sequence_id:
  case choice_id: 
  case all_id: {
    current = create_particle_handler(emptiable, memelm, memberop, 
				      depth_, 
				      tagname, mtype);
    res = current->query_element(tagname, attrs);
    break;
  }
  case array_id: {
    current = create_array_handler(emptiable, memelm, 
				   depth_, 
				   tagname, mtype);
    res = current->query_element(tagname, attrs);
    break;
  }
  case ptrmember_id: {
    current = create_ptrmember_handler(emptiable, memelm, depth_, 
				       tagname, mtype);
    res = current->query_element(tagname, attrs);
    break;
  }
  case any_id: { // <xs:any>
    current = create_any_handler(*static_cast<any*>(memelm), 
				 depth_ + 1, 
				 tagname, mtype);
    res = current->query_element(tagname, attrs);
    break;
  }
  case any_array_id: { // <xs:any minOccurs="x" maxOccurs="XX">.....
    current = 
      create_any_array_handler(emptiable,
			       *static_cast<any_array*>(memelm),
			       depth_, 
			       mtype);
    res = current->query_element(tagname, attrs);
    break;
  }
  case simpletype_id:    // simpletype is not a particle.
  case simplecontent_id: // simplecontent is not a particle.
  case enclose_id:       // enclose is not a particle.
  case disclose_id:      // disclose is not a particle.
  case fixed_id:         // fixed is always element.(XML-Schema spec. restriction of akaxiso.)
  case wildcard_attribute_id: // attribute is not handled here.
  case wildcard_id:
    assert(!"Internal error.  Must not reach here.");
  }

  if (res == ok) {
    emptiable = false;
    return ok;  // Element given by tagname found.
  }
  // Rollup.
  context_.rollup_to(current);

  // Remove current.
  assert(context_.top() == current);
  context_.pop();

  assert((res == skip) || (res == invalid));
  cache_.clear();

  if (res == invalid) {
    context_.report_no_element(tagname, __FILE__, __LINE__);
  }
  return res; // skip or invalid.
}


validation_result sequence_handler::end_element(const qname &tagname){
  // No element left.  OK to end parse. 
  if (current_ == itend_)
    return ok;

  for (; current_ != itend_; ++current_) {
    // Disclose to terminate sequence handler. 
    if (current_->get_schematype() == disclose_id)
      return ok;

    // If element could not be skipped, it's an validation error.
    if (!can_skip_member()) {
      report_end_sequence_failure(current_->get_name(), __FILE__, __LINE__);
      return invalid;
    }
  }
  // All elements are skipped.  OK to terminate sequence handler.
  return skip;
}


bool sequence_handler::parse_entity(const std::string &entity) {
  return true;
}


node sequence_handler::get_node() {
  return node(seq_, sop_);
}


void sequence_handler::receive_child(const node &child) {
  assert(&child.op() == &current_->op());
  cache_.flush();
  ++current_;
}


bool sequence_handler::can_skip_member() {
  return current_->is_emptiable();
}


bool sequence_handler::can_skip() {
  if (emptiable_)
    return true;

  while (current_ != itend_) {
    bool ok_to_skip = can_skip_member();
    if (!ok_to_skip)
      return false;
    ++current_;
  }
  return true;
}

void sequence_handler::abort() {
  cache_.clear();
}

void sequence_handler::report_end_sequence_failure(const aka2::qname &tagname, 
						   const char *filename, 
						   const unsigned long linenum) {
  std::string message = tag(context_.qualified_name(tagname)) 
    + " expected before ending sequence.";
  context_.report(message, __FILE__, __LINE__);
}
