/* -*- c++ -*- */
#include "serializer.h"
#include "formatter_base.h"
#include "../../framework/member_cache.h"
#include "../../framework/any.h"
#include "../../framework/any_op.h"
#include "../../framework/namespace_statics.h"
#include "../../framework/operators.h"
#include "../../libselection.h"

using namespace aka2;

void aka2::serialize(const document &doc, std::ostream &ostm) {
  aka2::xml_serializer ser;
  ser.serialize(doc, ostm);
}

serializer::serializer(int tab)
  : indent_str_(tab, ' '), formatter_(0) {
  ecomp_holder_ = system_entity_complements().clone(true);
  ecomp_ = ecomp_holder_.get();
}

serializer::~serializer() {}


void serializer::set_entity_complements(entity_complements &ecomp) {
  ecomp_ = &ecomp;
}


void serializer::serialize(const document &doc, std::ostream &ostm) {
  do_serialize(doc.get_root().ptr(), doc.get_props(), doc.get_name(), ostm);
}

void serializer::using_prefix(const std::string &prefix) {
  id_type id = get_prefix_id(prefix);
  nsids_.push_back(id);
}

void serializer::using_uri(const std::string &uri) {
  id_type id = get_namespace_id(uri);
  nsids_.push_back(id);
}

void serializer::default_ns_prefix(const std::string &prefix) {
  id_type id = ecomp_->get_prefixes().get_prefix_id(prefix);
  const std::string &uri = g_namespaces_.get_namespace_uri(id);
  default_ns_uri(uri);
}

void serializer::default_ns_uri(const std::string &uri) {
  ecomp_->get_prefixes().assign_prefix("", uri);
}

void serializer::do_serialize(const void *e, const element_props &props, const qname &name, std::ostream &ostm) {
  if (ecomp_ == 0) {
    ecomp_holder_ = shared_ptr<entity_complements>(system_entity_complements().clone(true));
    ecomp_ = ecomp_holder_.get();
  }

  formatter_->prepare(ostm);
  indent_ = 0;
  is_root_ = true;
  serialize_internal(e, props, name);
  formatter_->finish();
}

void serializer::new_line() {
  std::string line = "\n";
  for (int i = indent_; i != 0; --i)
    line += indent_str_;
  formatter_->write(line);
}

bool serializer::is_sequence_empty(const void *e, const member_types &mtypes) {
  for (member_types::const_iterator it = mtypes.begin();
       it != mtypes.end(); ++it) {
    if (!is_member_empty(e, *it))
      return false;
  }
  return true;
}

bool serializer::is_all_empty(const void *e, const member_map &mmap) {
  for (member_map::const_iterator it = mmap.begin();
       it != mmap.end(); ++it) {
    if (!is_member_empty(e, it->second))
      return false;
  }
  return true;
}

bool serializer::is_member_empty(const void *e, const member_type &mtype) {

  const_member_cache cache(e, mtype.op(), mtype.getter());
  cache.prepare();
  
  switch (mtype.get_schematype()) {
  case simpletype_id:
  case simplecontent_id:
    return false;
  case array_id: {
    const array_op &aop = static_cast<const array_op&>(mtype.op());
    if (aop.size(cache.value()) != 0)
      return false;
    break;
  }
  case sequence_id: {
    if (mtype.is_element())
      return false;
    const sequence_op &sop = static_cast<const sequence_op&>(mtype.op());
    if (!is_sequence_empty(cache.value(), sop.get_member_types()))
      return false;
    break;
  }
  case choice_id: {
    if (mtype.is_element())
      return false;
    const choice_op &cop = static_cast<const choice_op&>(mtype.op());
    if (cop.size(cache.value()) != 0)
      return false;
    break;
  }
  case all_id: {
    if (mtype.is_element())
      return false;
    const all_op &aop = static_cast<const all_op&>(mtype.op());
    if (!is_all_empty(cache.value(), aop.get_member_map()))
      return false;
    break;
  }
  case ptrmember_id: {
    const ptrmember_op &pop = static_cast<const ptrmember_op&>(mtype.op());
    if (!pop.is_null(cache.value()))
      return false;
    break;
  }
  case any_array_id: {
    const any_array &ar = *static_cast<const any_array*>(cache.value());
    return ar.empty();
  }
  case enclose_id:
  case disclose_id: {
    return false; // enclose, disclose is always written.
  }
  case any_id: {
    return static_cast<const any*>(cache.value())->empty();
  }
  case wildcard_id: {
    const wildcard &wc = *static_cast<const wildcard*>(cache.value());
    return wc.empty();
  }
  case fixed_id: {
    return false; // !!!!!!!!!!!!!! is_fixed_empty();
  }
  case wildcard_attribute_id:
    assert(!"Must not reach here.");
  }
  return true;
}

void serializer::inc_ns_depth() {
  if (ns_stack_.empty()) {
    ns_set new_set;
    new_set.insert(empty_token); // empty prefix and "xml" prefix is XML-instrinsic prefixes.
    new_set.insert(xml_ns_token);
    ns_stack_.push(new_set); // push empty one.
  }
  else {
    const ns_set &nsset = ns_stack_.top();
    ns_stack_.push(nsset);
  }
}

void serializer::dec_ns_depth() {
  ns_stack_.pop();
}

bool serializer::is_new_ns(id_type id) const {
  for (ids::const_iterator it = nsids_.begin(); it != nsids_.end();
       ++it) {
    if (*it == id)
      return false;
  }

  const ns_set &nsset = ns_stack_.top();
  return nsset.find(id) == nsset.end();
}

void serializer::use_temp_nsdecl(id_type id) {
  assert(is_new_ns(id));
  ns_set &nsset = ns_stack_.top();
  nsset.insert(id);
}
