/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_QNAME_H__
#define AKAXISO2_FRAMEWORK_QNAME_H__

/**
 * @file akaxiso2/framework/qname.h
 * @brief Qualified name class. 
 */

#include <akaxiso2/framework/types.h>
#include <string>

namespace aka2 {

  class prefix_map;

  /**
   * @brief qualified name
   */
  class qname {
  public:
    friend struct qname_less;
    /** @brief default constructor */
    explicit qname() : namespace_id_(empty_token) {}
    /** 
     * @brief constructor by using name as std::string. 
     * @param rawname name
     * @exception aka2::error thrown if rawname is in a wrong format, 
     * or namespace prefix not registered.
     */
    explicit qname(const std::string &rawname) { set(rawname); }

    explicit qname(id_type namespace_id, const std::string &localname) {
      set(namespace_id, localname);
    }

    /**
     * @brief constructor by using uri and localname.
     * @param uri namespace URI
     * @param name local name
     */
    void set(const std::string &uri, const std::string &name);

    void set(const id_type namespace_id, const std::string &name);

    /**
     * @brief set name
     * @param rawname name
     * @exception aka2::error thrown if rawname is in wrong format, 
     * or namespace prefix not registered.
     */
    void set(const std::string &rawname);

    void set(const std::string &rawname,
	     const prefix_map &pfs);
    
    /**
     * @brief returns true if qname is qualified.
     * @return true if qualified, otherwise false. 
     */
    bool is_qualified() const { return namespace_id_ != empty_token; }

    /**
     * @brief return qualified name.
     * @return returns qualified name if qualified, or localname if not qualified. 
     */
    std::string qualified() const;

    std::string qualified(const prefix_map &pfmap) const;
    const std::string &prefix(const prefix_map &pfs) const;

    /**
     * @brief get prefix of this name.
     * @return prefix string.
     */
    const std::string &prefix() const;
    
    const id_type get_namespace_id() const { return namespace_id_; }

    /**
     * @brief get local name.
     * @return local name
     */
    std::string &local();

    /**
     * @brief get local name (const version)
     * @return local name
     */
    const std::string &local() const;

    bool operator ==(const qname &rhs) const {
      return (namespace_id_ == rhs.namespace_id_) && (name_ == rhs.name_);
    }
    bool operator !=(const qname &rhs) const {
      return (namespace_id_ != rhs.namespace_id_) || (name_ != rhs.name_);
    }

    /** 
     * @brief empty.
     * @return true if name string is empty, otherwise false.
     */
    bool empty() const;

  private:
    void validate_name() const;

    id_type namespace_id_;
    std::string name_;
    mutable const char *prefix_cache_;
  };

  struct qname_less {
    bool operator ()(const qname &lhs, const qname &rhs) const {
      if (lhs.namespace_id_ < rhs.namespace_id_)
        return true;
      else if (lhs.namespace_id_ == rhs.namespace_id_)
        return (lhs.name_ < rhs.name_);
      return false;
    }
  };

  inline std::ostream & operator <<(std::ostream &ostm, const aka2::qname &name) {
    ostm << name.qualified();
    return ostm;
  }

  inline bool operator <(const aka2::qname &lhs, const aka2::qname &rhs) {
    return aka2::qname_less()(lhs, rhs);
  }

} // namespace aka2

#endif
