/* -*- c++ -*- */
#include "array_handler.h"

#include "sequence_handler.h"
#include "choice_handler.h"
#include "all_handler.h"
#include "simplecontent_handler.h"
#include "simpletype_handler.h"

#include "../../framework/operators.h"
#include "../../framework/membertype.h"

#include <cassert>

using namespace aka2;

array_handler::array_handler(bool emptiable, void *e, 
			     int depth,
			     const qname &tagname, const element_props &props,
			     parser_context &context) 
  : handler(emptiable, depth, props, context), array_(e), 
    e_(0), aop_(static_cast<const array_op&>(props.op())), count_(0), tagname_(tagname) {
  is_element_ = props.is_element();
  
}

array_handler::~array_handler() {
  if (e_ != 0)
    abort();
}


validation_result array_handler::query_element(const qname &tagname, const attribute_values &attrs) {

  if (is_element_)
    return parse_element(tagname, attrs, e_);
  else
    return find_particle(tagname, attrs, e_);
}


validation_result array_handler::query_next(const qname &tagname, const attribute_values &attrs) {

  void *e = 0;

  if (is_element_) {
    if (context_.compare_tagname(tagname,  tagname_))
      return ok;
    return props_.get_occurrence().in_range(count_) ? skip : invalid;
  }
  else {
    validation_result res = find_particle(tagname, attrs, e);
    if (e != 0)
      aop_.get_item_op().destroy(e);
    return res;
  }
}

validation_result array_handler::parse_element(const qname &tagname, const attribute_values &attrs,
					       void *&e) {

  if (tagname != tagname_) {
    if (can_skip())
      return skip;
    context_.report_wrong_occurrence(tagname, props_.get_occurrence(), count_, 
				     __FILE__, __LINE__);
    return invalid;
  }

  emptiable_ = false; //Element found, therefore not empty.

  const element_op &op = aop_.get_item_op();
  e = op.create();

  switch (op.get_schematype()) {
  case sequence_id:
  case choice_id:
  case all_id: { 
    create_particle_handler(false, e, op, 
			    depth_ + 1, 
			    tagname, props_); // item element.
    return ok;
  }
  case simplecontent_id:
  case simpletype_id: {
    create_simple_handler(e, op, 
			  depth_ + 1, 
			  tagname, props_); // item element.
    return ok;
  }
  case fixed_id: {
    create_fixed_handler(e, static_cast<const fixed_op&>(op), 
			 depth_ + 1, 
			 tagname, props_);
    return ok;
  }

  case array_id: // nested array is not accepted.
  case ptrmember_id: // ptrmember should not be under array.
  case enclose_id: // The use of enclose/disclose is restricted for <xs:sequence>.
  case disclose_id:
  case any_id: // toplevel_any, toplevel_any_array should not be under array.
  case any_array_id:
  case wildcard_attribute_id:  // attribute is not handled here.
  case wildcard_id:
    assert(!"Must not reach here.");
  }
  return invalid;
}

validation_result array_handler::find_particle(const qname &tagname, 
					       const attribute_values &attrs,
					       void *&e) {

  bool child_emptiable = emptiable_ || props_.is_emptiable();
  const element_op &op = aop_.get_item_op();
  e = op.create();


  handler *handler = 0;

  switch (op.get_schematype()) {
  case sequence_id:
  case choice_id: 
  case all_id: {
    handler = create_particle_handler(child_emptiable, e, op, depth_, tagname, props_);
    validation_result res = handler->query_element(tagname, attrs);
    if (res != ok) {
      context_.rollup_to(handler);
      context_.top()->abort();
      context_.pop();
    }
    return res;
  }
  case simplecontent_id:
  case simpletype_id:
  case array_id: // nested array does not have any meaning.
  case ptrmember_id: // ptrmember should not be under array.
  case enclose_id: // The use of enclose/disclose is restricted for <xs:sequence>.
  case disclose_id:
  case any_array_id: // any, any_array should not be under array.
  case any_id:
  case fixed_id: // fixed should not be under array.
  case wildcard_attribute_id: // attribute is not handled here.
  case wildcard_id: 
    // any child is processed under toplevel_any_handler or toplevel_any_array_handler.
    assert(!"Must not reach here.");
  }
  return invalid;
}

validation_result array_handler::end_element(const qname &tagname) {
  if (can_skip())
    return ok;
  context_.report_wrong_occurrence(tagname, props_.get_occurrence(), count_,
				   __FILE__, __LINE__);
  return invalid;
}


bool array_handler::parse_entity(const std::string &entity) {
  return true;
}

node array_handler::get_node() {
  return node(array_, aop_);
}

void array_handler::receive_child(const node &child) {
  assert(&aop_.get_item_op() == &child.op());
  assert(e_ == child.ptr());
  aop_.push(array_, child.ptr());
  aop_.get_item_op().destroy(child.ptr());
  ++count_; // Increase occurrence count.
  e_ = 0;
}

bool array_handler::can_skip() {
  if (emptiable_)
    return true;
  return props_.get_occurrence().in_range(count_);
}

void array_handler::abort() {
  if (e_ != 0) {
    aop_.get_item_op().destroy(e_);
    e_ = 0;
  }
}
