#include "handler.h"
#include "../../framework/membertype.h"
#include "../../util/string_funcs.h"

#include "sequence_handler.h"
#include "all_handler.h"
#include "choice_handler.h"
#include "simpletype_handler.h"
#include "simplecontent_handler.h"
#include "array_handler.h"
#include "ptrmember_handler.h"
#include "closure_handler.h"
#include "wildcard_handler.h"
#include "any_handler.h"
#include "any_array_handler.h"
#include "fixed_handler.h"
#include "../../framework/member_cache.h"
#include "../../framework/namespaces.h"
#include "../../util/sstream.h"

#include <iostream>


using namespace aka2;

parser_context::parser_context(entity_complements *ecomp) : 
  locator_(new locator()), source_name_("(not specified)"), 
  silent_(false), ignore_any_(false) {
  locator_ = new locator();
  if (ecomp == 0) {
    ecomp_holder_ = system_entity_complements().clone(false);
    ecomp_ = ecomp_holder_.get();
  }
  else
    ecomp_ = ecomp;
}


parser_context::~parser_context() { 
  while (!handlers_.empty()) {
    handlers_.back()->abort();
    handlers_.pop_back();
  }
  delete locator_; 
}

void parser_context::reset() {
  ecomp_->reset_for_new_document();
}

void parser_context::push_tagname(const qname &tagname) {
  tagnames_.push_back(tagname);
}

void parser_context::pop_tagname() {
  tagnames_.pop_back();
}

void parser_context::report(const std::string &message, const std::string &path,
			    const char *filename, const unsigned long linenum) const {
  if (silent_)
    return;

  throw positioned_error(message, filename, linenum,
			 path,
			 source_name_, 
			 locator_->get_linenumber(),
			 locator_->get_columnnumber());
}

void parser_context::report(const std::string &message, 
			    const char *filename, const unsigned long linenum) const {
  if (silent_)
    return;

  std::string path = get_current_path();
  report(message, path, filename, linenum);
}


void parser_context::report_attribute_error(const std::string &message, 
					    const qname &attrname,
					    const char *filename, 
					    const unsigned long linenum) const {
  if (silent_)
    return;
  
  std::string path = get_current_path() + "@" + qualified_name(attrname);
  report(message, path, __FILE__, __LINE__);
}


void parser_context::report_wrong_occurrence(const qname &tagname,
					     const occurrence &occ, int actual,
					     const char *filename, 
					     const unsigned long linenum) const {
  if (silent_)
    return;

  std::ostringstream ostm;
  ostm << "Wrong occurrence: expected [" << occ.minOccurs_ << ", " << occ.maxOccurs_ << "], "
       << "actual " << actual << '.';
  report(ostm.rdbuf()->str(), filename, linenum);
}


void parser_context::report_no_element(const qname &tagname,
				       const char *filename, const unsigned long linenum) const {
  if (silent_)
    return;

  std::string str = "Element " + tag(qualified_name(tagname)) + " not found.";
  report(str, filename, linenum);
}

void parser_context::report_no_element(const qname &expected, const qname &actual,
				       const char *filename, const unsigned long linenum) const {
  if (silent_)
    return;

  std::string str = "Element " + tag(qualified_name(expected))
    + " is expected, but " + tag(qualified_name(actual)) + " found.";
  report(str, filename, linenum);
}

void parser_context::report_end_element_failure(const aka2::qname &tagname, 
						const char *filename, 
						const unsigned long linenum) const {
  std::string message = "Failed to end element, " + tag(qualified_name(tagname)) + ".";
  message += "  Other elements would be required, but not appeared.";
  report(message, __FILE__, __LINE__);
}

std::string parser_context::get_current_path() const {
  std::string path;
  for (tagname_stack::const_iterator it = tagnames_.begin();
       it != tagnames_.end(); ++it) {
    path += "/" + qualified_name(*it);
  }
  return path;
}

std::string parser_context::qualified_name(const qname &name) const {
  return name.qualified(ecomp_->get_prefixes());
}


void handler::parse_attributes(void *e, const element_op &op, attribute_values &attrs) {
  
  const attribute_types *atypes = op.get_attribute_types();
  const attribute_type *anytypes = op.get_anyattr_type();

  if ((atypes == 0) && (anytypes == 0)) { 
    // No attribute is defined, but found in document.
    if (attrs.size() != 0)
      context_.report("Attribute is not declared in xiso.", 
		      __FILE__, __LINE__);
    return;
  }
  
  if (atypes != 0) {

    for (attribute_types::const_iterator it = atypes->begin();
	 it != atypes->end(); ++it) {
    
      // Namespace defaulting is not applied.
      attribute_values::const_iterator itattr = attrs.find(it->get_name());

      if (itattr == attrs.end()) {
	if (it->is_required()) {
	  std::ostringstream ostm;
	  ostm << "Required attribute(" 
	       << tag(context_.qualified_name(it->get_name())) 
	       << ") not found.";
	  context_.report_attribute_error(ostm.rdbuf()->str(), it->get_name(), 
					  __FILE__, __LINE__);
	}
	continue;
      }
    
      const attribute_type &atype = *it;
      if (atype.get_schematype() == fixed_id) {
	if (!atype.is_fixed_string(itattr->second, context_.get_entity_complements())) {
	  std::ostringstream ostm;
	  ostm << "Fixed attribute(" 
	       << tag(context_.qualified_name(it->get_name())) 
	       << ") has wrong value." << std::endl
	       << "Expected : "; 
	  atype.write_fixed_string(ostm, context_.get_entity_complements());
	  ostm << ", Actual : " << itattr->second << std::endl;
	  context_.report_attribute_error(ostm.rdbuf()->str(), itattr->first, 
					  __FILE__, __LINE__);
	}
      }      
      else {
	const attribute_type &mtype = *it;
	assert(mtype.get_schematype() == simpletype_id);
	mtype.set_member_to_default(e);
	member_cache cache(e, mtype.op(), mtype.getter());

	cache.prepare(true);

	try {
	  static_cast<const simpletype_op&>(mtype.op()).
	    read_text(cache.value(), itattr->second, context_.get_entity_complements());
	}
	catch ( const std::exception &e ) {
	  context_.report_attribute_error(e.what(), itattr->first, 
					  __FILE__, __LINE__);
	}
	cache.flush();
      }
      attrs.erase(itattr->first);
    }

    if (attrs.empty())
      return;
  } 
  

  if (anytypes == 0) { 
    std::ostringstream ostm;
    ostm << "Attribute(s) ";
    for (attribute_values::iterator it = attrs.begin(); it != attrs.end(); ++it) {
      ostm << quote(context_.qualified_name(it->first))
	   << " ";
    }
    ostm << "appear(s) in the document, but not defined.";
    context_.report(ostm.rdbuf()->str(), __FILE__, __LINE__);
    return;
  }

  member_cache cache(e, anytypes->op(), anytypes->getter());
  cache.prepare(false);
  wc_attributes &wcattrs = *static_cast<wc_attributes*>(cache.value());
  assert(&wcattrs != 0);
  
  for (attribute_values::const_iterator anyit = attrs.begin();
       anyit != attrs.end(); ++anyit) {
    if (!anytypes->is_any_name(anyit->first)) {
      std::string message = "attribute name " 
	+ tag(context_.qualified_name(anyit->first))
	+ " does not match.";
      context_.report_attribute_error(message, anyit->first, 
				      __FILE__, __LINE__);
    }
    wc_attribute wcattr;
    wcattr.name_ = anyit->first;
    wcattr.value_ = anyit->second;
    wcattrs.push_back(wcattr);
  }
  cache.flush();
}


handler* handler::create_particle_handler(bool emptiable,
					  void *e, const element_op &op,
					  int depth, 
					  const qname &tagname, const element_props &props) {
  switch (op.get_schematype()) {
  case sequence_id:
  case choice_id:
  case all_id: 
    return create_handler(emptiable, e, op, depth, tagname, props, context_);
  case array_id: // They are not particle.
  case ptrmember_id:
  case simpletype_id:
  case simplecontent_id:
  case any_id:
  case any_array_id:
  case enclose_id:
  case disclose_id:
  case fixed_id:
  case wildcard_attribute_id:
  case wildcard_id:
    assert(!"Must not reach here.");
  }
  return 0;
}


handler* handler::create_simple_handler(void *e, const element_op &op,
					int depth, 
					const qname &tagname, const element_props &props) {
  // This is for xs:simpleType/xs:simpleContent.
  switch (op.get_schematype()) {
  case simpletype_id:
  case simplecontent_id:
  case fixed_id:
    return create_handler(false, e, op, depth, tagname, props, context_);
  case ptrmember_id:
  case any_id:
  case any_array_id:
  case sequence_id:
  case choice_id:
  case all_id: 
  case array_id:
  case enclose_id:
  case disclose_id:
  case wildcard_attribute_id:
  case wildcard_id:
    assert(!"Must not reach here.");
  }
  return 0;
}




array_handler* handler::create_array_handler(bool emptiable, void *e, int depth, 
					     const qname &tagname, const element_props &props) {
  array_handler *h = new array_handler(emptiable, e, depth, 
				       tagname, props, context_);
  context_.push(h);
  return h;
}

ptrmember_handler* handler::create_ptrmember_handler(bool emptiable,
						     void *e, int depth, 
						     const qname &tagname, 
						     const element_props &props) {
  assert(props.op().get_schematype() == ptrmember_id);
  ptrmember_handler *h = new ptrmember_handler(emptiable, e, depth, 
					       tagname, props, context_);
  context_.push(h);
  return h;
}


handler* handler::create_handler(bool emptiable,
				 void *e, const element_op &op,
				 int depth, 
				 const qname &tagname, const element_props &props, 
				 parser_context &context) {

  handler *h;
  
  switch (op.get_schematype()) {
  case sequence_id:
    h = new sequence_handler(emptiable, e, static_cast<const sequence_op&>(op),
			     depth,
			     props, 
			     context);
    break;
  case choice_id: 
    h = new choice_handler(emptiable, e, static_cast<const choice_op&>(op),
			   depth, 
			   props,
			   context);
    break;
  case all_id:
    h = new all_handler(emptiable, e, static_cast<const all_op&>(op),
			depth,
			props, 
			context);
    break;
  case simplecontent_id:
    h = new simplecontent_handler(e, static_cast<const simplecontent_op&>(op),
				  depth, 
				  props,
				  context);
    break;
  case simpletype_id:
    h = new simpletype_handler(e, static_cast<const simpletype_op&>(op),
			       depth, 
			       props,
			       context);
    break;
  case fixed_id:
    h = new fixed_handler(e, static_cast<const fixed_op&>(op),
			  depth, 
			  tagname, props, 
			  context);
    break;
  case any_id:
    h = new any_handler(*static_cast<any*>(e),
			depth,
			tagname, props,
			context);
    break;
  case array_id: 
    h = new array_handler(emptiable, e, 
			  depth, 
			  tagname, props, 
			  context);
    break;
  case wildcard_id:
    h = new wildcard_handler(*static_cast<wildcard*>(e),
			     depth,
			     tagname, props,
			     context);
    break;
  case any_array_id:
  case ptrmember_id:
  case enclose_id:
  case disclose_id:
  case wildcard_attribute_id:
    assert(!"Must not reach here.");
  }
  context.push(h);
  return h;
}

fixed_handler *handler::create_fixed_handler(void *e, const fixed_op &fop,
					     int depth,
					     const qname &tagname, const element_props &props) {
  fixed_handler *h = new fixed_handler(e, fop, depth, tagname, props, context_);
  context_.push(h);
  return h;
}

handler *handler::create_wildcard_handler(wildcard &an, int depth,
					  const qname &tagname, 
					  const element_props &props) {
  handler *h = new wildcard_handler(an, depth, tagname, props, context_);
  context_.push(h);
  return h;
}


handler *handler::create_any_handler(any &an, int depth,
				     const qname &tagname, 
				     const element_props &props) {
  if (!props.is_element()) {
    if (!props.is_any_name(tagname)) {
      std::string message = "tagname " 
	+ tag(context_.qualified_name(tagname))
	+ " does not match.";
      context_.report(message, __FILE__, __LINE__);
    }
  }

  handler *h = new any_handler(an, depth, tagname, props, context_);
  context_.push(h);
  return h;
}

handler *handler::create_any_array_handler(bool emptiable,
					   any_array &anys, 
					   int depth, 
					   const element_props &props) {
  handler *h = new any_array_handler(emptiable, anys, depth, props, context_);
  context_.push(h);
  return h;
}


bool handler::seek_forward(const qname &tagname, const attribute_values &attrs) {
  context_.set_ignore_any(true);

  handler_stack saved = context_.get_stack();
  
  context_.silent(true);
  context_.pop(); // pop myself.
  handler *h;

  bool found = false;

  if (context_.empty()) {
    found = false;
  }
  else {
    try {
      do {
	h = context_.top().get();
	validation_result res = h->query_next(tagname, attrs);
	
	if (res == ok) { // Matched element found.
	  found = true;
	  break;
	}
	else if (res == invalid) {// Cannot found matched one.
	  found = false;
	  break;
	}
	assert(res == skip);
	context_.pop();
	
      } while ((!context_.empty()) && (h->get_depth() >= depth_));
    }
    catch ( ... ) {
      assert(!"Exception must not be thrown.");
    }
  }
  context_.set_stack(saved);
  context_.silent(false);
  context_.set_ignore_any(false);
  return found;
}

bool parser_context::compare_tagname(const qname &lhs, const qname &rhs) const {
  if (lhs == rhs)
    return true;
  if (qualified_name(lhs) == qualified_name(rhs))
    return true;
  return false;
}

void parser_context::rollup_to(const handler *h) {
  while (handlers_.back() != h) {
    handlers_.back()->abort();
    handlers_.pop_back();
  }
}
