/* -*- c++ -*- */
#ifndef AKAXISO2_XML_DESERIALIZER_HANDLER_H__
#define AKAXISO2_XML_DESERIALIZER_HANDLER_H__

#include <akaxiso2/framework/types.h>
#include <akaxiso2/framework/qname.h>
#include <akaxiso2/framework/operators.h>
#include <akaxiso2/framework/member.h>
#include <akaxiso2/framework/membertype.h>
#include <akaxiso2/framework/any.h>
#include <akaxiso2/framework/entity_complements.h>
#include <akaxiso2/XML/deserializer/locator.h>

namespace aka2 {

  /* also declared in document_handler.h. */
  typedef std::map<qname, std::string, qname_less> attribute_values;

  enum validation_result {
    ok = 0,
    skip = 1,
    invalid = 2
  };

  struct parser_context;

  class array_handler;
  class ptrmember_handler;
  class fixed_handler;

  struct handler {
    virtual ~handler(){}
    virtual validation_result query_element(const qname &tagname, const attribute_values &attrs) = 0;
    virtual validation_result end_element(const qname &tagname) = 0;
    virtual validation_result query_next(const qname &tagname, const attribute_values &attrs) = 0;
    // Mainly used by simpletypes.
    virtual bool parse_entity(const std::string &entity) = 0;

    // Mainly used for choices.
    virtual node get_node() = 0;
    virtual void receive_child(const node &child) = 0;
    virtual bool can_skip() = 0;
    virtual void abort() = 0; 

    int get_depth() const { return depth_; }

    void parse_attributes(void *e, const element_op &op, attribute_values &attrs);

    handler* create_particle_handler(bool emptiable, void *e, const element_op &op,
				     int depth, 
				     const qname &tagname, const element_props &props);
    handler* create_simple_handler(void *e, const element_op &op,
				   int depth, 
				   const qname &tagname, const element_props &props);
    array_handler* create_array_handler(bool emptiable,
					void *e,
					int depth, 
					const qname &tagname, const element_props &props);
    ptrmember_handler* create_ptrmember_handler(bool emptiable, 
						void *e, int depth, 
						const qname &tagname, 
						const element_props &props);
    fixed_handler *create_fixed_handler(void *e, const fixed_op &op,
					int depth,
					const qname &tagname, const element_props &props);

    handler *create_wildcard_handler(wildcard &wc, int depth,
				     const qname &tagname, const element_props &props);
    handler *create_any_handler(any &an, int depth,
				const qname &tagname, const element_props &props);
    
    handler *create_any_array_handler(bool emptiable,
				      any_array &anys, int depth, 
				      const element_props &props);
    static handler* create_handler(bool emptiable, void *e, const element_op &op,
				   int depth, 
				   const qname &tagname, const element_props &props, 
				   parser_context &context);


 protected:
    handler(bool emptiable, int depth, const element_props &props, parser_context &context) 
      : emptiable_(emptiable), depth_(depth), props_(props), context_(context) {}
    handler(bool emptiable, const handler &h, int depth, const element_props &props) : 
      emptiable_(emptiable), depth_(depth), props_(props), context_(h.context_){}

    bool seek_forward(const qname &tagname, const attribute_values &attrs);

    bool emptiable_;
    int depth_;
    const element_props &props_;
    parser_context &context_;
  };


  typedef shared_ptr<handler> handler_ptr;
  typedef std::vector<handler_ptr> handler_stack;
  typedef std::vector<qname> tagname_stack;


  struct parser_context {
    parser_context(entity_complements *ecomp);
    ~parser_context();

    void reset();

    bool compare_tagname(const qname &lhs, const qname &rhs) const;

    void push(handler *h) { handlers_.push_back(handler_ptr(h)); }
    handler_ptr top() { return handlers_.back(); }
    void pop() { handlers_.pop_back(); }
    bool empty() const { return handlers_.empty(); }

    void push_tagname(const qname &tagname);
    void pop_tagname(); 

    const locator* get_locator() const { return locator_; }
    void set_locator(const locator *loc) {
      delete locator_;
      locator_ = loc;
    }

    void set_source_name(const std::string &source_name) {
      source_name_ = source_name;
    }
    const std::string& get_source_name() const {
      return source_name_;
    }

    entity_complements &get_entity_complements() { return *ecomp_; }
    handler_stack get_stack() { return handlers_; }
    void set_stack(handler_stack &to_set) { handlers_ = to_set; }

    void rollup_to(const handler *h);


    void report(const std::string &message, const std::string &path,
		const char *filename, const unsigned long linenum) const;
    void report(const std::string &message, 
		const char *filename, const unsigned long linenum) const;
    void report_attribute_error(const std::string &message, 
				const aka2::qname &attrname,
				const char *filename, 
				const unsigned long linenum) const;
    void report_wrong_occurrence(const qname &tagname,
				 const occurrence &occ, int actual,
				 const char *filename, const unsigned long linenum) const;
    void report_no_element(const qname &tagname, 
			   const char *filename, const unsigned long linenum) const;
    void report_no_element(const qname &expected, const qname &actual, 
			   const char *filename, const unsigned long linenum) const;
    void report_end_element_failure(const qname &tagname, 
				    const char *filename, const unsigned long linenum) const;
    
    std::string qualified_name(const aka2::qname &name) const;

    void silent(bool val) { silent_ =val; }

    bool ignore_any() const { return ignore_any_; }
    void set_ignore_any(bool val) { ignore_any_ = val; }

    std::string get_current_path() const;
  private:
    handler_stack handlers_;
    tagname_stack tagnames_;
    const locator *locator_;
    std::string source_name_;
    entity_complements* ecomp_;
    shared_ptr<entity_complements> ecomp_holder_;
    bool silent_;
    bool ignore_any_;
  };

} // namespace aka2

#endif
