/* -*- c++ -*- */
#ifndef AKAXISO2_XML_SERIALIZER_SERIALIZER_H__
#define AKAXISO2_XML_SERIALIZER_SERIALIZER_H__

/**
 * @file akaxiso2/XML/serializer/serializer.h
 * @brief serializer class (base of any serializer)
 */

#include <akaxiso2/framework/membertype.h>
#include <akaxiso2/framework/any.h>
#include <akaxiso2/framework/document.h>
#include <akaxiso2/framework/document_factory.h>
#include <akaxiso2/framework/entity_complements.h>
#include <akaxiso2/XML/serializer/formatter_base.h>
#include <akaxiso2/util/shared_ptr.h>
#include <set>
#include <stack>

namespace aka2 {

  /** 
   * @brief serialization function.
   *
   * Internally aka2::xml_serializer is used.
   * @param doc document to be serialized.
   * @param ostm std::ostream to output serialized XML document.
   */
  void serialize(const document &doc, std::ostream &ostm);
  
  /** @brief Base class of any serializer. */
  class serializer  {
  public:
    serializer(int tab);
    virtual ~serializer();

    void set_entity_complements(entity_complements &pcd);

    /** 
     * @brief Declare namespace prefix for namespace declared at the root node.
     * @param prefix namespace prefix declared by aka::xmlns().
     */
    void using_prefix(const std::string &prefix);

    /** 
     * @brief Declare namespace URI for namespace declared at the root node.
     * @param uri namespace URI declared by aka::xmlns().
     */
    void using_uri(const std::string &uri);

    /**
     * @brief Specify default namespace prefix.
     * @param prefix namespace prefix
     */
    void default_ns_prefix(const std::string &prefix);

    /**
     * @brief Specify default namespace URI.
     * @param uri namespace URI
     */
    void default_ns_uri(const std::string &uri);

    /**
     * @brief Serialize document instance.
     * @param doc document instance
     * @param ostm std::ostream
     */
    void serialize(const document &doc, std::ostream &ostm);
    
    /**
     * @brief Serialize given root instance to std::ostream.
     * @param root root instance.
     * @param name tagname of root element.
     * @param ostm std::ostream to output serialized XML document.
     */
    template <class R>
    void serialize(const R &root, const std::string &name, std::ostream &ostm) {
      aka2::qname qn(name);
      const element_props &props = system_document_factory().get_props(qn);
      do_serialize(&root, props, qn, ostm);
    }

  private:
    void do_serialize(const void *e, const element_props &props, const qname &name, std::ostream &ostm);
    virtual void serialize_internal(const void *e, const element_props &props,
				    const qname &name) = 0;

    int indent_;
    std::string indent_str_;

    int any_depth_;

  protected:
    const prefix_map& get_prefixes() const {
      return ecomp_->get_prefixes();
    }

    void inc_indent_level() { ++indent_; }
    void dec_indent_level() { --indent_; }
    void new_line();

    void inc_ns_depth();
    void dec_ns_depth();
    bool is_new_ns(id_type id) const;
    void use_temp_nsdecl(id_type id);

    static bool is_sequence_empty(const void *e, const member_types &mtypes);
    static bool is_all_empty(const void *e, const member_map &mmap);
    static bool is_member_empty(const void *e, const member_type &mtype);
    static bool is_any_empty(const void *e, const member_type &mtype);
    static bool is_fixed_empty(const void *e, const member_type &mtype); 

    /** Text formatting object for encoding-translation. */
    shared_ptr<formatter_base> formatter_;
    bool is_root_;

    typedef std::vector<id_type> ids;
    ids nsids_;
   
    typedef std::set<id_type> ns_set;
    typedef std::stack<ns_set> ns_stack;
    ns_stack ns_stack_;
    entity_complements *ecomp_;
    shared_ptr<entity_complements> ecomp_holder_;
  };

} // namespace aka2

#endif
