/* -*- c++ -*- */
#include "element_props.h"
#include "operators.h"
#include "fixed.h"
#include "../util/sstream.h"
#include "../util/string_funcs.h"
#include "namespace_statics.h"
#include "entity_complements.h"

#include <assert.h>
#include <algorithm>

#include "membertype.h"
#include "item.h"


using namespace aka2;



bool element_props::check_emptiable_recursive(const element_props &props, const element_op &op) {
  
  if (props.is_emptiable())
    return true;

  if (props.occ_.in_range(0))
    return true;
  
  switch (op.get_schematype()) {
  case sequence_id: {
    if (props.is_element())
      return false; 
    const sequence_op &sop = static_cast<const sequence_op&>(op);
    const member_types &mtypes = sop.get_member_types();
    for (member_types::const_iterator it = mtypes.begin();
	 it != mtypes.end(); ++it) {
      if (!check_emptiable_recursive(*it, it->op()))
	return false;
    }
    return true;
  }
  case all_id: {
    if (props.is_element())
      return false; 
    const all_op &aop = static_cast<const all_op&>(op);
    const member_map &mtypes = aop.get_member_map();
    for (member_map::const_iterator it = mtypes.begin();
	 it != mtypes.end(); ++it) {
      if (it->second.get_occurrence().minOccurs_ > 0)
	return false;
    }
    return true;
  }
  case choice_id: {
    if (props.is_element())
      return false;
    const choice_op &cop = static_cast<const choice_op&>(op);
    const item_types &itypes = cop.get_item_types();
    for (item_types::const_iterator it = itypes.begin();
	 it != itypes.end(); ++it) {
      if (check_emptiable_recursive(it->second, it->second.op()))
	return true;
    }
    return false;
  }
  case array_id: {
    if (props.get_occurrence().minOccurs_ == 0)
      return true;
    const array_op &pop = static_cast<const array_op&>(props.op());
    return check_emptiable_recursive(props, pop.get_item_op());
  }
  case ptrmember_id: {
    if (props.get_occurrence().minOccurs_ == 0)
      return true;
    const ptrmember_op &pop = static_cast<const ptrmember_op&>(props.op());
    return check_emptiable_recursive(props, pop.get_value_op());
  }
  case any_array_id:
    return props.get_occurrence().minOccurs_ == 0;
  case any_id:
    return true;
  case simpletype_id:
  case simplecontent_id:
  case enclose_id:
  case disclose_id:
  case wildcard_attribute_id:
  case wildcard_id:
  case fixed_id: // fixed should be always an element.  akaxiso restriction.!!!!!
    assert(props.is_element());
    return false;
  }
  
  assert(!"Must not reach here.");
  return false;
}

void element_props::check_emptiable() {
  emptiable_ = check_emptiable_recursive(*this, *op_);
}


const int aka2::unbounded = -1;

bool occurrence::in_range(int count) const {
  assert(count >=0);
  if (maxOccurs_ == aka2::unbounded)
    return count >= minOccurs_;

  assert(minOccurs_ <= maxOccurs_);
  return (minOccurs_ <= count) && (count <= maxOccurs_);
}

bool occurrence::is_array() const {
  return minOccurs_ != 1 || maxOccurs_ != 1;
}



bool element_props::is_fixed_string(const std::string &val, 
				    entity_complements &ecomp) const {
  const fixed_op *fop;
  if (get_schematype() == array_id) {
    const array_op &aop = static_cast<const array_op&>(*op_);
    assert(aop.get_item_op().get_schematype() == fixed_id);
    fop = &static_cast<const fixed_op&>(aop.get_item_op());
  }
  else if (get_schematype() == fixed_id) {
    fop = static_cast<const fixed_op*>(op_);
  }
  else {
    assert(!"Must not reach here.");
  }

  const simpletype_op &sop = fop->get_value_op();
  void *given = sop.create();

  bool res = true;
  try {
    sop.read_text(given, val, ecomp);
  }
  catch ( ... /*!!!!!*/) {
    res = false;
  }
  if (res)
    res = sop.equals(given, default_->value());
  sop.destroy(given);
  return res;
} 

void element_props::write_fixed_string(std::ostream &ostm, 
				       entity_complements &ecomp) const {
  const fixed_op *fop;
  if (get_schematype() == array_id) {
    const array_op &aop = static_cast<const array_op&>(*op_);
    assert(aop.get_item_op().get_schematype() == fixed_id);
    fop = &static_cast<const fixed_op&>(aop.get_item_op());
  }
  else if (get_schematype() == fixed_id) {
    fop = static_cast<const fixed_op*>(op_);
  }
  else {
    assert(!"Must not reach here.");
  }
  const simpletype_op &sop = fop->get_value_op();
  sop.write_text(default_->value(), ostm, ecomp);
}

void element_props::set_default(void *e) const {
  assert((get_schematype() == simpletype_id) || (get_schematype() == fixed_id));
  if (default_.get() == 0)
    return;
  if (default_->has_default())
    op_->copy(e, default_->value());
}

bool element_props::is_default(const void *e) const { 
  assert(is_default_specified());
  return op_->equals(e, default_->value());
}


void element_props::set_ns_list(const std::string &ns_list) {
  std::string trimmed = trim(ns_list);
  if (trimmed.empty())
    return;
  if (trimmed == "##any")
    ns_type_ = ns_any;
  else if (trimmed.find("##other:") == 0) {
    ns_type_ = ns_other;
    std::string target_ns = trimmed.substr(8);
    target_ns_id_ = get_namespace_id(target_ns);
  }
  else if (trimmed == "##local")
    ns_type_ = ns_local;
  else if (trimmed.substr(0, 2) == "##") {
    std::string message = "Wrong namespace specification, " + quote(trimmed) + '.';
    throw error(message, __FILE__, __LINE__);
  }
  else {
    isstream istm(ns_list);
    while (true) {
      std::string ns;
      istm >> ns;
      if (istm.fail())
	break;
      id_type id = g_namespaces_.register_namespace_uri(ns);
      ids_.push_back(id);
    }
    ns_type_ = ns_uri;
  }
}

bool element_props::is_any_name(const qname &name) const {
  switch (ns_type_) {
  case ns_any:
    return true;
  case ns_other:
    if (name.get_namespace_id() == empty_token)
      return false;
    if (target_ns_id_ == name.get_namespace_id())
      return false;
    return true;
  case ns_local:
    return ! name.is_qualified();
  case ns_uri:
    return std::find(ids_.begin(), ids_.end(), name.get_namespace_id()) != ids_.end();
  case ns_na:
    break;
  }
  assert(!"Must not reach here.");
  return false;
}

void element_props::set_default_op(default_op *defop) {
  assert(defop != 0);
  default_ = shared_ptr<default_op>(defop);
}
