/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_SIMPLECONTENT_H__
#define AKAXISO2_FRAMEWORK_SIMPLECONTENT_H__

/** 
 * @file akaxiso2/framework/simplecontent.h
 * @brief @ref aka_simplecontent -related classes/templates
 */

#include <akaxiso2/framework/operators.h>
#include <akaxiso2/framework/simpletype.h>
#include <akaxiso2/framework/attribute.h>
#include <akaxiso2/framework/construct.h>
#include <akaxiso2/framework/xmltype.h>
#include <akaxiso2/builtin/builtin_xiso.h>

namespace aka2 {

  template<class L>
  class simplecontent_op_dispatcher : public simplecontent_op {
  public:
    virtual schematype_id get_schematype() const { return simplecontent_id; }
    virtual std::string get_typename() const { return L::get_xmltype(); }
    /** creatable */
    virtual void construct(void *e) const {  L::construct(e); }
    virtual void copy_construct(void *e, const void *src) const { L::copy_construct(e, src); }
    virtual void destruct(void *e) const { L::destruct(e); }
    virtual size_t class_size() const { return L::class_size(); }
    virtual bool equals(const void *lhs, const void *rhs) const {
      return L::equals(lhs, rhs);
    }

    /** attribute_types getter. */
    virtual const attribute_types *get_attribute_types() const {
      return &L::attribute_types_;
    }
    virtual const attribute_type *get_anyattr_type() const {
      return L::get_anyattr_type();
    }
    virtual const member_type& get_valuetype() const { return L::value_type_; }

  };

  template<class L>
  struct simplecontent_statics {
    static member_type value_type_;
    static simplecontent_op_dispatcher<L> dispatcher_;
    static aka2::occurrence occ_;
  };

  template<class L>
  member_type simplecontent_statics<L>::value_type_;

  template<class L>
  aka2::occurrence simplecontent_statics<L>::occ_;

  template<class L>
  simplecontent_op_dispatcher<L> simplecontent_statics<L>::dispatcher_;


  /**
   * @brief template to define @ref aka_simplecontent leaf class.
   *
   * Leaf classes for @ref aka_simplecontent are derived from this template class.\n
   * @see @ref aka_simplecontent
   *
   * @param T value class type
   * @param L leaf class type
   */
  template<class T, class L=xiso::leaf<T> >
  class simplecontent : public attributes<L, T>, 
			public simplecontent_statics<L>, 
			public xmltype_statics<L> {
  public:
    typedef T value_type;

    virtual ~simplecontent(){}

    static void initialize() {
      if (!system_type_registry().add(L()))
	return;
      L::attribute_types_.clear();
      member_getter *getter = new null_getter();
      L::value_type_ = member_type(getter, 
				   nill_leaf::dispatcher_, 
				   false);
      L l; l.model();
    }
    static void uninitialize() {
      L::attribute_types_.clear();
    }


    static void construct(void *e) {
      new (e) T();
      simplecontent_construct(e, L::dispatcher_);
    }
    static void copy_construct(void *e, const void *src) {
      new (e) T(*static_cast<const T*>(src));
    }
    static size_t class_size() { return sizeof(T); }
    static void destruct(void *elm) { static_cast<T*>(elm)->~T(); }


    static bool equals(const void *lhs, const void *rhs) {
      return simplecontent_equals(lhs, rhs, L::dispatcher_);
    }

    static element_op* get_attribute_dispatcher() { return &L::dispatcher_; }
    static default_op* create_default_op() { return 0; }

    /**
     * @brief define simpleType value of simplecontent.
     * @param m pointer to member of simpleType value
     * @param vl leaf class for V
     */
    template<class V, class P, class VL>
    static void value(V P::* m, const VL& vl) {
      VL::initialize();
      member_getter *getter =
	create_ptr_getter(reinterpret_cast<T*>(0), m);
      L::value_type_ = member_type(getter, VL::dispatcher_, false);
      default_op *defop = VL::create_default_op();
      if (defop != 0)
	L::value_type_.set_default_op(defop);
    }

    /**
     * @brief define simpleType value of simplecontent.
     * @param m pointer to member of simpleType value
     */
    template<class V, class P>
    static void value(V P::* m) {
      value(m, xiso::leaf<V>());
    }

    /**
     * @brief define fixed value as simplecontent value.
     * @param V value type for fixed value
     */
    template<class V>
    struct fixed_value {
      /**
       * @brief Constructor to define fixed value
       * @param fixed fixed value
       */
      fixed_value(const std::string &fixed) {
	new_member(fixed, xiso::leaf<V>());
      }
      /**
       * @brief Constructor to define fixed value
       * @param fixed fixed value
       * @param vl value leaf class for V
       */
      template<class VL>
      fixed_value(const std::string &fixed, const VL &vl) {
      	new_member(fixed, VL());
      }
    private:
      template<class VL>
      void new_member(const std::string &fixed, const VL&) {
      	VL::initialize();

	if (VL::dispatcher_.get_schematype() != simpletype_id)
	  throw tagged_error("simpleContent", L::get_xmltype(), 
			     "fixed value should be a simpleType.",
			     __FILE__, __LINE__);

	L::value_type_ = aka2::member_type(new null_getter(), 
					   aka2::fixed<VL>::dispatcher_, 
					   false);
	L::value_type_.set_name(aka2::qname("&value"));
    	default_op *defop = VL::create_default_op();
	if (defop != 0)
	  L::value_type_.set_default_op(defop);
	L::value_type_.setup_default_value(fixed);
      }
    };

    /**
     * @brief set defalut for simplecontent value.  
     * @param defval default value
     */
    static void set_default(const std::string &defval) {
      if (L::value_type_.get_schematype() != simpletype_id)
	throw tagged_error("simpleContent", L::get_xmltype(), 
			   "default value should be a simpleType.",
			   __FILE__, __LINE__);
      L::value_type_.set_default(defval);
    }
  };

} // namespace aka2

#endif
