#include <akaxiso2/akaxiso2.h>

#include "osixaka2.h"
#include "osx_xiso.h"
#include "registry.h"


namespace {
  
  struct value_type {
    const char *cpp_;
    const char *leaf_;
    bool resolve_;
  };

  struct array_type {
    const char *cpp_;
    const char *typedef_;
    const char *leaf_;
  };

  struct predefined {
    const char *name_;
    value_type type_;
    array_type array_;
  };

  const predefined default_builtins[] = {
    {"std:string",       {0,                0, false}, {"aka:string_array",    "std::vector<std::string>",    "aka::string_array_leaf"   }},
    {"aka:nill",         {0,                0, false}, {"aka:nill_array",      "std::vector<aka::nill>",      "aka::nill_array_leaf"     }},
    {"int",              {0,                0, false}, {"aka:int_array",       "std::vector<int>",            "aka::int_array_leaf"      }},
    {"unsigned_int",     {0,                0, false}, {"aka:uint_array",      "std::vector<unsigned int>",   "aka::uint_array_leaf"     }},
    {"char",             {0,                0, false}, {"aka:char_array",      "std::vector<char>",           "aka::char_array_leaf"     }},
    {"unsigned char",    {0,                0, false}, {"aka:uchar_array",     "std::vector<unsigned char>",  "aka::uchar_array_leaf"    }},
    {"short",            {0,                0, false}, {"aka:short_array",     "std::vector<short>",          "aka::short_array_leaf"    }},
    {"unsigned short",   {0,                0, false}, {"aka:ushort_array",    "std::vector<unsigned short>", "aka::uchar_array_leaf"    }},
    {"long",             {0,                0, false}, {"aka:long_array",      "std::vector<short>",          "aka::long_array_leaf"     }},
    {"unsigned long",    {0,                0, false}, {"aka:ulong_array",     "std::vector<unsigned short>", "aka::ulong_array_leaf"    }},
    {"LONGLONG",         {0,                0, false}, {"aka:longlong_array",  "std::vector<LONGLONG>",       "aka::longlong_array_leaf" }},
    {"ULONGLONG",        {0,                0, false}, {"aka:ulonglong_array", "std::vector<ULONGLONG>",      "aka::ulonglong_array_leaf"}},
    {"float",            {0,                0, false}, {"aka:float_array",     "std::vector<float>",          "aka::float_array_leaf"    }},
    {"double",           {0,                0, false}, {"aka:double_array",    "std::vector<double>",         "aka::double_array_leaf"   }},
    {"bool",             {0,                0, false}, {"aka:bool_array",      "std::vector<bool>",           "aka::bool_array_leaf"     }},
    {"std:string",       {0,                0, false}, {"aka:string_array",    "std::vector<std::string>",    "aka::string_array_leaf"   }},
    {"aka:any",          {0,                0, false}, {"aka:any_array",       "std::vector<aka::any>",       0                          }},
    {"aka:wildcard",     {0,                0, false}, {0,                     0,                             0                          }},
    {0}
  };

  const predefined default_imports[] = {
    {"xs:byte",               {"char",           0,  true}, {"aka:char_array",              "std::vector<std::string>", 0}},
    {"xs:unsignedByte",       {"unsigned char",  0,  true}, {"aka:uchar_array",             "std::vector<std::string>", 0}},
    {"xs:short",              {"short",          0,  true}, {"aka:short_array",             "std::vector<std::string>", 0}},
    {"xs:unsignedShort",      {"unsigned short", 0,  true}, {"aka:ushort_array",            "std::vector<std::string>", 0}},
    {"xs:int",                {"long",           0,  true}, {"aka:long_array",              "std::vector<std::string>", 0}},
    {"xs:unsignedInt",        {"ULONGLONG",      0,  true}, {"aka:ulonglong_array",         "std::vector<std::string>", 0}},
    {"xs:long",               {"LONGLONG",       0,  true}, {"aka:longlong_array",          "std::vector<std::string>", 0}},
    {"xs:unsignedLong",       {"ULONGLONG",      0,  true}, {"aka:ulonglong_array",         "std::vector<std::string>", 0}},
    {"xs:integer",            {"long",           0,  true}, {"aka:long_array",              "std::vector<std::string>", 0}},
    {"xs:boolean",            {"bool",           0,  true}, {"aka:bool_array",              "std::vector<std::string>", 0}},
    {"xs:float",              {"float",          0,  true}, {"aka:float_array",             "std::vector<std::string>", 0}},
    {"xs:double",             {"double",         0,  true}, {"aka:double_array",            "std::vector<std::string>", 0}},
    {"xs:string",             {"std:string",     0,  true}, {"aka:string_array",            "std::vector<std::string>", 0}},
    {"xs:normalizedString",   {"std::string",    0, false}, {"xs:normalizedString_array",   "std::vector<std::string>", 0}},
    {"xs:token",              {"std::string",    0, false}, {"xs:token_array",              "std::vector<std::string>", 0}},
    {"xs:base64Binary",       {"std::string",    0, false}, {"xs:base64Binary_array",       "std::vector<std::string>", 0}},
    {"xs:hexBinary",          {"std::string",    0, false}, {"xs:hexBinary_array",          "std::vector<std::string>", 0}},
    {"xs:positiveInteger",    {"long",           0, false}, {"xs:positiveInteger_array",    "std::vector<long>",        0}},
    {"xs:negativeInteger",    {"long",           0, false}, {"xs:negativeInteger_array",    "std::vector<long>",        0}},
    {"xs:nonNegativeInteger", {"long",           0, false}, {"xs:nonNegativeInteger_array", "std::vector<long>",        0}},
    {"xs:nonPositiveInteger", {"long",           0, false}, {"xs:nonPositiveInteger_array", "std::vector<long>",        0}},
    {"xs:decimal",            {"long",           0, false}, {"xs:decimal_array",            "std::vector<long>",        0}},
    {"xs:time",               {"std::string",    0, false}, {"xs:time_array",               "std::vector<std::string>", 0}},
    {"xs:dateTime",           {"std::string",    0, false}, {"xs:dateTime_array",           "std::vector<std::string>", 0}},
    {"xs:duration",           {"std::string",    0, false}, {"xs:duration_array",           "std::vector<std::string>", 0}},
    {"xs:date",               {"std::string",    0, false}, {"xs:date_array",               "std::vector<std::string>", 0}},
    {"xs:gMonth",             {"std::string",    0, false}, {"xs:gMonth_array",             "std::vector<std::string>", 0}},
    {"xs:gYear",              {"std::string",    0, false}, {"xs:gYear_array",              "std::vector<std::string>", 0}},
    {"xs:gYearMonth",         {"std::string",    0, false}, {"xs:gYearMonth_array",         "std::vector<std::string>", 0}},
    {"xs:gDay",               {"std::string",    0, false}, {"xs:gDay_array",               "std::vector<std::string>", 0}},
    {"xs:gMonthDay",          {"std::string",    0, false}, {"xs:gMonthDay_array",          "std::vector<std::string>", 0}},
    {"xs:Name",               {"std::string",    0, false}, {"xs:Name_array",               "std::vector<std::string>", 0}},
    {"xs:QName",              {"aka:qname",    "xs::QName_leaf", false}, {"xs:QName_array", "std::vector<aka:qname>",   0}},
    {"xs:NCName",             {"std::string",    0, false}, {"xs:NCName_array",             "std::vector<std::string>", 0}},
    {"xs:anyURI",             {"std::string",    0, false}, {"xs:anyURI_array",             "std::vector<std::string>", 0}},
    {"xs:language",           {"std::string",    0, false}, {"xs:language_array",           "std::vector<std::string>", 0}},
    {"xs:ID",                 {"std::string",    "xs::ID_leaf" , false},    {"xs:ID_array", "std::vector<std::string>", 0}},
    {"xs:IDREF",              {"std::string",    0, false}, {"xs:IDREF_array",              "std::vector<std::string>", 0}},
    {"xs:IDREFS",             {"std::string",    0, false}, {"xs:IDREFS_array",             "std::vector<std::string>", 0}},
    {"xs:ENTITY",             {"std::string",    0, false}, {"xs:ENTITY_array",             "std::vector<std::string>", 0}},
    {"xs:ENTITIES",           {"std::string",    0, false}, {"xs:ENTITIES_array",           "std::vector<std::string>", 0}},
    {"xs:NOTATION",           {"std::string",    0, false}, {"xs:NOTATION_array",           "std::vector<std::string>", 0}},
    {"xs:NMTOKEN",            {"std::string",    0, false}, {"xs:NMTOKEN_array",            "std::vector<std::string>", 0}},
    {"xs:NMTOKENS",           {"std::string",    0, false}, {"xs:NMTOKENS_array",           "std::vector<std::string>", 0}},
    {"xs:anySimpleType",      {"std::string",    0, false}, {"aka:string_array",            "std::vector<std::string>", 0}},
    {"xs:anyType",            {"aka:any",        0, false}, {"aka:any_array",               "std::vector<aka::any>",    0}},
    {"aka:nill",              {"aka:nill",       0, false}, {"aka:nill_array",              0,     "aka::nill_array_leaf"}},
    {0, {0, 0, false}, {0, 0, 0}}
  };

  struct default_substitution {
    const char *original_;
    const char *substituted_;
  };

  const char *cpp_keywords[] = {
    "and", "bool", "compl", "do", "export", "goto", "namespace", "or_eq",
    "return", "struct", "try", "using", "xor", "and_eq", "break", "const",
    "double", "extern", "if", "new", "private", "short", "switch", "typedef", 
    "virtual", "xor_eq", "asm", "case", "const_cast", "dynamic_cast", "false",
    "inline", "not", "protected", "signed", "template", "typeid" ,"void",
    "auto", "catch", "continue", "else", "float", "int", "not_eq", "public", 
    "sizeof", "this", "typename", "volatile", "bitand", "char", "default",
    "enum", "for", "long", "operator", "register", "static", "throw", "union",
    "wchar_t", "bitor", "class", "delete", "explicit", "friend", "mutable",
    "or", "reinterpret_cast", "static_cast", "true", "unsigned", "while", 0
  };
  
  struct default_escape {
    const char *to_escape_;
    const char *escaped_; 
  };

  const default_escape default_escapes[] = {
    {"-", "_"},
    {".", "_"},
    {0}
  };

  bool builtin_exists(const osx::type_array &types, const std::string &name) {
    for (osx::type_array::const_iterator it = types.begin();
	 it != types.end(); ++it) {
      if (it->name_ == name)
	return true;
    }
    return false;
  }
  bool import_exists(const osx::imports_choice &types, const std::string &name) {
    for (osx::imports_choice::const_iterator it = types.begin();
	 it != types.end(); ++it) {
      if (aka::item_of(*it, name))
	return true;
    }
    return false;
  }

  bool substitution_exists(const osx::substitution_array &subs, const std::string &original) {
    for (osx::substitution_array::const_iterator it = subs.begin();
	 it != subs.end(); ++it) {
      if (it->original_ == original)
	return true;
    }
    return false;
  }

  bool escape_exists(const osx::escape_array &ess, const std::string &to_escape) {
    for (osx::escape_array::const_iterator it = ess.begin();
	 it != ess.end(); ++it) {
      if (it->to_escape_ == to_escape)
	return true;
    }
    return false;
  }
  
  osx::type construct_type(const predefined *predef) {
    
    osx::type type;
    
    type.name_ = predef->name_;
    if (predef->type_.leaf_ != 0)
      type.primitive_.leaf_.reset(new std::string(predef->type_.leaf_));
    if (predef->type_.cpp_ != 0)
      type.primitive_.typedef_.reset(new std::string(predef->type_.cpp_));
    
    if (predef->array_.cpp_ != 0) {
      osx::builtin_array array;
      array.name_ = predef->array_.cpp_;
      if (predef->array_.leaf_ != 0)
	array.primitive_.leaf_.reset(new std::string(predef->array_.leaf_));
      if (predef->array_.typedef_ != 0)
	array.primitive_.typedef_.reset(new std::string(predef->array_.typedef_));
      type.array_.reset(new osx::builtin_array(array));
    }
    
    return type;
  }
  
  bool namespace_exists(const osx::directives_choice &dirs, const std::string &prefix) {
    for (osx::directives_choice::const_iterator it = dirs.begin();
	 it != dirs.end(); ++it) {
      if (!aka::item_of(*it, prefix))
	continue;
      const osx::_namespace &ns = aka::item_cast<osx::_namespace>(*it);
      if (ns.prefix_ == prefix)
	return true;
    }
    return false;
  }
  
}

using namespace osx;

void osixaka2::create_default_preference() {

  pref_.reset(new preference());
  aka::construct_element(*pref_, osx::preference_leaf());

  // member
  pref_->format_.member_.prefix_ = "";
  pref_->format_.member_.suffix_ = "_";

  // array
  pref_->format_.array_.prefix_ = "";
  pref_->format_.array_.suffix_ = "_array";
  pref_->format_.array_.container_ = "std::list";

  // choice.
  pref_->format_.choice_.container_ = "std::list";
  // simplecontent
  pref_->format_.simplecontent_.valuename_ = "value";

  // particle array
  pref_->format_.particle_item_.prefix_ = "";
  pref_->format_.particle_item_.suffix_ = "_item";

  merge_defaults();

}

void osixaka2::load_defaults() {
  create_default_preference();
}

void osixaka2::load_osixaka1_defaults() {
  create_default_preference();
  pref_->format_.choice_.container_ = "std::vector";
  pref_->format_.array_.container_ = "std::vector";
  pref_->format_.array_.prefix_ = "";
  pref_->format_.array_.suffix_ = "s_type";
}

void osixaka2::use_MFC_member_style() {
  assert(pref_.get() != 0);
  pref_->format_.member_.prefix_ = "m_";
  pref_->format_.member_.suffix_ = "";
}

void osixaka2::generate_template(std::ostream &file) {
  osx::preference pref = *pref_;
  pref.escapes_.escape_.clear();
  pref.substitutions_.substitution_.clear();
  pref.directives_.c0_.clear();
  pref.internal_.reset(0);

  osx::escape es;
  es.to_escape_ = "character_to_escape";
  es.escaped_ = "escaped_character";
  pref.escapes_.escape_.push_back(es);

  osx::substitution sub;
  sub.original_ = "token_to_substitute";
  sub.substituted_ = "substituted_token";
  pref.substitutions_.substitution_.push_back(sub);

  aka::xml_serializer ser;
  ser.default_ns_prefix("osx");
  ser.serialize(pref, "osx:preference", file);
}

void osixaka2::load_preference(const std::string &filename) {
  aka::document doc = aka::deserialize_file(filename);
  pref_.reset(aka::adopt_root<osx::preference>(doc));
  merge_defaults();
}

void osixaka2::save_preference(std::ostream &file) {
  aka::xml_serializer ser;
  ser.default_ns_prefix("osx");
  ser.serialize(*pref_, "osx:preference", file);
}

void osixaka2::merge_defaults() {

  // add internal type.
  if (pref_->internal_.get() == 0) {
    pref_->internal_.reset(new osx::internal);
  }
  osx::internal &internal = *pref_->internal_;

  // setup builtin types.

  const predefined *predef;

  for (predef = default_builtins; predef->name_ != 0; ++predef) {
    if (builtin_exists(internal.builtins_.type_, predef->name_))
      continue;
    osx::type type = construct_type(predef);
    internal.builtins_.type_.push_back(type);
  }


  osx::directives_choice_leaf::binder dirbind(pref_->directives_.c0_);

  osx::_namespace ns;
  if (!namespace_exists(pref_->directives_.c0_, "aka")) {
    ns.prefix_ = "aka";
    ns.uri_ = aka::get_namespace_uri("aka");
    dirbind.push_back(ns, "osx:namespace");
  }
  if (!namespace_exists(pref_->directives_.c0_, "std")) {
    ns.prefix_ = "std";
    ns.uri_ = aka::get_namespace_uri("std");
    dirbind.push_back(ns, "osx:namespace");
  }
  if (!namespace_exists(pref_->directives_.c0_, "osx")) {
    ns.prefix_ = "osx";
    ns.uri_ = aka::get_namespace_uri("osx");
    dirbind.push_back(ns, "osx:namespace");
  }

  osx::include include;
  include.system_ = true;
  include.value_ = "akaxiso2/builtin/builtin.h";
  dirbind.push_back(include, "osx:include");

  include.value_ = "akaxiso2/builtin/schema_builtin.h";
  dirbind.push_back(include, "osx:include");

  include.value_ = "akaxiso2/content_model.h";
  dirbind.push_back(include, "osx:xiso_include");

  include.value_ = "akaxiso2/akaxiso2.h";
  dirbind.push_back(include, "osx:ximpl_include");


  osx::imports_choice_leaf::binder binder(internal.imports_.c0_);
  for (predef = default_imports; predef->name_ != 0; ++predef) {
    if (import_exists(internal.imports_.c0_, predef->name_))
      continue;

    if (predef->type_.resolve_) {
      osx::resolve resolve;
      resolve.name_ = predef->name_;
      resolve.value_ = predef->type_.cpp_;
      binder.push_back(resolve, "osx:resolve");
    }
    else {
      osx::type type = construct_type(predef);
      binder.push_back(type, "osx:type");
    }
  }

  osx::substitutions &sub_elms = pref_->substitutions_;
  for (const char **sub = cpp_keywords; *sub != 0; ++sub) {
    if (substitution_exists(pref_->substitutions_.substitution_, *sub))
      continue;
    osx::substitution sub_elm;
    sub_elm.original_ = *sub;
    sub_elm.substituted_ = std::string("_") + *sub;
    sub_elms.substitution_.push_back(sub_elm);
  }

  osx::escapes &escapes = pref_->escapes_;
  for (const default_escape *es = default_escapes; es->to_escape_ != 0; ++es) {
    if (escape_exists(pref_->escapes_.escape_, es->to_escape_))
      continue;
    osx::escape es_elm;
    es_elm.to_escape_ = es->to_escape_;
    es_elm.escaped_ = es->escaped_;
    escapes.escape_.push_back(es_elm);
  }
}

void osixaka2::commit_preference() {
  assert(pref_.get() != 0);
  registry_.set_preference(*pref_);
}
