#include "processor.h"
#include "exception.h"

#include <algorithm>
#include <akaxiso2/util/scoped_ptr.h>
#include <akaxiso2/util/string_funcs.h>


using namespace osx;


bool processor::is_form_qualified(const std::string &val) {
  if (aka::trim(val) == "unqualified")
    return false;
  if (aka::trim(val) == "qualified")
    return true;

  throw aka::error(val + " does not specify form(qualified/unqualified).", __FILE__, __LINE__);
  return false;
}

bool processor::is_use_required(const std::string &val) {
  return aka::trim(val) == "required";
}

bool processor::is_use_prohibited(const std::string &val) {
  return aka::trim(val) == "prohibited";
}

aka::qname processor::create_attribute_name(const aka::qname &name, 
					    const std::string &form) {

  bool qualified = false;
  if (!form.empty())
    qualified = is_form_qualified(form);
  else if (attribute_qualified_)
    qualified = true;

  if (qualified)
    return name;

  if (name.get_namespace_id() == target_ns_id_)
    return aka::qname(name.local());
  else
    return name;
}


aka::qname processor::create_element_name(const aka::qname &name, 
					  const std::string &form) {
  bool qualified = false;
  if (!form.empty())
    qualified = is_form_qualified(form);
  else if (element_qualified_)
    qualified = true;

  if (qualified)
    return name;

  if (name.get_namespace_id() == target_ns_id_)
    return aka::qname(name.local());
  else
    return name;
}


aka::qname processor::create_anonymous_name(const aka::qname &name, const std::string &postfix,
					    size_t index) {
  std::ostringstream ostm;
  if (!name.local().empty())
    ostm << name.local() << '_';
  ostm << postfix << index;
  return aka::qname(name.get_namespace_id(), ostm.rdbuf()->str());
}

aka::qname processor::create_classname(const aka::qname &name, 
				       const std::string &classname) const {
  if (classname.empty())
    return name;
  return aka::qname(target_ns_id_, classname);
}


aka::qname processor::create_local_name(const std::string &classname) const {
  if (!classname.empty())
    return aka::qname(classname);
  return aka::qname("*");
}


bool processor::is_attrDeclsEmpty(const xs::attrDecls &attrDecls) {
  return attrDecls.c0_.empty() && (attrDecls.anyAttribute_.get() == 0);
}



void processor::scan(registry_unit &unit) {

  current_unit_ = &unit;
  target_ns_id_ = unit.target_ns_id_;
  const xs::schema &schema = *unit.schema_;

  attribute_qualified_ = is_form_qualified(schema.attributeFormDefault_);
  element_qualified_ = is_form_qualified(schema.elementFormDefault_);

  const xs::schemaTops &tops = schema.schemaTops_;
  for (xs::schemaTops::const_iterator it = tops.begin(); 
       it != tops.end(); ++it) {
    const xs::schemaTop &top = it->schemaTop_;

    for (xs::schemaTop::const_iterator tit = top.begin(); tit != top.end(); ++tit) {
      const aka::qname &name = tit->get_name();
      if (name == aka::qname("&redefinable")) {
	const xs::redefinable &redefinable = aka::item_cast<xs::redefinable>(*tit);
	process_redefinable(redefinable);
      }
      else if (name == aka::qname("xs:element")) {
	const xs::topLevelElement &el = 
	  aka::item_cast<const xs::topLevelElement>(*tit);
	aka::qname name(target_ns_id_, el.name_);
	topLevelElements_.add(name, &el);
      }
      else if (name == aka::qname("xs:attribute")) {
	const xs::topLevelAttribute &attr = aka::item_cast<const xs::topLevelAttribute>(*tit);
	aka::qname name(target_ns_id_, attr.name_);
	topLevelAttributes_.add(name, &attr);
      }
      else if (name == aka::qname("xs:notation")) {
	// Nothing to do here.
      }
      else {
	assert(!"Must not reach here.");
      }
    }
  }
}


void processor::process(registry_unit &unit) {

  current_unit_ = &unit;
  target_ns_id_ = unit.target_ns_id_;


  // process namespace prefix entry.
  for (namespaces::iterator it = unit.namespaces_.begin();
       it != unit.namespaces_.end(); ++it) {
    aka::xmlns(it->prefix_, it->uri_);
  }

  const xs::schema &schema = *unit.schema_;

  attribute_qualified_ = is_form_qualified(schema.attributeFormDefault_);
  element_qualified_ = is_form_qualified(schema.elementFormDefault_);

  {
    for (topLevelSimpleTypes::iterator it = topLevelSimpleTypes_.begin();
	 it != topLevelSimpleTypes_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_) {
	process_topLevelSimpleType(it->second.name_, it->second.t_);
      }
    }
  }

  {
    for (topLevelAttributes::iterator it = topLevelAttributes_.begin();
	 it != topLevelAttributes_.end(); ++it)
      if (it->first.get_namespace_id() == target_ns_id_)
	process_topLevelAttribute(it->second.name_, *it->second.t_);
  }
  
  {
    for (topLevelAttributeGroups::iterator it = topLevelAttributeGroups_.begin();
	 it != topLevelAttributeGroups_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_)
	process_topLevelAttributeGroup(it->second.name_, it->second.t_);
    }
  }
  
  {
    for (topLevelComplexTypes::iterator it = topLevelComplexTypes_.begin();
	 it != topLevelComplexTypes_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_) {
	process_topLevelComplexType(it->second.name_, it->second.t_);
      }
    }
  }

  {
    for (namedGroups::iterator it = namedGroups_.begin();
	 it != namedGroups_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_) {
	process_namedGroup(it->second.name_, it->second.t_);
      }
    }
  }
  
  {
    for (topLevelElements::iterator it = topLevelElements_.begin();
	 it != topLevelElements_.end(); ++it)
      if (it->first.get_namespace_id() == target_ns_id_)
	check_reference(*it->second.t_); 
  }

  {
    for (topLevelElements::iterator it = topLevelElements_.begin();
	 it != topLevelElements_.end(); ++it) {
      if (it->first.get_namespace_id() == target_ns_id_)
	process_topLevelElement(it->second.name_, it->second.t_); 
    }
  }

}


type_ref processor::process_topLevelElement(const aka::qname &name, 
					    const xs::topLevelElement *el) {

  element_def *processed_edef = registry_.element_get(name);
  if (processed_edef != 0)
    return processed_edef->type_;
  
  if (verbose_)
    ostm_ << "processing <xs:element name=\"" << name << "\"/>" << std::endl;
    
  el = topLevelElements_.get(name);
  if (el == 0) {
    raise_name_error("xs:element", name, "not defined.", __FILE__, __LINE__);
  }

  if (!el->substitutionGroup_.empty() || !el->block_.empty())
    throw not_supported(__FILE__, __LINE__);

  element_def edef;
  edef.abstract_ = el->abstract_;

  if (!el->type_.empty()) {
    if (!el->c0_.empty())
      throw aka::error("both @type and a particle are specified.", __FILE__, __LINE__);
    edef.type_ = type_ref(el->type_, false, false);
  }
  else if (!el->c0_.empty()) {
    assert(el->type_.empty()); // Must satisfy.
    edef.type_ = process_typeChoice(name, el->c0_, false);
  }
  else {
    /** empty <xs:element name="foo"/> */
    edef.type_ = registry_.get_nill_type();
  }
  assert(!edef.type_.empty());

  if (!el->fixed_.empty()) {
    edef.is_fixed_ = true;
    edef.default_ = el->fixed_;
  }
  else if (!el->default_.empty()) {
    edef.is_fixed_ = false;
    edef.default_ = el->default_;
  }

  processed_edef = current_unit_->elements_.insert(name, edef);
  return processed_edef->type_;
}

void processor::check_reference(const xs::topLevelElement &el) {
  if (el.type_.empty())
    return;
  if (topLevelComplexTypes_.get(el.type_) != 0)
    return;
  if (registry_.is_xs_any(el.type_))
    return;
  if (registry_.get_imported_type(el.type_) != 0)
    return;
  if (registry_.simpleType_get(el.type_) != 0)
    return;

  raise_name_error("type", el.type_, "element@type not found.", __FILE__, __LINE__);
}

void processor::process_redefinable(const xs::redefinable &redefinable) {

  for (xs::redefinable::const_iterator rit = redefinable.begin();
       rit != redefinable.end(); ++rit) {
    const aka::qname &redefname = rit->get_name();
    if (redefname == aka::qname("xs:simpleType")) {
      const xs::topLevelSimpleType &st = 
	aka::item_cast<const xs::topLevelSimpleType>(*rit);
      aka::qname stname(target_ns_id_, st.name_);
      topLevelSimpleTypes_.add(stname, &st);
    }
    else if (redefname == aka::qname("xs:complexType")) {
      const xs::topLevelComplexType &ct = 
	aka::item_cast<const xs::topLevelComplexType>(*rit);
      aka::qname ctname(target_ns_id_, ct.name_);
      topLevelComplexTypes_.add(ctname, &ct);
    }
    else if (redefname == aka::qname("xs:group")) {
      const xs::namedGroup &gr = aka::item_cast<const xs::namedGroup>(*rit);
      aka::qname name(target_ns_id_, gr.name_);
      namedGroups_.add(name, &gr);
    }
    else if (redefname == aka::qname("xs:attributeGroup")) {
      const xs::namedAttributeGroup &ag = 
	aka::item_cast<const xs::namedAttributeGroup>(*rit);
      aka::qname name(target_ns_id_, ag.name_);
      topLevelAttributeGroups_.add(name, &ag);
    }
    else {
      assert(!"Must not reach here.");
    }
  }
}


const gattribute_def *processor::process_topLevelAttribute(const aka::qname &name, 
							   const xs::topLevelAttribute &tat) {

  if (registry_.globalAttribute_get(name) != 0) {
    return registry_.globalAttribute_get(name);
  }

  if (verbose_)
    ostm_ << "processing <xs:attribute name=\"" << name << "\"/>" << std::endl;

  gattribute_def attrdef;

  attrdef.name_ = name;
  if (!tat.type_.empty()) {
    if (tat.simpleType_.get() != 0)
      raise_name_error("xs:attribute", name, "Both @type and xs:simpleType are specified.",
		       __FILE__, __LINE__);
    attrdef.type_ = process_topLevelSimpleType(tat.type_);
  }
  else if (tat.simpleType_.get() != 0) {
    assert(tat.type_.empty());
    attrdef.type_ = process_localSimpleType(name, *tat.simpleType_.get(), 
					    current_unit_->simpleTypes_);
  }
  else {
    // ur-type.
    attrdef.type_ = registry_.get_anySimpleType();
  }

  if (!tat.default_.empty()) {
    attrdef.default_ = tat.default_;
  }
  else if (!tat.fixed_.empty()) {
    attrdef.default_ = tat.fixed_;
    attrdef.is_fixed_ = true;
  }

  return current_unit_->gattributes_.insert(attrdef.name_, attrdef);
}

type_ref processor::process_topLevelSimpleType(const aka::qname &name, 
					       const xs::topLevelSimpleType *st) {

  /**
   * extension/restriction/union are not processed.
   */

  if (registry_.get_imported_type(name) != 0) {
    return type_ref(registry_.get_imported_type(name));
  }
  
  const user_class *us = registry_.simpleType_get(name);
  if (us != 0) {
    assert(us->id_ == simpletype_id);
    return type_ref(us);
  }
  
  if (verbose_)
    ostm_ << "processing <xs:simpleType name=\"" << name << "\"/>" << std::endl;

  st = topLevelSimpleTypes_.get(name);
  
  if (!st->osx_resolve_.empty() && !st->osx_leaf_.empty()) {
    raise_name_error("xs:simpleType", name, "Both @leaf and @resolve are specified.",
		     __FILE__, __LINE__);
  }
  
  /** osx:resolve */
  if (!st->osx_resolve_.empty()) {
    if (!st->osx_classname_.empty()) {
      ostm_ << name 
	    << ": @osx:classname will be ignored because @osx:resolve is specified." << std::endl;
    }
    imported_type *imported = current_unit_->imports_.add(import_resolve, name);
    imported->resolve_ = st->osx_resolve_;
    return type_ref(imported);
  }
  
  /** simpleType is imported by osx:leaf. */
  if (!st->osx_leaf_.empty()) {
    if (!st->osx_classname_.empty()) {
      ostm_ << name 
	    << ": @osx:classname will be ignored because @osx:leaf is specified." << std::endl;
    }
    imported_type *imported = 
      current_unit_->imports_.add(import_user, name);
    if (st->osx_typedef_.empty())
      imported->typedef_ = st->osx_typedef_;
    else
      imported->typedef_ = "std::string";
    imported->leaf_ = st->osx_leaf_;
    return type_ref(imported);
  }
  else {
    // For user class generation.
    type_ref value_type = 
      process_simpleDerivation(name, st->simpleDerivation_, current_unit_->local_simpleTypes_);
    user_class *us = 
      user_class::user_simpletype(name,
				  create_classname(name, st->osx_classname_),
				  value_type, current_unit_->simpleTypes_);
    return type_ref(us);
  }
}

type_ref processor::process_topLevelComplexType(const aka::qname &name, 
						const xs::topLevelComplexType *ct) {

  if (complexType_processing_.find(name) != complexType_processing_.end()) {
    /** cyclic processing. */
    raise_name_error("xs:complexType" , name, "cyclic dependency found.",
		     __FILE__, __LINE__); 
  }  

  const user_class *def = registry_.complexType_get(name);
  if (def != 0)
    return type_ref(def);

  complexType_processing_.insert(name);

  if (ct == 0)
    ct = topLevelComplexTypes_.get(name);
  if (ct == 0)
    raise_name_error("xs:complexType", name, "top-level complexType not defined.",
		     __FILE__, __LINE__);

  if (verbose_)
    ostm_ << "processing <xs:complexType name=\"" << name << "\"/>" << std::endl;

  type_ref type = process_complexTypeModel(name, 
					   create_classname(name, ct->osx_classname_),
					   ct->complexTypeModel_, current_unit_->complexTypes_);
  
  complexType_processing_.erase(name);
  
  return type;
}

const attributeGroup_def *processor::process_topLevelAttributeGroup(const aka::qname &name, 
								    const xs::namedAttributeGroup *ag) {
  if (attrG_processing_.find(name) != attrG_processing_.end()) {
    raise_name_error("xs:attributeGroup", name, "cyclic dependency found.",
		     __FILE__, __LINE__); 
  }

  if (registry_.attributeGroup_get(name) != 0)
    return registry_.attributeGroup_get(name);

  attrG_processing_.insert(name);

  if (ag == 0)
    ag = topLevelAttributeGroups_.get(name);

  if (ag == 0)
    raise_name_error("xs:attributeGroup", name, "undefined type.",
		     __FILE__, __LINE__);

  if (verbose_)
    ostm_ << "processing <xs:attributeGroup name=\"" << name << "\"/>" << std::endl;

  attributeGroup_def def;
  def.name_ = name;
  process_attrDecls(name, def, ag->attrDecls_);
  attrG_processing_.erase(name);
  return current_unit_->attributeGroups_.insert(def.name_, def);
}


void processor::process_attrDecls(const aka::qname &hint,
				  attribute_defs &attrdefs, const xs::attrDecls &decls) {
  
  for (xs::attrDecls_c0::const_iterator ait = decls.c0_.begin();
       ait != decls.c0_.end(); ++ait) {
    if (ait->get_name() == aka::qname("xs:attribute")) {
      const xs::attribute &attr = aka::item_cast<xs::attribute>(*ait);
      process_attribute(hint, attrdefs, attr);
    }
    else if (ait->get_name() == aka::qname("xs:attributeGroup")) {
      const xs::attributeGroupRef &attrGroupRef =
	aka::item_cast<const xs::attributeGroupRef>(*ait);
      process_topLevelAttributeGroup(attrGroupRef.ref_);
      const attribute_defs *grdefs = registry_.attributeGroup_get(attrGroupRef.ref_);
      attrdefs.attributes_.insert(attrdefs.attributes_.end(), 
				  grdefs->attributes_.begin(), grdefs->attributes_.end());
      if (grdefs->has_anyAttribute_) {
	attrdefs.has_anyAttribute_ = grdefs->has_anyAttribute_;
	attrdefs.namespace_list_ = grdefs->namespace_list_;
      }
    }
    else {
      assert(!"Must not reach here.");
    }
  }

  xs::wildcard *anyAttribute = decls.anyAttribute_.get();

  /**
  if (attrdefs.has_anyAttribute_ && (anyAttribute != 0)) {
    std::string message = "xs:anyAttribute is declared in xs:attributeGroup";
    message += "\"" + hint.qualified() + "\".";
    throw fatal_error(message, __FILE__, __LINE__);
  }
  */

  if (anyAttribute != 0) {
    attrdefs.has_anyAttribute_ = true;
    attrdefs.namespace_list_ = anyAttribute->namespace_;
  }
}

attribute_type processor::get_attribute_type(const aka::qname &hint, const xs::attribute &attr) {
  
  attribute_type attrtype;
  if (!attr.name_.empty())
    attrtype.name_ = create_attribute_name(aka::qname(attr.name_), attr.form_);
  else {
    // referenced attribute has global name.
    attrtype.name_ = attr.ref_;
  }
  
  /** Get type */
  if (!attr.ref_.empty()) {
    if (!attr.type_.empty() || (attr.simpleType_.get() != 0))
      raise_name_error("xs:attribute", hint, "both @type and xs:simpleType are specified.", __FILE__, __LINE__);
    const gattribute_def *gattr = registry_.globalAttribute_get(attr.ref_);
    if (gattr == 0) {
      raise_name_error("xs:attribute", attr.ref_, "global attribute not declared.", 
		       __FILE__, __LINE__);
    }
    attrtype.type_ = gattr->type_;
    attrtype.default_ = gattr->default_;
    attrtype.is_fixed_ = gattr->is_fixed_;
  }
  else if (!attr.type_.empty()) {
    if (attr.simpleType_.get() != 0)
      raise_name_error("xs:attribute", attrtype.name_, "both @type and xs:simpleType are specified.", 
		       __FILE__, __LINE__);
    attrtype.type_ = process_topLevelSimpleType(attr.type_);
  }
  else if (attr.simpleType_.get() != 0) {
    aka::qname name(hint);
    name.local() += "_" + attr.name_;
    attrtype.type_ = 
      process_localSimpleType(name, *attr.simpleType_.get(), current_unit_->simpleTypes_);
  }
  else {
    // ur-type.
    attrtype.type_ = registry_.get_anySimpleType();
  }

  if (!attr.default_.empty()) {
    if (is_use_required(attr.use_))
      raise_name_error("xs:attribute", attrtype.name_, "@use=required but @default is specified.",
		       __FILE__, __LINE__);
    attrtype.default_ = attr.default_;
  }
  else if (!attr.fixed_.empty()) {
    attrtype.default_ = attr.fixed_;
    attrtype.is_fixed_ = true;
  }
  else if (is_use_required(attr.use_)) {
    attrtype.required_ = true;
  }

  return attrtype;
}

void processor::process_attribute(const aka::qname &hint,
				  attribute_defs &attrdefs, const xs::attribute &attr) {
  attribute_type attrtype = get_attribute_type(hint, attr);
  attrdefs.attributes_.push_back(attrtype);
}

void processor::restrict_attrDecls(const aka::qname &hint,
				   attribute_defs &attrdefs, const xs::attrDecls &attrDecls) {

  for (xs::attrDecls_c0::const_iterator ait = attrDecls.c0_.begin();
       ait != attrDecls.c0_.end(); ++ait) {
    if (ait->get_name() == aka::qname("xs:attribute")) {
      
      const xs::attribute &attr = aka::item_cast<xs::attribute>(*ait);

      if (is_use_prohibited(attr.use_)) {
	/**
	 * allow appearance of @type.
	 */

	// name, ref both must not be empty. or both must not be non-empty.
	if (attr.name_.empty() && attr.ref_.empty())
	  raise_name_error("xs:attribute", aka::qname(attr.name_), "Both @name and @ref are specified.",
			   __FILE__, __LINE__);
	if (!attr.name_.empty() && !attr.ref_.empty())
	  raise_name_error("xs:attribute", aka::qname(attr.name_), "Neither @name nor @ref is specified.",
			   __FILE__, __LINE__);
	if (!attr.name_.empty())
	  prohibit_attribute(attrdefs, aka::qname(attr.name_));
	else if (!attr.ref_.empty()) {
	  prohibit_attribute(attrdefs, attr.ref_);
	}
      }
      else {
	attribute_type atype = get_attribute_type(hint, attr);
	restrict_attribute(attrdefs, atype);
      }
    }
    else if (ait->get_name() == aka::qname("xs:attributeGroup")) {
      const xs::attributeGroupRef &ref = 
	aka::item_cast<const xs::attributeGroupRef>(*ait);
      const attributeGroup_def *def = registry_.attributeGroup_get(ref.ref_);
      for (attribute_types::const_iterator it = def->attributes_.begin();
	   it != def->attributes_.end(); ++it) {
	const attribute_type &atype = *it;
	restrict_attribute(attrdefs, atype);
	// Don't have to consider restriction, because attributeGroup does not have @use.
      }
    }
    else {
      assert(!"Must not reach here.");
    }
  }

  xs::wildcard *ana = attrDecls.anyAttribute_.get();
  if (ana != 0) {
    if (attrdefs.has_anyAttribute_)
      attrdefs.namespace_list_ = ana->namespace_;
    else {
      raise_name_error("xs:restriction", hint, "Try to extend anyAttribute in xs:restriction.",
		       __FILE__, __LINE__);
    }
  }
}

void processor::restrict_attribute(attribute_defs &attrdefs, 
				   const attribute_type &atype) {
  
  attribute_type *to_restrict = attrdefs.get(atype.name_);
  if (to_restrict == 0)
    raise_name_error("xs:attribute", atype.name_, "could not restrict because undefined in super-particle.",
		     __FILE__, __LINE__);  
  to_restrict->required_ = atype.required_;
  to_restrict->default_ = atype.default_;
  to_restrict->is_fixed_ = atype.is_fixed_;

  /**
   * xs:restriction of xs:anyAttribute is not considered.
   */
}

void processor::prohibit_attribute(attribute_defs &attrdefs,
				   const aka::qname &tagname) {
  attrdefs.erase(tagname);
}

void processor::extend_attrDecls(const aka::qname &name,
				 attribute_defs &attrdefs, const xs::attrDecls &attrDecls) {

  for (xs::attrDecls_c0::const_iterator ait = attrDecls.c0_.begin();
       ait != attrDecls.c0_.end(); ++ait) {
    if (ait->get_name() == aka::qname("xs:attribute")) {
      const xs::attribute &attr = aka::item_cast<xs::attribute>(*ait);
      extend_attribute(name, attrdefs, attr);
    }
    else if (ait->get_name() == aka::qname("xs:attributeGroup")) {
      const xs::attributeGroupRef &agref = aka::item_cast<xs::attributeGroupRef>(*ait);
      const xs::namedAttributeGroup *ag = topLevelAttributeGroups_.get(agref.ref_);
      extend_attrDecls(name, attrdefs, ag->attrDecls_);
    }
    else {
      assert(!"Must not reach here.");
    }
  }
}


void processor::extend_attribute(const aka::qname &name,
				 attribute_defs &attrdefs, const xs::attribute &attr) {

  if (!attr.name_.empty() && ! attr.ref_.empty())
    raise_name_error("xs:attribute", name, "both @name and @ref are specified.", 
		     __FILE__, __LINE__);
  
  if (!attr.name_.empty()) {
    /**
     * Attribute extension is not strictly checked. 
     *  - existence of superclass attribute declarations are not checked.
     */
    if (attrdefs.get(aka::qname(attr.name_)) == 0)
      process_attribute(name, attrdefs, attr);
  }
  else if (!attr.ref_.empty()) {
    /**
     * Attribute extension is not strictly checked. 
     *  - existence of superclass attribute declarations are not checked.
     */
    if (attrdefs.get(attr.ref_) == 0)
      process_attribute(name, attrdefs, attr);
  }
  
  /**
   * anyAttribute is not processed.
   */
}


type_ref processor::process_namedGroup(const aka::qname &name, const xs::namedGroup *gr) {
  
  const user_class *us = registry_.group_get(name);
  if (us != 0)
    return type_ref(us);

  if (gr == 0)
    gr = namedGroups_.get(name);
  if (gr == 0)
    raise_name_error("xs:group", name, "undefined type.", __FILE__, __LINE__);

  if (verbose_) {
    ostm_ << "processing <xs:group name=\"" << name << "\" ";
    if (!gr->osx_classname_.empty())
      ostm_ << "osx:classname=\"" << gr->osx_classname_ << "\"";
    ostm_ << "\"/>" << std::endl;
  }

  type_ref type;

  assert(gr->c0_.size() == 1);
  const aka::item &item = gr->c0_.front();
  if (aka::item_of(item, "xs:sequence")) {
    const xs::simpleExplicitGroup &seg = aka::item_cast<xs::simpleExplicitGroup>(item);
    if (!seg.osx_classname_.empty()) {
      ostm_ << name << ": xs:sequence@osx:classname under xs:group is ignored." << std::endl;
    }
    return process_sequence(name, 
			    create_classname(name, gr->osx_classname_),
			    aka::occurrence(), seg.nestedParticle_, current_unit_->groups_);
  }
  else if (aka::item_of(item, "xs:choice")) {
    const xs::simpleExplicitGroup &seg = aka::item_cast<xs::simpleExplicitGroup>(item);
    if (!seg.osx_classname_.empty()) {
      ostm_ << name << ": xs:choice@osx:classname under xs:group is ignored." << std::endl;
    }
    return process_choice(name, 
			  create_classname(name, gr->osx_classname_),
			  aka::occurrence(), seg.nestedParticle_, current_unit_->groups_);
  }
  else if (aka::item_of(item, "xs:all")) {
    const xs::namedGroup_sequence_c0_all &all = 
      aka::item_cast<xs::namedGroup_sequence_c0_all>(item);
    if (!all.osx_classname_.empty()) {
      ostm_ << name << ": xs:all@osx:classname under xs:group is ignored." << std::endl;
    }
    return process_all(name, create_classname(name, gr->osx_classname_),
		       aka::occurrence(), all.allModel_.element_, current_unit_->groups_);
  }

  assert(!"Must not reach here.");
  return type;
}

type_ref processor::process_typeChoice(const aka::qname &name, const xschoice &choice, 
				       bool is_toplevel) {
  if (choice.size() != 1)
    raise_name_error("xs:complexType", name, "two or more particles are specified.",
		     __FILE__, __LINE__);

  const aka::item &item = *choice.begin();
  if (aka::item_of(item, "xs:simpleType")) {
    user_classes *uss = is_toplevel ? 
      &current_unit_->simpleTypes_ : &current_unit_->local_simpleTypes_;
    return process_localSimpleType(name, 
				   aka::item_cast<const xs::localSimpleType>(item),
				   *uss);
  }
  else if (aka::item_of(item, "xs:complexType")) {
    user_classes *uss = is_toplevel ? 
      &current_unit_->complexTypes_ : &current_unit_->local_complexTypes_;
    return process_localComplexType(name, aka::item_cast<const xs::localComplexType>(item),
				    *uss);
  }
  assert(!"Must not reach here.");
  return type_ref();
}

type_ref processor::process_localSimpleType(const aka::qname &name, 
					    const xs::localSimpleType &lst,
					    user_classes &uss) {
  
  /**
   * akaxiso/osixaka does not support simpleType validation.
   * simpleTypes are simply a typedef-name of superclass or imported type.
   */

  if (!lst.osx_resolve_.empty()) {
    if (!lst.osx_classname_.empty()) {
      ostm_ << name 
	    << ": @osx:classname will be ignored because @osx:resolve is specified." << std::endl;
    }
    imported_type *imported = current_unit_->imports_.add(import_resolve, name);
    imported->resolve_ = lst.osx_resolve_;
    return type_ref(imported);
  }

  type_ref type = process_simpleDerivation(name, lst.simpleDerivation_, uss); 
  user_class *def = 
    user_class::user_simpletype(name, 
				create_classname(name, lst.osx_classname_), 
				type, uss);
  return type_ref(def);
}

type_ref processor::process_localComplexType(const aka::qname &name,
					     const xs::localComplexType &lct,
					     user_classes &uss) {
  return process_complexTypeModel(name, 
				  create_classname(name, lct.osx_classname_),
				  lct.complexTypeModel_, uss);
}


type_ref processor::process_complexTypeModel(const aka::qname &name, 
					     const aka::qname &classname,
					     const xs::complexTypeModel &model,
					     user_classes &uss) {
  
  if (model.size() == 0) { // <xs:complexType name="hoge"/>
    const user_class *def = user_class::user_sequence(name, classname, uss);
    return type_ref(def);
  }
  
  type_ref type;
  assert(model.size() == 1);
  const aka::item &item = model.front();
  if (aka::item_of(item, "xs:simpleContent")) {
    const xs::simpleContent &sc = aka::item_cast<xs::simpleContent>(item);
    type = process_simpleContent(name, classname, sc, uss);
  }
  else if (aka::item_of(item, "xs:complexContent")) {
    const xs::complexContent &cc = aka::item_cast<xs::complexContent>(item);
    type = process_complexContent(name, classname, cc, uss);
  }
  else if (aka::item_of(item, "&complexTypeModelParticle")) {
    const xs::complexTypeModelParticle &cdef = 
      aka::item_cast<xs::complexTypeModelParticle>(item);
    user_class *seqdef = user_class::user_sequence(name, classname, uss);
    
    element_type etype = process_complexTypeParticle(name, cdef.typeDefParticle_);
    if (!etype.empty())
      seqdef->etypes_.push_back(etype);
    process_attrDecls(name, seqdef->adefs_, cdef.attrDecls_);
    type = type_ref(seqdef);
  }

  assert(type.us_ != 0);
  type.us_->check_dup_attributes();
  return type;
}

type_ref processor::process_simpleContent(const aka::qname &name,
					  const aka::qname &classname,
					  const xs::simpleContent &sc,
					  user_classes &uss) {
  const xs::simpleContent_choice &choice = sc.c0_;
  assert(choice.size() == 1);

  const aka::item &item = choice.front();
  if (aka::item_of(item, "xs:extension"))
    return process_simpleExtensionType(name, classname,
				       aka::item_cast<xs::simpleExtensionType>(item), uss);
  else if (aka::item_of(item, "xs:restriction"))
    return process_simpleRestrictionType(name, classname,
					 aka::item_cast<xs::simpleRestrictionType>(item), uss);
  
  assert(!"Must not reach here.");
  return type_ref();
}


type_ref processor::process_simpleExtensionType(const aka::qname &name, 
						const aka::qname &classname,
						const xs::simpleExtensionType &se,
						user_classes &uss) {
  const user_class *super_class = registry_.simpleType_get(se.base_);
  if (super_class != 0) {
    if (super_class->id_ != simplecontent_id)  {
      /** currently osixaka2 XML-Schema processor does not confirm 
       * simpleType inheritance by simpleContent. */
      throw not_implemented(__FILE__, __LINE__);
    }

    user_class *def = user_class::derive(super_class, name, classname, uss);
    process_attrDecls(name, def->adefs_, se.attrDecls_);
    return type_ref(def);
  }

  const imported_type *imported = registry_.get_imported_type(se.base_);
  if (imported != 0) { 
    user_class *def = 
      user_class::user_simplecontent(name, classname, type_ref(imported), uss);
    process_attrDecls(name, def->adefs_, se.attrDecls_);
    return type_ref(def);
 }

  raise_name_error("xs:extention", se.base_, "Failed to find type.", 
		   __FILE__, __LINE__);
  return type_ref();
}

type_ref processor::process_simpleRestrictionType(const aka::qname &name, 
						  const aka::qname &classname,
						  const xs::simpleRestrictionType &sr,
						  user_classes &uss) {
  const user_class *super_class = registry_.simpleType_get(sr.base_);
  if (super_class != 0) {
    if (super_class->id_ != simpletype_id)
      raise_name_error("xs:simpleType/xs:restriction", name, "only simpleType is restricted.",
		       __FILE__, __LINE__);
    user_class *def = user_class::derive(super_class, name, classname, uss);

    /**
     * !!! osixaka does not support prohibit checks.
     */
    return type_ref(def);
  }

  const imported_type *imported = registry_.get_imported_type(sr.base_);
  if (imported != 0) {
    user_class *def = user_class::user_simplecontent(name, classname, type_ref(imported), uss);
    return type_ref(def);
  }

  raise_name_error("xs:restriction", sr.base_, "failed to find type.",
		   __FILE__, __LINE__);
  return type_ref();
}


element_type processor::process_complexTypeParticle(const aka::qname &name,
						    const xs::typeDefParticle &particle) {
  
  if (particle.empty()) {
    return element_type();
  }

  assert(particle.size() == 1); // Must contain one paticle.
  const aka::item &item = particle.front();
  if (aka::item_of(item, "xs:group")) {
    element_type etype;
    const xs::groupRef &gr = aka::item_cast<xs::groupRef>(item);
    etype.type_ = process_namedGroup(gr.ref_); // Make sure referred group processed.
    etype.name_ = aka::qname(gr.ref_.local());
    etype.occ_ = occurrence(gr);
    etype.is_group_ = true;
    return etype;
  }

  element_type el;
  type_ref particle_type;

  if (aka::item_of(item, "xs:all")) {
    const xs::all &all = aka::item_cast<xs::all>(item);
    if (!all.osx_classname_.empty()) {
      ostm_ << name 
	    << ": xs:all@osx:classname might be ignored during akaxisonization." << std::endl;
    }
    aka::qname allname = name;
    allname.local() += "_all";
    particle_type = process_all(allname, 
				create_classname(allname, all.osx_classname_),
				occurrence(all), all.allModel_.element_,
				current_unit_->local_classes_);
    el.occ_ = aka::occurrence(); //don't use occurrence(all);
    el.name_ = aka::qname("*");
  }
  else if (aka::item_of(item, "xs:choice")) {
    const xs::explicitGroup &gr = aka::item_cast<xs::explicitGroup>(item);
    if (!gr.osx_classname_.empty()) {
      ostm_ << name 
	    << ": xs:choice@osx:classname might be ignored during akaxisonization." << std::endl;
    }
    aka::qname choicename(name);
    choicename.local() += "_choice";
    particle_type = process_choice(choicename, 
				   create_classname(choicename, gr.osx_classname_),
				   occurrence(gr), gr.nestedParticle_,
				   current_unit_->local_classes_);
    el.occ_ = aka::occurrence(); //don't use occurrence(gr);
    el.name_ = aka::qname("*");
  }
  else if (aka::item_of(item, "xs:sequence")) {
    const xs::explicitGroup &gr = aka::item_cast<const xs::explicitGroup>(item);
    if (!gr.osx_classname_.empty()) {
      ostm_ << name 
	    << ": xs:sequence@osx:classname might be ignored during akaxisonization." << std::endl;
    }
    aka::qname seqname = name;
    seqname.local() += "_sequence";
    particle_type = process_sequence(seqname, 
				     create_classname(seqname, gr.osx_classname_),
				     occurrence(gr), gr.nestedParticle_,
				     current_unit_->local_classes_);
    el.occ_ = aka::occurrence(); //don't use occurrence(gr);
    el.name_ = aka::qname("*");
  }
  else {
    assert(!"Must not reach here.");
  }
  
  el.type_ = particle_type;
  el.is_group_ = true;
  return el;
}


type_ref processor::process_sequence(const aka::qname &name, 
				     const aka::qname &classname,
				     const aka::occurrence &occ,
				     const xs::nestedParticle &np,
				     user_classes &uss) {
  if (is_array(occ)) {
    // item.
    aka::qname item_name = registry_.create_particle_item_name(name);
    aka::qname item_classname = registry_.create_particle_item_name(classname);
    user_class *itemdef = user_class::user_sequence(item_name, item_classname, uss); 
    process_nestedParticle(itemdef->etypes_, item_name, np);

    // xs:sequence as array.
    user_class *seqdef = user_class::user_array(name, classname, 
						occ, type_ref(itemdef),
						uss);
    return type_ref(seqdef);
  }
  else {
    user_class *def = user_class::user_sequence(name, classname, uss);
    process_nestedParticle(def->etypes_, name, np);
    return type_ref(def);
  }
}

type_ref processor::process_all(const aka::qname &name, 
				const aka::qname &classname,
				const aka::occurrence &occ,
				const xs::narrowMaxMin_array &ams,
				user_classes &uss) {

  if (is_array(occ)) {
    // item.
    aka::qname item_name = registry_.create_particle_item_name(name);
    aka::qname item_classname = registry_.create_particle_item_name(classname);
    user_class *itemdef = user_class::user_all(item_name, item_classname, uss);
    process_all_members(itemdef->etypes_, item_name, ams);
    
    // xs:all as array.
    user_class *alldef = user_class::user_array(name, classname, 
						occ, type_ref(itemdef),
						uss);
    return type_ref(alldef);
  }
  else {
    user_class *def = user_class::user_all(name, classname, uss);
    process_all_members(def->etypes_, name, ams);
    def->check_dup_elements();
    return type_ref(def);
  }

}

type_ref processor::process_choice(const aka::qname &name, 
				   const aka::qname &classname,
				   const aka::occurrence &occ,
				   const xs::nestedParticle &np,
				   user_classes &uss) {

  user_class *def = user_class::user_choice(name, classname, occ, uss);
  process_nestedParticle(def->etypes_, name, np);
  return type_ref(def);
}

type_ref processor::process_simpleDerivation(const aka::qname &name, 
					     const xs::simpleDerivation &der,
					     user_classes &uss) {
  assert(der.size() == 1);
  const aka::item &item = der.front();
  if (aka::item_of(item, "xs:restriction")) {
    const xs::restriction &re = aka::item_cast<xs::restriction>(item);
    if (!re.base_.empty() && (re.simpleRestrictionModel_.simpleType_.get() != 0))
      raise_name_error("xs:restriction", name, "both xs:restriction@base and local simpleType is specified.",
		       __FILE__, __LINE__);
    if (re.simpleRestrictionModel_.simpleType_.get() != 0)
      return process_simpleRestrictionModel(name, re.simpleRestrictionModel_, uss);
    else if (!re.base_.empty()) {
      // facets ignored.
      return process_topLevelSimpleType(re.base_);
    }
    else {
      raise_name_error("xs:simpleType", name, "Neither xs:restriction@base nor local simpleType specified.",
		       __FILE__, __LINE__);
    }
  }
  return registry_.get_anySimpleType();
}

type_ref processor::process_simpleRestrictionModel(const aka::qname &name,
						   const xs::simpleRestrictionModel &srm,
						   user_classes &uss) {
  aka::qname restricted_name = name;
  restricted_name.local() += "_restricted";
  
  // facets ignored.
  return process_localSimpleType(restricted_name, *srm.simpleType_.get(), uss);
}

void processor::process_all_members(element_types &etypes, 
				    const aka::qname &hint, const xs::narrowMaxMin_array &ams) {

  for (xs::narrowMaxMin_array::const_iterator it = ams.begin();
       it != ams.end(); ++it) {
    element_type membertype = 
      process_allElementMember(*it, hint);
    etypes.push_back(membertype);
  }
}


element_type processor::process_localElement(const xs::localElement &el, const aka::qname &hint) {
  // block, form, nillable is ignored.

  type_ref type;
  aka::qname name;

  if (!el.ref_.empty()) {
    if (!el.type_.empty() || !el.c0_.empty())
      raise_name_error("xs:element", el.ref_, "@type or a particle specified though @ref is specified.",
		       __FILE__, __LINE__);
    // referred type is top-level element.
    type = type_ref(el.ref_, false, true);
    name = el.ref_;
  }
  else if (!el.type_.empty()) {
    name = aka::qname(target_ns_id_, el.name_);

    if (!el.c0_.empty()) {
      raise_name_error("xs:element", name, "both @type and child xs:complexType are specified.",
			__FILE__, __LINE__);
    }

    // top-level complexType/simpleType.
    const user_class *ct = registry_.complexType_get(el.type_);
    const user_class *st = registry_.simpleType_get(el.type_);
    const imported_type *imported = registry_.get_imported_type(el.type_);
    if (ct != 0)
      type = type_ref(ct);
    else if (st != 0)
      type = type_ref(st);
    else if (imported != 0)
      type = type_ref(imported);
    else if (topLevelComplexTypes_.exists(el.type_) || topLevelSimpleTypes_.exists(el.type_))
      type = type_ref(el.type_, false, false);
    if (type.empty()) {
      raise_name_error("xs:element", name, "@type not found.", __FILE__, __LINE__);
    }
  }
  else if (!el.c0_.empty()) {
    name = aka::qname(target_ns_id_, el.name_);
    aka::qname choicename = aka::qname(target_ns_id_, hint.local() + '_' + name.local());
    type = process_typeChoice(choicename, el.c0_, false);
  }
  else {
    /* @ref, @type is not spcified.  content is ur-type. */
    name = aka::qname(target_ns_id_, el.name_);
    type = registry_.get_anyType();
  }

  assert(!name.empty());
  assert(!type.empty());

  element_type et;

  et.name_ = create_element_name(name, el.form_);
  et.type_ = type;
  et.occ_ = aka::occurrence(el.minOccurs_, el.maxOccurs_);

  if (!el.fixed_.empty()) {
    if (!el.default_.empty()) {
      raise_name_error("xs:element", name, "both @fixed and @default are specified.",
		       __FILE__, __LINE__);
    }
    et.default_ = el.fixed_;
    et.is_fixed_ = true;
  }
  else if (!el.default_.empty()) {
    et.default_ = el.default_;
  }
  return et;
}


element_type processor::process_allElementMember(const xs::narrowMaxMin &ame, 
						 const aka::qname &hint) {
  xs::localElement lel;

  lel.attributes_ = ame.attributes_;
  lel.id_ = ame.id_;
  lel.annotation_ = ame.annotation_;
  lel.ref_ = ame.ref_;
  lel.name_ = ame.name_;
  lel.c0_ = ame.c0_;
  lel.identityConstraint_ = ame.identityConstraint_;
  lel.type_ = ame.type_;
  lel.default_ = ame.default_;
  lel.fixed_ = ame.fixed_;
  lel.nillable_ = ame.nillable_;
  lel.block_ = ame.block_;
  lel.form_ = ame.form_;
  lel.minOccurs_ = ame.minOccurs_;
  lel.maxOccurs_ = ame.maxOccurs_;

  return process_localElement(lel, hint);
}


element_type processor::process_member_groupRef(const xs::groupRef &ref) {
  element_type et;
  et.name_ = aka::qname(ref.ref_.local());
  et.type_ = process_namedGroup(ref.ref_);
  et.occ_ = aka::occurrence(ref.minOccurs_, ref.maxOccurs_);
  et.is_group_ = true;
  return et;
}

element_type processor::process_any(const xs::any &any) {
  element_type et;
  et.occ_ = aka::occurrence(any.minOccurs_, any.maxOccurs_);
  et.name_ = aka::qname("any");
  et.type_ = registry_.get_anyType();
  et.is_group_ = true;
  et.namespace_list_ = any.namespace_;
  return et;
}

type_ref processor::process_complexContent(const aka::qname &name, 
					   const aka::qname &classname,
					   const xs::complexContent &cc,
					   user_classes &uss) {
  const xs::complexContent_choice &contents = cc.c0_;
  type_ref type;
  assert(contents.size() == 1);
  const aka::item &item = contents.front();
  if (aka::item_of(item, "xs:restriction")) {
    type = process_complexRestrictionType(name, 
					  classname,
					  aka::item_cast<xs::complexRestrictionType>(item),
					  uss);
  }
  else if (aka::item_of(item, "xs:extension")) {
    type = process_extensionType(name, 
				 classname,
				 aka::item_cast<xs::extensionType>(item),
				 uss);
  }
  else {
    assert(!"Must not reach here.");
  }
  return type;
}

type_ref processor::process_complexRestrictionType(const aka::qname &name, 
						   const aka::qname &classname,
						   const xs::complexRestrictionType &cr,
						   user_classes &uss) {
  type_ref super_class = process_topLevelComplexType(cr.base_);

  /**
   * !!! xs:restriction checks are not implemented.
   */

  user_class *def = user_class::user_sequence(name, classname, uss);
  element_type etype;
  etype = process_complexTypeParticle(name, cr.typeDefParticle_);
  
  if (!etype.empty())
    def->etypes_.push_back(etype);
  def->adefs_ = super_class.us_->adefs_;
  
  restrict_attrDecls(name, def->adefs_, cr.attrDecls_);
  return type_ref(def);
}

type_ref processor::process_extensionType(const aka::qname &name, 
					  const aka::qname &classname,
					  const xs::extensionType &et,
					  user_classes &uss) {

  type_ref super_class = process_topLevelComplexType(et.base_);
  user_class *def = user_class::derive(super_class.us_, name, classname, uss);
  
  if (def->etypes_.empty()) {
    element_type etype = 
      process_complexTypeParticle(name, et.typeDefParticle_);
    if (!etype.empty())
      def->etypes_.push_back(etype);
  }
  else {
    element_type el = process_complexTypeParticle(name, et.typeDefParticle_);
    if (!el.empty()) {
      aka::qname particlename(name);
      aka::qname particle_classname(classname);
      particlename.local() += "_particle";
      particle_classname.local() += "_particle";
      user_class *particle_def = user_class::user_sequence(particlename, particle_classname,
							  current_unit_->local_classes_);
      particle_def->etypes_ = def->etypes_;
      particle_def->etypes_.push_back(el);

      element_type child_el;
      child_el.name_ = aka::qname(name.local()); // extended member name.
      child_el.type_ = type_ref(particle_def);
      child_el.is_group_ = true;
      def->etypes_.clear();
      def->etypes_.push_back(child_el);
    }
  }
  extend_attrDecls(name, def->adefs_, et.attrDecls_);
  return type_ref(def);

  /**
   * !!! @block should be checked here, though osixaka2 does not support.
   */

}


void processor::process_nestedParticle(element_types &etypes, const aka::qname &hint, 
				       const xs::nestedParticle &np) {
  
//   size_t all_index = 0;
  size_t choice_index = 0;
  size_t sequence_index = 0;

  for (xschoice::const_iterator it = np.begin(); 
       it != np.end(); ++it) {
    const aka::item &item = *it;
    element_type el;


    if (aka::item_of(item, "xs:element")) {
      el = process_localElement(aka::item_cast<const xs::localElement>(item), hint);
    }
    else if (aka::item_of(item, "xs:group")) {
      el = process_member_groupRef(aka::item_cast<const xs::groupRef>(item));
    }
    else if (aka::item_of(item, "xs:choice")) {
      aka::qname choicename = create_anonymous_name(hint, "c", choice_index);
      el = process_localChoice(choicename, aka::item_cast<const xs::explicitGroup>(item));
      ++choice_index;
    }
    else if (aka::item_of(item, "xs:sequence")) {
      aka::qname sequencename = create_anonymous_name(hint, "s", sequence_index);
      el = process_localSequence(sequencename, aka::item_cast<const xs::explicitGroup>(item));
      ++sequence_index;
    }
    else if (aka::item_of(item, "xs:any")) {
      el = process_any(aka::item_cast<const xs::any>(item));
      el.is_group_ = true;
    }
    else if (aka::item_of(item, "xs:all")) {
      // "xs:all" should not be in nested particles. 
      assert(!"Must not reach here.");
    }
    else {
      assert(!"Must not reach here.");
    }
    etypes.push_back(el);
  }
//   etypes.check_dup_names();
}


element_type processor::process_localAll(const aka::qname &name, const xs::all &all) {
  aka::qname classname = create_classname(name, all.osx_classname_);
  type_ref type = process_all(name, classname,
			      occurrence(all), all.allModel_.element_, 
			      current_unit_->local_classes_);
  element_type et;
  et.name_ = create_local_name(all.osx_classname_);
  et.occ_ = aka::occurrence(); //occurrence(all);
  et.type_ = type;
  et.is_group_ = true;
  return et;
}

element_type processor::process_localChoice(const aka::qname &name, const xs::explicitGroup &cho) {
  aka::qname classname = create_classname(name, cho.osx_classname_);
  type_ref type = process_choice(name, classname,
				 occurrence(cho), cho.nestedParticle_,
				 current_unit_->local_classes_);
  element_type et;
  et.name_ = create_local_name(cho.osx_classname_);
  et.occ_ = aka::occurrence(); //occurrence(cho);
  et.type_ = type;
  et.is_group_ = true;
  return et;
}

element_type processor::process_localSequence(const aka::qname &name, 
					      const xs::explicitGroup &seq) {
  aka::qname classname = create_classname(name, seq.osx_classname_);
  type_ref type = process_sequence(name, classname,
				   occurrence(seq), seq.nestedParticle_,
				   current_unit_->local_classes_);
  element_type et;
  et.occ_ = aka::occurrence(); //occurrence(seq);
  et.name_ = create_local_name(seq.osx_classname_);
  et.type_ = type;
  et.is_group_ = true;
  return et;
}
