#include "xml_serializer.h"
#include "../../util/scoped_ptr.h"
#include "../../framework/closure.h"
#include "../../framework/any_op.h"
#include "../../framework/member_cache.h"
#include "../../framework/namespaces.h"
#include "../../util/sstream.h"
#include "../../framework/literal.h"
#include "../../framework/document_factory.h"
#include "../../framework/entity_complements.h"

using namespace aka2;


void aka2::serialize(const document &doc, std::ostream &ostm) {
  aka2::xml_serializer ser;
  ser.serialize(doc, ostm);
}


xml_serializer::xml_serializer(const std::string &encoding,
			       transcoder_factory create)
  : serializer(2, encoding, create), ns11_enabled_(true) {
}


xml_serializer::~xml_serializer() {}


void xml_serializer::serialize(const document &doc, std::ostream &ostm) {
  prepare(ostm);
  serialize_internal(doc.get_root().ptr(), doc.get_props(), doc.get_name());
  finish();
}


void xml_serializer::do_serialize(const void *e, const qname &name, 
				  std::ostream &ostm) {
  const element_props &props = system_document_factory().get_props(name);
  prepare(ostm);
  serialize_internal(e, props, name);
  finish();
}


void xml_serializer::serialize_internal(const void *e, const element_props &props, 
					const qname &name) {

  formatter_.write_uni(literal::xmldecl_begin());
  if (!encoding_.empty()) {
    formatter_.write_uni(literal::encoding_begin());
    formatter_.write_uni(ecomp_->to_pivot(encoding_));
    formatter_.write_uni(literal::encoding_end());
  }
  formatter_.write_uni(literal::xmldecl_end());

  write_element(e, props, props.op());

  formatter_.write_uni(literal::new_line());
}

void xml_serializer::write_ns(id_type nsid) {
  pstring line;
  line += literal::space;
  line.append(literal::xmlns());
  const pstring &prefix = ecomp_->get_prefixes().get_prefix(nsid);
  if (!prefix.empty()) {
    line.append(1, literal::colon);
    line.append(prefix);
  }
  line += literal::equal;
  line += literal::quote;
  line.append(get_namespace_uri(pivot, nsid));
  line += literal::quote;
  formatter_.write_uni(line.data(), line.size());
}

void xml_serializer::write_namespace_attributes() {
  for (ids::const_iterator it = nsids_.begin(); it != nsids_.end(); ++it)
    write_ns(*it);
  is_root_ = false;
}

void xml_serializer::write_element(const void *e, const element_props &props,
				   const element_op &op) {

  const qname &tagname = props.get_name();

  switch (op.get_schematype()) {
  case sequence_id: {
    write_sequence(tagname, e, static_cast<const sequence_op&>(op));
    break;
  }
  case choice_id: {
    write_choice(tagname, e, static_cast<const choice_op&>(op));
    break;
  }
  case all_id: {
    const all_op &aop = static_cast<const all_op&>(op);
    write_all(tagname, e, aop);
    break;
  }
  case simplecontent_id: {
    write_simplecontent(tagname, e, static_cast<const simplecontent_op&>(op));
    break;
  }
  case array_id: {
    write_array(e, props, static_cast<const array_op&>(op));
    break;
  }  
  case ptrmember_id: {
    write_ptrmember(e, props);
    break;
  }
  case simpletype_id: {
    write_simpletype(tagname, e, static_cast<const simpletype_op&>(op));
    break;
  }
  case fixed_id: {
    write_fixed_simpletype(props);
    break;
  }
  case any_id: {
    write_any(props, *static_cast<const any*>(e));
    break;
  }
  case any_array_id: {
    write_any_array(props, *static_cast<const any_array*>(e));
    break;
  }
  case enclose_id:
  case disclose_id:
  case wildcard_attribute_id:
  case wildcard_id:
    assert(!"Must not reach here.");
  }
}

void xml_serializer::write_element_entity(const void *e, const element_props &props,
					  const element_op &op) {

  switch (op.get_schematype()) {
  case sequence_id: {
    write_sequence_entity(e, static_cast<const sequence_op&>(op));
    break;
  }
  case choice_id: {
    write_choice_entity(e, static_cast<const choice_op&>(op));
    break;
  }
  case all_id: {
    write_all_entity(e, static_cast<const all_op&>(op));
    break;
  }
  case ptrmember_id: {
    write_ptrmember(e, props);
    break;
  }
  case array_id: {
    if (props.is_element())
      write_array(e, props, static_cast<const array_op&>(op));
    else
      write_group_array(e, props, static_cast<const array_op&>(op));
    break;
  }
  case any_id: {
    write_any(props, *static_cast<const any*>(e));
    break;
  }
  case any_array_id: {
    write_any_array(props, *static_cast<const any_array*>(e));
    break;
  }
  case fixed_id: {
    write_fixed_simpletype(props);
    break;
  }
  case simplecontent_id:
  case simpletype_id:
  case enclose_id:
  case disclose_id:
  case wildcard_attribute_id:
  case wildcard_id: // wildcard is written in write_any.
    assert(!"Must not reach here.");
  }
}

void xml_serializer::write_sequence(const qname &tagname, 
				    const void *e, const sequence_op &sop) {
  if (is_sequence_empty(e, sop.get_member_types())) {
    new_line();
    write_nill_element(tagname, e, sop);
  }
  else { 
    new_line();
    write_beginning_tag(tagname, e, sop);
    inc_indent_level();
    
    write_sequence_entity(e, sop);
    
    dec_indent_level();    
    new_line();
    write_ending_tag(tagname);
  }
}

void xml_serializer::write_sequence_entity(const void *e, const sequence_op &sop) {
  const member_types &members = sop.get_member_types();
  for (member_types::const_iterator it = members.begin(); it != members.end(); ++it) {

    const member_type &mtype = *it;
    const_member_cache cache(e, mtype.op(), mtype.getter());
    cache.prepare();

    switch (mtype.get_schematype())         {
    case enclose_id: {
      new_line();
      write_beginning_tag(mtype.get_name(), 0, enclose_op());
      inc_indent_level();
      break;
    }
    case disclose_id: {
      dec_indent_level();
      new_line();
      write_ending_tag(mtype.get_name());
      break;
    }
    case simpletype_id:
    case simplecontent_id:        {
      assert(mtype.is_element());
      write_element(cache.value(), mtype, mtype.op());
      break;
    }
    case fixed_id: {
      write_fixed_simpletype(mtype);
      break;
    }
    case sequence_id:
    case choice_id:
    case all_id: {
      if (it->is_element())
        write_element(cache.value(), mtype, mtype.op());
      else
        write_element_entity(cache.value(), mtype, mtype.op());
      break;
    }
    case array_id: {
      if (it->is_element())
	write_array(cache.value(), mtype, static_cast<const array_op&>(mtype.op()));
      else
	write_group_array(cache.value(), mtype, static_cast<const array_op&>(mtype.op()));
      break;
    }
    case ptrmember_id: {
      if (it->is_element())
	write_ptrmember(cache.value(), mtype);
      else
	write_ptrmember_entity(cache.value(), mtype);
      break;
    }
    case any_id:
      write_any(mtype, *static_cast<const any*>(cache.value()));
      break;
    case any_array_id:
      write_any_array(mtype, *static_cast<const any_array*>(cache.value()));
      break;
    case wildcard_attribute_id:
    case wildcard_id:
      assert(!"Must not reach here.");
    }
  }
}

void xml_serializer::write_choice(const qname &tagname, 
				  const void *e, const choice_op &cop) {
  if (cop.empty(e)) {
    new_line();
    write_nill_element(tagname, e, cop);
  }
  else {
    new_line();
    write_beginning_tag(tagname, e, cop);
    
    inc_indent_level();

    write_choice_entity(e, cop);

    dec_indent_level();

    new_line();
    write_ending_tag(tagname);
  }
}

void xml_serializer::write_choice_entity(const void *e, const choice_op &cop) {
  scoped_ptr<item_iterator> ait(cop.get_iterator(e));
  while (ait->has_next()) {
    const item *i = ait->next();
    if (i->is_element()) {
      write_element(i->get_node().ptr(), i->get_props(), 
		    i->get_props().op());
    }
    else
      write_element_entity(i->get_node().ptr(), i->get_props(),
			   i->get_props().op());
  }
  ait.reset();
}

void xml_serializer::write_all(const qname &tagname, 
			       const void *e, const all_op &aop) {
  if (is_all_empty(e, aop.get_member_types())) {
    new_line();
    write_nill_element(tagname, e, aop);
  }
  else {
    new_line();
    write_beginning_tag(tagname, e, aop);
    
    inc_indent_level();
    
    write_all_entity(e, aop);
    
    dec_indent_level();
    
    new_line();
    write_ending_tag(tagname);
  }
}

void xml_serializer::write_all_entity(const void *e, const all_op &aop) {
  const member_types &mmap = aop.get_member_types();
  for (member_types::const_iterator it = mmap.begin();
       it != mmap.end(); ++it) {
    const member_type &mtype = *it;

    assert(it->is_element());

    const_member_cache cache(e, mtype.op(), mtype.getter());
    cache.prepare();
    
    switch (mtype.get_schematype()) {
    case simpletype_id:
    case simplecontent_id:
    case sequence_id:
    case choice_id:
    case all_id:
    case array_id:
    case ptrmember_id: 
    case fixed_id:
      write_element(cache.value(), mtype, mtype.op());
      break;
    case any_id:
      write_any(mtype, *static_cast<const any*>(cache.value()));
      break;
    case any_array_id:
      write_any_array(mtype, *static_cast<const any_array*>(cache.value()));
      break;
    case enclose_id:
    case disclose_id:
    case wildcard_attribute_id:
    case wildcard_id:
      assert(!"Must not reach here.");
    }
  }
}

void xml_serializer::write_simplecontent(const qname &tagname,
					 const void *e, const simplecontent_op &sop) {
  const member_type &vt = sop.get_valuetype();
  const_member_cache cache(e, vt.op(), vt.getter());
  cache.prepare();

  schematype_id id = vt.get_schematype();
  std::ostringstream ostring;

  switch (id) {
  case simpletype_id: {
    const simpletype_op &simop = static_cast<const simpletype_op&>(vt.op());
    simop.write_text(cache.value(), ostring, *ecomp_);
    break;
  }
  case fixed_id: {
    vt.write_fixed_string(ostring, *ecomp_);
    break;
  }
  case simplecontent_id:
  case sequence_id:
  case choice_id:
  case all_id:
  case array_id:
  case ptrmember_id: 
  case enclose_id:
  case disclose_id:
  case any_id:
  case any_array_id:
  case wildcard_attribute_id:
  case wildcard_id:
    assert(!"must not reach here.");
  }

  const std::string &value = ostring.rdbuf()->str();
  if (value.empty()) {
    new_line();
    write_nill_element(tagname, e, sop);
  }
  else {
    new_line();
    write_beginning_tag(tagname, e, sop);
    formatter_.write_text_entity(ecomp_->to_pivot(value));
    write_ending_tag(tagname);
  }
}

void xml_serializer::write_array(const void *e, 
				 const element_props &props, 
				 const array_op &aop) {
  scoped_ptr<array_iterator> ait(aop.get_iterator(e));
  const element_op &iop = aop.get_item_op();
  while (ait->has_next()) {
    const void *item = ait->next();
    write_element(item, props, iop);
  }
  ait.reset();
}


void xml_serializer::write_group_array(const void *e, 
				       const element_props &props, 
				       const array_op &aop) {
  scoped_ptr<array_iterator> ait(aop.get_iterator(e));
  const element_op &iop = aop.get_item_op();
  assert(iop.get_schematype() != simpletype_id);
  while (ait->has_next()) {
    const void *item = ait->next();
    write_element_entity(item, props, iop);
  }
  ait.reset();
}


void xml_serializer::write_ptrmember(const void *e, const element_props &props) {
  const ptrmember_op &pop = static_cast<const ptrmember_op&>(props.op());
  if (!pop.is_null(e)) {
    const void *deref_e = pop.dereference(e);
    const element_op &vop = pop.get_value_op();
    write_element(deref_e, props, vop);
  }
}

void xml_serializer::write_ptrmember_entity(const void *e, const element_props &props) {
  const ptrmember_op &pop = static_cast<const ptrmember_op&>(props.op());
  if (!pop.is_null(e)) {
    const void *deref_e = pop.dereference(e);
    const element_op &vop = pop.get_value_op();
    write_element_entity(deref_e, props, vop);
  }
}


void xml_serializer::write_simpletype(const qname &tagname,
				      const void *e, const simpletype_op &sop) {
  std::ostringstream ostring;
  sop.write_text(e, ostring, *ecomp_);
  if (ostring.rdbuf()->str().empty()) {
    new_line();
    write_nill_element(tagname, e, sop);
  }
  else {
    new_line();
    write_beginning_tag(tagname, e, sop);
    const std::string &lcp_entity = ostring.rdbuf()->str();
    formatter_.write_text_entity(ecomp_->to_pivot(lcp_entity));
    write_ending_tag(tagname);
  }
}


void xml_serializer::write_fixed_simpletype(const element_props &props) {
  std::ostringstream ostm;
  props.write_fixed_string(ostm, *ecomp_);
  const element_op &vop = static_cast<const element_op&>(props.op());

  if (ostm.rdbuf()->str().empty()) {
    new_line();
    write_nill_element(props.get_name(), 0, vop);
  }
  else {
    new_line();
    write_beginning_tag(props.get_name(), 0, vop);
    const std::string &lcp_entity = ostm.rdbuf()->str();
    formatter_.write_text_entity(ecomp_->to_pivot(lcp_entity));
    write_ending_tag(props.get_name());
  }
}


void xml_serializer::write_beginning_tag(const qname &tagname, 
					 const void *e, const element_op &op) {
  static const pchar_t opening_tag0[] = {'<', 0};
  static const pchar_t opening_tag1[] = {'>', 0};

  inc_ns_depth();

  formatter_.write_uni(opening_tag0, 1);
  formatter_.write_uni(tagname.qualified(pivot, ecomp_->get_prefixes()));

  if (is_root_)
    write_namespace_attributes();

  if (ns11_enabled_)
    write_ns11_declaration(tagname, op, e);

  write_attributes(e, op);

  formatter_.write_uni(opening_tag1, 1);
}

void xml_serializer::write_ending_tag(const qname &tagname) {
  static const pchar_t closing_tag0[] = { '<', '/', 0 };
  static const pchar_t closing_tag1[] = { '>', 0 };
  formatter_.write_uni(closing_tag0, 2);
  formatter_.write_uni(tagname.qualified(pivot));
  formatter_.write_uni(closing_tag1, 1);
  dec_ns_depth();
}

void xml_serializer::write_nill_element(const qname &tagname, 
					const void *e, const element_op &op) {

  static const pchar_t nill_tag0[] = { '<', 0 };
  static const pchar_t nill_tag1[] = { '/', '>', 0 };

  inc_ns_depth();

  formatter_.write_uni(nill_tag0, 1);
  formatter_.write_uni(tagname.qualified(pivot, ecomp_->get_prefixes()));

  if (is_root_)
    write_namespace_attributes();

  if (ns11_enabled_)
    write_ns11_declaration(tagname, op, e);

  write_attributes(e, op);

  formatter_.write_uni(nill_tag1, 2);
  dec_ns_depth();
}

void xml_serializer::write_attributes(const void *elm, const element_op &op) {
  if (op.get_attribute_types() != 0) {
    const attribute_types &attrs = *op.get_attribute_types();
    for (attribute_types::const_iterator it = attrs.begin(); it != attrs.end(); ++it) {
      const attribute_type &attr = *it;
      const element_op *op = &attr.op();
      const ptrmember_op *pop = 0;

      if (attr.get_schematype() == ptrmember_id) {
	pop = &static_cast<const ptrmember_op&>(attr.op());
	op = &pop->get_value_op();
      }

      assert((op->get_schematype() == simpletype_id) || 
	     (op->get_schematype() == fixed_id));
      
      if (op->get_schematype() == fixed_id) {
        if (attr.is_required()) {
	  formatter_.write_pchar(literal::space);

          pstring line;
          line.append(it->get_name().qualified(pivot, ecomp_->get_prefixes()));
	  line += literal::equal;
	  line += literal::quote;
          formatter_.write_uni(line);

	  /** !!!!!!!! */
          std::ostringstream ostring;
          it->write_fixed_string(ostring, *ecomp_);
	  const std::string &fixed_lcp = ostring.rdbuf()->str();
          formatter_.write_attribute_entity(ecomp_->to_pivot(fixed_lcp));
          formatter_.write_pchar(literal::quote);
        }
      }
      else {
	const_member_cache cache(elm, attr.op(), attr.getter());
	cache.prepare();
	
	const void *mem;
	if (pop != 0) {
	  /* dereference elm */
	  if (pop->is_null(cache.value())) // attribute does not appear, then continue.
	    continue;
	  mem = pop->dereference(cache.value());
	}
	else {
	  mem = cache.value();
	}
	std::ostringstream ostring;
	static_cast<const simpletype_op*>(op)->write_text(mem, ostring, *ecomp_);
	bool nothing_to_write = 
	  ostring.rdbuf()->str().empty() || attr.is_attribute_default(elm);
	
	if ((!nothing_to_write) || attr.is_required()) {
	  const pstring &prefix = attr.get_name().qualified(pivot, ecomp_->get_prefixes());
	  const std::string lcp_entity = ostring.rdbuf()->str();
	  formatter_.write_pchar(literal::space);
	  formatter_.write_uni(prefix);
	  formatter_.write_pchar(literal::equal);
	  formatter_.write_pchar(literal::quote);
	  formatter_.write_attribute_entity(ecomp_->to_pivot(lcp_entity));
	  formatter_.write_pchar(literal::quote);
	}
      }
    }
  }

  if (op.get_anyattr_type() != 0) {
    const attribute_type &atype = *op.get_anyattr_type();
    const_member_cache cache(elm, atype.op(), atype.getter());
    cache.prepare();
    const wc_attributes &anyattrs = *static_cast<const wc_attributes*>(cache.value());
    
    for (wc_attributes::const_iterator anyit = anyattrs.begin();
         anyit != anyattrs.end(); ++anyit) {
      formatter_.write_pchar(literal::space);
      formatter_.write_uni(anyit->name_.qualified(pivot, ecomp_->get_prefixes()));
      formatter_.write_pchar(literal::equal);
      formatter_.write_pchar(literal::quote);
      formatter_.write_attribute_entity(ecomp_->to_pivot(anyit->value_));
      formatter_.write_pchar(literal::quote);
    }
  }
}


void xml_serializer::write_ns_decl_if_new(id_type nsid) { 
  if (is_new_ns(nsid)) {
    write_ns(nsid);
    use_temp_nsdecl(nsid);
  }
}


/* any */
void xml_serializer::write_ns11_declaration(const qname &tagname, const element_op &op, 
					    const void *e) {
  id_type tagname_id = tagname.get_namespace_id();
  write_ns_decl_if_new(tagname_id);

  if (op.get_schematype() == any_id) {
    assert(!"Not implelented."); // !!!!!!!!!!!!!!!!!!!!!!!!!!!!
  }
  else if (op.get_schematype() == wildcard_id) {
    const wildcard &wc = *static_cast<const wildcard*>(e);
    for (wc_attributes::const_iterator it = wc.attributes_.begin();
	 it != wc.attributes_.end(); ++it)
      write_ns_decl_if_new(it->name_.get_namespace_id());
  }
  else {
    const attribute_types *atypes = op.get_attribute_types();
    if (atypes == 0)
      return;
    for (attribute_types::const_iterator it = atypes->begin();
         it != atypes->end(); ++it) {
      if (it->get_schematype() == fixed_id) {
        if (it->is_required())
      	  write_ns_decl_if_new(it->get_name().get_namespace_id());
      }
      else {
        std::ostringstream ostring;
      	const_member_cache cache(e, it->op(), it->getter());
	      cache.prepare();
        const element_op *eop = &it->op();
    	if (eop->get_schematype() == ptrmember_id) {
        const ptrmember_op *pop = static_cast<const ptrmember_op*>(eop);
        if (!pop->is_null(cache.value())) {
          const void *value = pop->dereference(cache.value());
          const element_op &vop = pop->get_value_op();
          assert(vop.get_schematype() == simpletype_id);
          static_cast<const simpletype_op&>(vop).write_text(value, ostring, *ecomp_);
        }          
      }
    	else {
		    assert(eop->get_schematype() == simpletype_id);
        static_cast<const simpletype_op&>(*eop).write_text(cache.value(), ostring, *ecomp_);
     	}

    	bool nothing_to_write = true;
	    ostring.rdbuf()->str().empty() || it->is_attribute_default(e) ;
    	if (!nothing_to_write || it->is_required())
	      write_ns_decl_if_new(it->get_name().get_namespace_id());
      }
    }

    const attribute_type *atype = op.get_anyattr_type();
    if (atype != 0) {
      const_member_cache cache(e, atype->op(), atype->getter());
      cache.prepare();
      const wc_attributes *attrs = static_cast<const wc_attributes*>(cache.value());
      for (wc_attributes::const_iterator it = attrs->begin(); it != attrs->end(); ++it) {
	if (!it->value_.empty())
	  write_ns_decl_if_new(it->name_.get_namespace_id());
      }
    }
  }
}



void xml_serializer::write_any(const element_props &props, const any &an) {
  if (an.document_.empty()) {
    write_wildcard(props, an.wc_);
  }
  else {
    const document &doc = an.document_;
    const element_props &doc_props = doc.get_props();
    write_element(doc.get_root().ptr(), doc_props, doc_props.op());
  }
}

void xml_serializer::write_any_array(const element_props &props, const any_array &ar) {
  for (any_array::const_iterator it = ar.begin(); it != ar.end(); ++it)
    write_any(props, *it);
}

void xml_serializer::write_wildcard(const element_props &props, const wildcard &wc) {

  if (props.is_element())
    write_anyType(props.get_name(), wc);
  else
    write_anyType(wc.name_, wc);
}

void xml_serializer::write_wildcard_array(const element_props &props, const wc_array &ar) {
  if (ar.empty())
    return;

  if (props.is_element()) {
    for (wc_array::const_iterator it = ar.begin(); it != ar.end(); ++it)
      write_anyType(props.get_name(), *it);
  }
  else {
    for (wc_array::const_iterator it = ar.begin(); it != ar.end(); ++it)
      write_anyType(it->name_, *it);
  }
}

void xml_serializer::write_anyType(const qname &tagname, const wildcard &wc) {
  new_line();

  if ((wc.children_.size() == 0) && wc.value_.empty())
    write_nill_element(tagname, &wc, wildcard_op::dispatcher_);
  else {
    write_beginning_tag(tagname, &wc, wildcard_op::dispatcher_);

    if (wc.children_.size() == 0) 
      formatter_.write_text_entity(ecomp_->to_pivot(wc.value_));
    else {
      inc_indent_level();
      for (wc_array::const_iterator it = wc.children_.begin();
	   it != wc.children_.end(); ++it) {
	write_anyType(it->name_, *it);
      }
      dec_indent_level();
      new_line();
    }
    write_ending_tag(tagname);
  }
}

void xml_serializer::write_anyType_array(const qname &tagname, const wc_array &ar) {
  if (ar.empty())
    return;
  for (wc_array::const_iterator it = ar.begin();
       it != ar.end(); ++it) {
    write_anyType(tagname, *it);
  }      
}



bool xml_serializer::is_sequence_empty(const void *e, const member_types &mtypes) {
  for (member_types::const_iterator it = mtypes.begin();
       it != mtypes.end(); ++it) {
    if (!is_member_empty(e, *it))
      return false;
  }
  return true;
}

bool xml_serializer::is_all_empty(const void *e, const member_types &mmap) {
  for (member_types::const_iterator it = mmap.begin();
       it != mmap.end(); ++it) {
    if (!is_member_empty(e, *it))
      return false;
  }
  return true;
}

bool xml_serializer::is_member_empty(const void *e, const member_type &mtype) {

  const_member_cache cache(e, mtype.op(), mtype.getter());
  cache.prepare();
  
  switch (mtype.get_schematype()) {
  case simpletype_id:
  case simplecontent_id:
    return false;
  case array_id: {
    const array_op &aop = static_cast<const array_op&>(mtype.op());
    if (aop.size(cache.value()) != 0)
      return false;
    break;
  }
  case sequence_id: {
    if (mtype.is_element())
      return false;
    const sequence_op &sop = static_cast<const sequence_op&>(mtype.op());
    if (!is_sequence_empty(cache.value(), sop.get_member_types()))
      return false;
    break;
  }
  case choice_id: {
    if (mtype.is_element())
      return false;
    const choice_op &cop = static_cast<const choice_op&>(mtype.op());
    if (cop.size(cache.value()) != 0)
      return false;
    break;
  }
  case all_id: {
    if (mtype.is_element())
      return false;
    const all_op &aop = static_cast<const all_op&>(mtype.op());
    if (!is_all_empty(cache.value(), aop.get_member_types()))
      return false;
    break;
  }
  case ptrmember_id: {
    const ptrmember_op &pop = static_cast<const ptrmember_op&>(mtype.op());
    if (!pop.is_null(cache.value()))
      return false;
    break;
  }
  case any_array_id: {
    const any_array &ar = *static_cast<const any_array*>(cache.value());
    return ar.empty();
  }
  case enclose_id:
  case disclose_id: {
    return false; // enclose, disclose is always written.
  }
  case any_id: {
    return static_cast<const any*>(cache.value())->empty();
  }
  case wildcard_id: {
    const wildcard &wc = *static_cast<const wildcard*>(cache.value());
    return wc.empty();
  }
  case fixed_id: {
    return false; // !!!!!!!!!!!!!! is_fixed_empty();
  }
  case wildcard_attribute_id:
    assert(!"Must not reach here.");
  }
  return true;
}

