#include "builtin_xiso.h"
#include "../util/platform.h"
#include "../util/sstream.h"
#include "../util/string_funcs.h"
#include "../framework/entity_complements.h"
#include "../content_model.h"
#include "validator.h"

void aka2::initialize_builtins() {
  xiso::leaf<char>::initialize();
  xiso::leaf<unsigned char>::initialize();
  xiso::leaf<short>::initialize();
  xiso::leaf<unsigned short>::initialize();
  xiso::leaf<long>::initialize();
  xiso::leaf<unsigned long>::initialize();
  xiso::leaf<LONGLONG>::initialize();
  xiso::leaf<ULONGLONG>::initialize();
  xiso::leaf<unsigned int>::initialize();
  xiso::leaf<bool>::initialize();
  xiso::leaf<float>::initialize();
  xiso::leaf<double>::initialize();
  xiso::leaf<std::string>::initialize();
  xiso::leaf<aka2::nill>::initialize();
}

namespace xiso {

  void leaf<char>::model() {
    xmltype("aka2:char");
  }

  void leaf<char>::read_unicode_text(void *elm, const aka2::pstring &entity, 
				     aka2::entity_complements &ecomps) {
    long value;
    std::string digits = xs::to_NMTOKEN(entity, "char", ecomps);
    aka2::isstream istm(digits);
    istm >> value;
    if (istm.fail()) {
      throw aka2::error(std::string("Failed to parse char value(") + istm.rdbuf()->str() + ").",
			__FILE__, __LINE__);
    }
    else if ((value < -128) || (127 < value)) {
      throw aka2::error(std::string("char value(") + istm.rdbuf()->str() + ") is out of range.",
			__FILE__, __LINE__);
    }
    char &ch = *static_cast<char*>(elm);
    ch = (char)value;
  }

  void leaf<char>::write_text(const void *elm, std::ostream &ostm, 
			      aka2::entity_complements &ecomp) {
    
    const char &uch = *static_cast<const char*>(elm);
    ostm << (const long)uch;
  }


  void leaf<unsigned char>::model() {
    xmltype("aka2:unsigned_char");
  }

  void leaf<unsigned char>::read_unicode_text(void *elm, const aka2::pstring &entity, 
					      aka2::entity_complements &ecomps) {

    long value;
    std::string digits = xs::to_NMTOKEN(entity, "unsigned char", ecomps);

    aka2::isstream istm(digits);
    istm >> value;
    if (istm.fail()) {
      throw aka2::error(std::string("Failed to parse unsigned char value(") 
			+ istm.rdbuf()->str() + ").",
			__FILE__, __LINE__);
    }
    if ((value < 0) || (255 < value))
      throw aka2::error(std::string("unsigned char value(") + 
			istm.rdbuf()->str() + ") is out of range.",
			__FILE__, __LINE__);
    unsigned char &ch = *static_cast<unsigned char*>(elm);
    ch = (unsigned char)value;

  }

  void leaf<unsigned char>::write_text(const void *elm, std::ostream &ostm, 
				       aka2::entity_complements &ecomp) {
    
    const unsigned char &uch = *static_cast<const unsigned char*>(elm);
    ostm << (const unsigned long)uch;
  }

  void leaf<short>::model() {
    xmltype("aka2:short");
  }

  void leaf<unsigned  short>::model() {
    xmltype("aka2:unsigned_short");
  }

  void leaf<long>::model() {
    xmltype("aka2:long");
  }

  void leaf<unsigned long>::model() {
    xmltype("aka2:unsigned_long");
  }

  void leaf<LONGLONG>::model() {
    xmltype("aka2:long_long");
  }

  void leaf<LONGLONG>::write_text(const void *elm, std::ostream &ostm, 
				  aka2::entity_complements &ecomp) {
    const LONGLONG &ll = *static_cast<const LONGLONG*>(elm);
#ifdef _MSC_VER
    char buf[64];
    sprintf(buf, "%I64d", ll);
    ostm << buf;
#else
    ostm << ll;    
#endif
  }


  void leaf<ULONGLONG>::model() {
    xmltype("aka2:unsigned_long_long");
  }

  void leaf<ULONGLONG>::write_text(const void *elm, std::ostream &ostm, 
				   aka2::entity_complements &ecomp) {
    const ULONGLONG &ull = *static_cast<const ULONGLONG*>(elm);
#ifdef _MSC_VER
    char buf[128];
    sprintf(buf, "%I64u", ull);
    ostm << buf;
#else
    ostm << ull;    
#endif
  }

  void leaf<int>::model() {
    xmltype("aka2:int");
  }

  void leaf<unsigned int>::model() {
    xmltype("aka2:unsigned_int");
  }

  void leaf<bool>::model() {
    xmltype("aka2:bool");
  }

  void leaf<bool>::read_unicode_text(void *elm, const aka2::pstring &entity,
				     aka2::entity_complements &ecomps) {
    bool &bval = *static_cast<bool*>(elm);
    std::string lcpstr = xs::to_token(entity, ecomps);
    if ((lcpstr == "false") || (lcpstr == "0"))
      bval = false;
    else if ((lcpstr == "true") || (lcpstr == "1"))
      bval = true;
    else
      throw aka2::error("Wrong bool value " + aka2::quote(lcpstr) + ".",
			__FILE__, __LINE__);
  }


  void leaf<bool>::write_text(const void *elm, std::ostream &ostm,
			      aka2::entity_complements &ecomp) {
    bool val = *static_cast<const bool*>(elm);
    if (val)
      ostm << "true";
    else
      ostm << "false";
  }
  

  void leaf<float>::model() {
    xmltype("aka2:float");
  }

  void leaf<double>::model() {
    xmltype("aka2:double");
  }

  void leaf<std::string>::model() {
    xmltype("aka2:string");
  }

  void leaf<std::string>::read_unicode_text(void *elm, const aka2::pstring &entity,
					    aka2::entity_complements &ecomp) {
    std::string &value = *reinterpret_cast<std::string*>(elm);
    value = ecomp.to_lcp(entity);
  }

  void leaf<aka2::nill>::model() {
    xmltype("aka2:nill");
  }
} // namespace xiso



