/* -*- c++ -*- */
#ifndef AKAXISO2_CONTENT_MODEL_H__
#define AKAXISO2_CONTENT_MODEL_H__

/**
 * @file akaxiso2/content_model.h
 * @brief @ref datamodel
 */

#include <akaxiso2/configuration.h>
#include <akaxiso2/framework/entity_complements.h>
#include <akaxiso2/framework/simpletype.h>
#include <akaxiso2/framework/any.h>

#include <akaxiso2/framework/attribute.h>
#include <akaxiso2/framework/array.h>
#include <akaxiso2/framework/sequence.h>
#include <akaxiso2/framework/choice.h>
#include <akaxiso2/framework/all.h>
#include <akaxiso2/framework/simplecontent.h>
#include <akaxiso2/framework/document.h>
#include <akaxiso2/framework/document_factory.h>

#include <akaxiso2/builtin/builtin_xiso.h>
#include <akaxiso2/builtin/schema_builtin_xiso.h>


namespace aka2 {

  /**
   * @brief Declare document type.
   *
   * The root class of the document type is specified by the leaf class, L.
   * @param name tag name of the document.
   * @param l leaf class instance.
   * @see @ref document
   */
  template<class L>
  inline void doctype(const std::string &name, const L &l) {
    reset_for_new_document();
    L::initialize();
    system_document_factory().register_document_type(qname(name), 
						     L::dispatcher_);
  }

  /**
   * @brief Declare a xs:anyType document type.
   *
   * The document type is aka2::any (xs:anyType).
   * @param name tag name of the document.
   * @see @ref document
   */
  inline void doctype(const std::string &name) {
    reset_for_new_document();
    system_document_factory().register_document_type(qname(name), wildcard_op::dispatcher_);
  }

  /** 
   * @brief Declare fixed value document type.
   *
   * The document type is fixed simpleType.
   * @param name tag name of the root element.
   * @param l leaf class instance.
   * @param fixed_val fixed value of the root simpleType.  
   * @see @ref document
   */
  template<class L>
  inline void doctype(const std::string &name, const L &l, const std::string &fixed_val) {
    reset_for_new_document();
    system_document_factory().register_document_type(qname(name), 
						     fixed<L>::dispatcher_, 
						     fixed<L>::create_default_op(),
						     fixed_val,
						     system_entity_complements());
  }

  bool element_equals(const void *lhs, const void *rhs, const element_op& op);
  void element_construct(const node &nd);

  /**
   * @brief Compare serializable class instances.
   * @param lhs to compare.
   * @param rhs to compare.
   * @param l leaf class for class T.
   * @return true if serializable members/contents defined in the given leaf class are the same, otherwise false.  
   */
  template<class T, class L>
  inline bool equals(const T &lhs, const T &rhs, const L &l) {
    return element_equals(&lhs, &rhs, L::dispatcher_);
  }

  /**
   * @brief Compare serializable class instances.
   * @param lhs Instance of T to be compared.
   * @param rhs instance of T to be compared.
   * @return true if serializable members/contents defined in the given leaf class are the same, otherwise false.  
   */
  template<class T>
  inline bool equals(const T &lhs, const T &rhs) {
    return element_equals(&lhs, &rhs, xiso::leaf<T>::dispatcher_);
  }

  /**
   * @brief Construct instance.
   *
   * Some serializable member(s) can have default value(s), which is specified in model() method of a leaf class.  This function will initialize members with default value(s) if specified.
   * @param e instance to construct.
   * @param l leaf class for class T.
   */
  template<class T, class L>
  inline void construct_element(T &e, const L &l) {
    element_construct(node(&e, L::dispatcher_));
  }

  /**
   * @brief Construct serializable instance.
   *
   * Some serializable member(s) can have default value(s), which is specified in model() method of a leaf class.  This function will initialize members with default value(s) if given.\n
   * The leaf class of xiso::leaf<T> is used.
   * @param e instance to construct.
   */
  template<class T>
  inline void construct_element(T &e) {
    element_construct(node(&e, xiso::leaf<T>::dispatcher_));
  }

  /**
   * @brief Copy serializable instance.
   * @param dest to be copied.
   * @param src to copy.
   * @param l leaf class of T.
   */
  template<class T, class L>
  inline void copy_element(T &dest, const T &src, const L &l) {
    L::dispatcher_.copy(&dest, &src);
  }

  /**
   * @brief Copy serializable instance.
   *
   * The leaf class of xiso::leaf<T> is used. 
   * @param dest instance of T to be copied.
   * @param src instance of T to copy.
   */
  template<class T>
  inline void copy_element(T &dest, const T &src) {
    xiso::leaf<T>::dispatcher_.copy(&dest, &src);
  }

  /**
   * @brief Replicate serializable instance.
   *
   * A new instance of T is created on heap, and contents of src are copied with leaf class, L.  
   * @param src instance to be replicated.
   * @param l leaf class of T.
   * @return replicated instance.
   */
  template<class T, class L>
  inline T* replicate_element(const T &src, const L &l) {
    void *ret = L::dispatcher_.replicate(&src);
    return static_cast<T*>(ret);
  }

  /**
   * Replicate serializable instance.
   * A new instance of T is created on heap, and contents of src are copied 
   * by using the leaf class of xiso::leaf<L>.
   * @param src instance to be replicated
   * @return replicated instance
   */
  template<class T>
  inline T* replicate_element(const T &src) {
    void *ret = xiso::leaf<T>::dispatcher_.replicate(&src);
    return static_cast<T*>(ret);
  }

} // namespace aka2

/**
 * @namespace aka2
 * @brief     Namespace for akaxiso framework.
 */

/**
 * @namespace aka
 * @brief     Version independent akaxiso namespace name.  Please use aka:: intead of aka2:: 
 *
 * aka = Adaptable tookKit for Abstracted data model.
 */

/**
 * @namespace xiso
 * @brief     Implementation of leaf<> template classes are defined in this namespace.
 *
 * xiso = XML type-Infomation for Serializable Objects.
*/

/**
 * @namespace xs
 * @brief     Implementation of leaf classes for XML-Schema builtin types.
*/


#endif

