/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_ATTRIBUTE_H__
#define AKAXISO2_FRAMEWORK_ATTRIBUTE_H__

/**
 * @file akaxiso2/framework/attribute.h
 * @brief @ref attribute -related classes/templates
 */

#include <akaxiso2/framework/membertype.h>
#include <akaxiso2/framework/ptrmember.h>
#include <akaxiso2/framework/fixed.h>
#include <akaxiso2/framework/any_op.h>

namespace aka2 {


  /**
   * @brief template to define attributes for @ref aka_sequence, 
   * @ref aka_all and @ref aka_simplecontent.
   *
   * This class provides classes/functions to define attributes.
   * for @ref aka_sequence, @ref aka_all and @ref aka_simplecontent .
   * @param L leaf class.
   * @param T value class.
   * @see aka2::sequence<>, aka2::all<>, aka2::simplecontent<>
   */
  template<class L, class T>
  class  attributes {
  public:
	static attribute_type *define_attribute(const std::string &tagname,
					    member_getter *mgetter,
					    element_op &op,
					    default_op *defop) {
	attribute_type attr = aka2::attribute_type(mgetter, op); 
	attr.set_name(qname(tagname));
	if (defop != 0)
	  attr.set_default_op(defop);
	L::attribute_types_.add(attr); // add dup-name check? !!!!!!!!
	return &L::attribute_types_.back();
    }
    
    /**
     * @brief Helper class to define attribute
     *
     * Helper class to define attributes 
     * in aka2::sequence<>, aka2::all<> and aka2::simplecontent<>.
     * @see aka2::attributes::attribute
     */
    struct _attribute {
      /**
       * @brief Constructor for attribute definition
       * @param tagname tag name
       * @param member pointer to a member of a value class of T
       */
      template<class P, class V>
      _attribute(const std::string &tagname, V P::* member) {
      	new_attribute(tagname, member, xiso::leaf<V>());
      }
      
      /**
       * @brief Constructor for attribute definition
       * @param tagname tag name
       * @param member pointer to a member of a value class of T
       * @param vl leaf class for V
       */
      template<class P, class V, class VL>  
      _attribute(const std::string &tagname, V P::* member, const VL& vl) {
	new_attribute(tagname, member, VL());
      }
      
      /**
       * @brief Set default attribute value
       *
       * Specifies default value
       * @param defval default value
       */
      void set_default(const std::string &defval) {
	if (attr_ == 0)
	  throw tagged_error("attribute", tagname_, "could not have a default value.",
			     __FILE__, __LINE__); // attribute already specified.
	attr_->setup_default_value(defval, system_entity_complements());
	attr_->set_required(false); // When default is specified, '@use' should be "optional". 
	attr_ = 0;
      }
      
      /**
       * @brief Set required
       *
       * Specifies required flag.
       * @param val true if required, othrewise false
       */
      void required(bool val) {
	if (attr_ == 0)
	  throw tagged_error("attribute", tagname_, "could not have its use as required.",
		      __FILE__, __LINE__);
	attr_->set_required(val);
	attr_ = 0;
      }

    private:
      template<class P, class V, class VL>
      void new_attribute(const std::string &tagname, V P::*member, const VL) {
	tagname_ = tagname;
	VL::initialize();
	member_getter *getter = 
	  create_ptr_getter(reinterpret_cast<T*>(0), member);
	attr_ = define_attribute(tagname, getter, 
				 VL::dispatcher_, VL::create_default_op());
      }
      attribute_type *attr_;
      std::string tagname_;
    };

    /**
     * @brief typedef name of _attribute.
     *
     * Use attribute instead of _attribute.
     * _attribute is typedef'ed for VC6 workaround.
     */
    typedef _attribute attribute;

    /**
     * @brief Helper class to define attribute
     *
     * Helper class to define attributes 
     * in aka2::sequence<>, aka2::all<> and aka2::simplecontent<>.
     * @see aka2::attributes::attribute
     */
    struct _ptrattribute {
      /**
       * @brief Constructor for attribute definition
       * @param tagname tag name
       * @param member pointer to a member of a value class of T
       */
      template<class P, class V>
      _ptrattribute(const std::string &tagname, V P::* member) {
      	new_ptrattribute(tagname, member, xiso::leaf<V>());
      }

      /**
       * @brief Constructor for attribute definition
       * @param tagname tag name
       * @param member pointer to a member of a value class of T
       * @param vl leaf class for V
       */
      template<class P, class V, class VL>  
      _ptrattribute(const std::string &tagname, V P::* member, const VL& vl) {
	new_ptrattribute(tagname, member, VL());
      }

      /**
       * @brief Set default attribute value
       *
       * Specifies default value
       * @param defval default value
       */
      void set_default(const std::string &defval) {
	if (attr_ == 0)
	  throw tagged_error("attribute", tagname_, "could not have a default value.",
			     __FILE__, __LINE__); // attribute already specified.
	attr_->setup_default_value(defval, system_entity_complements());
	attr_->set_required(false); // When default is specified, '@use' should be "optional". 
	attr_ = 0;
      }

    private:
      template<class P, class V, class VL>
      void new_ptrattribute(const std::string &tagname, V P::*member, const VL&) {
	tagname_ = tagname;
	VL::initialize();
	member_getter *getter = 
	  create_ptr_getter(reinterpret_cast<T*>(0), member);
	element_op &op = aka2::ptrmember_op_stub<V, VL>::dispatcher_;
	attr_ = define_attribute(tagname, getter, op, VL::create_default_op());
      }
      attribute_type *attr_;
      std::string tagname_;
    };

    /**
     * @brief typedef name of _attribute.
     *
     * Use attribute instead of _attribute.
     * _attribute is typedef'ed for VC6 workaround.
     */
    typedef _ptrattribute ptrattribute;

    /**
     * @brief define any attribute.
     *
     * Define wildcard attributes by using aka2::any_attributes 
     * ( <xs:anyAttribute> in XML Schema ).
     * @param member member of aka2::any_attributes
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     */
    template<class P>
    void any_attribute(wc_attributes P::*member, const std::string &ns_list = "##any")  {
      member_getter *mgetter = 
	create_ptr_getter(reinterpret_cast<T*>(0), member);
      any_attributes_ = aka2::attribute_type(mgetter);
      any_attributes_.set_ns_list(ns_list);
    }
    
    /**
     * @brief define fixed attribute.
     *
     * Define attribute with fixed value.\n
     * @param V value class
     */
    template<class V>
    struct fixed_attribute {
      /**
       * @brief Constructor to define fixed attribute
       * @param tagname tag name
       * @param fixed_value fixed value
       */
      fixed_attribute(const std::string &tagname, const std::string &fixed_value) {
	new_attribute(tagname, fixed_value, xiso::leaf<V>());
      }

      /**
       * @brief Constructor to define fixed attribute
       * @param tagname tag name
       * @param fixed_value fixed value
       * @param vl leaf class for V
       */
      template<class VL>
      fixed_attribute(const std::string &tagname, const std::string &fixed_value, const VL &vl) {
	new_attribute(tagname, fixed_value, VL());
      }

      /**
       * @brief Set required
       *
       * Specifies required flag.
       * @param val true if required, othrewise false
       */
      void required(bool val) { attr_->set_required(val); }
    private:
      template<class VL>
      void new_attribute(const std::string &tagname, const std::string &fixed_value, const VL&) {
	VL::initialize();
	member_getter *mg = new null_getter();
	attr_ = define_attribute(tagname, mg, fixed<VL>::dispatcher_, VL::create_default_op());
	attr_->setup_default_value(fixed_value, system_entity_complements());
      }
      attribute_type *attr_;
    };
    
    static const attribute_type* get_anyattr_type() {
      if (any_attributes_.empty())
	return 0;
      return &any_attributes_;
    }
    
    static attribute_types attribute_types_;
    static attribute_type any_attributes_;
  };
  
  template<class L, class T>
  attribute_types attributes<L, T>::attribute_types_;
  
  template<class L, class T>
  attribute_type attributes<L, T>::any_attributes_;

}


#endif
