/* -*- c++ -*- */
#include "document.h"
#include "model_check.h"
#include "document_factory.h"

#include "../util/string_funcs.h"

namespace aka2 {
  extern document_factory g_doc_factory_;
}

using namespace aka2;

document::document(void* root, const element_props &def)
  : root_(node(root, def.op())), props_(&def) {}


document::~document() {
}

const qname &document::get_name() const { 
  return props_->get_name(); 
}

const element_op& document::get_op() const { 
  return props_->op(); 
}


bool aka2::document_of(const aka2::document &i, const std::string &tagname) {
  return aka2::qname(pivot, to_pivot(tagname)) == i.get_name();
}

bool aka2::document_of(const aka2::document &i, const qname &tagname) {
  return tagname == i.get_name();
}

bool operator==(const document &lhs, const document &rhs) {
  if (&lhs.get_op() != &rhs.get_op())
    return false;
  const_node lnd = lhs.get_root();
  const_node rnd = rhs.get_root();
  if (lnd.empty() && rnd.empty())
    return true;
  if (lnd.empty() || rnd.empty())
    return false;

  return lnd.op().equals(lnd.ptr(), rnd.ptr());
}


element_props &document_factory::register_document_type(const qname &docname, 
							const element_op &op){
  std::pair<doctype_map::iterator, bool> res;

  if (op.get_schematype() == choice_id) {
    const choice_op &cop = static_cast<const choice_op&>(op);
    bool emptiable = cop.get_emptiable();
    element_props props(op, emptiable);
    props.set_name(docname);
    props.set_occurrence(cop.get_occurrence());
    res = doctype_map_.insert(doctype_map::value_type(docname, props));
  }
  else {
    element_props props(op, false);
    props.set_name(docname);
    res = doctype_map_.insert(doctype_map::value_type(docname, props));
  }

  if (!res.second) {
    std::string message = "Document " + quote(docname) + " is already registered.";
    throw error(message, __FILE__, __LINE__);
  }

  res.first->second.check_emptiable();
  return res.first->second;
}


void document_factory::register_document_type(const qname &docname, 
					      const element_op& op,
					      default_op *defop,
					      const std::string &defval,
					      entity_complements &ecomps){
  element_props &props = register_document_type(docname, op);
  if (!check_has_default(op.get_schematype())) {
    std::string message = "Document " + quote(docname)
      + " must be simpleType to have default value.";
    throw error(message, __FILE__, __LINE__);
  }
  props.set_default_op(defop);
  props.setup_default_value(defval, ecomps);
  props.check_emptiable();
}

bool document_factory::is_document_type_registered(const qname &docname) const {
  doctype_map::const_iterator it = doctype_map_.find(docname);
  return it != doctype_map_.end();
}

document document_factory::create_named_document(const qname &docname){
  doctype_map::iterator it = doctype_map_.find(docname);
  if (it == doctype_map_.end()) {
    std::string errmsg = "Document " + quote(docname) + " not found.";
    throw error(errmsg.c_str(), __FILE__, __LINE__);
  }
  void* root = it->second.op().create();

  return document(root, it->second);
}

const element_props& document_factory::get_props(const qname &docname) const {
  doctype_map::const_iterator it = doctype_map_.find(docname);
  if (it == doctype_map_.end()) {
    std::string message = "Document " + quote(docname) + " not found.";
    throw error(message, __FILE__, __LINE__);
  }
  return it->second;
}

