/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_QNAME_H__
#define AKAXISO2_FRAMEWORK_QNAME_H__

/**
 * @file akaxiso2/framework/qname.h
 * @brief Qualified name class. 
 */

#include <akaxiso2/framework/types.h>
#include <akaxiso2/transcoders/pivot_transcoder.h>
#include <string>


namespace aka2 {

  class prefix_map;

  /**
   * @brief qualified name.
   *
   * Qualified name has the form of 'prefix:localname', 
   * where 'prefix' is associated with URI.
   * Akaxiso2's qname represent XML's qualified name.
   * Name strings are internally preserved in UTF-8.
   * Users can use Local-code-page versions of methods.
   * If you need custom-encoding conversion, use methods with UTF8-transcoder.
   * Akaxiso2 serves transcoders as aka::utf8 and aka::lcp (Local-code-page).
   */
  class qname {
    friend struct qname_less;

    id_type namespace_id_;
    pstring name_;
    mutable const pchar_t *prefix_cache_;

    void validate_name() const;

    void set(const pivot_tag &trc, const pstring &rawname, const prefix_map &pfs);
    void set(const pivot_tag &trc, const pstring &uri, const pstring &localname);
    void set(const pivot_tag &trc, id_type namespace_id, const pstring &localname);

  public:
    /** @brief default constructor */
    explicit qname() : namespace_id_(empty_token) {}

    /** 
     * @brief constructor by using name as std::string. 
     * @param trc transcoder instance.
     * @param rawname name
     * @exception aka2::error thrown if rawname is in a wrong format, 
     * or namespace prefix not registered.
     */
    explicit qname(const pivot_tag &trc, const pstring &rawname);

    explicit qname(const pivot_tag &trc, const pstring &rawname, const prefix_map &pfs);

    /**
     * @brief constructor by using uri and localname.
     * @param trc transcoder instance.
     * @param uri namespace URI
     * @param localname local name
     */
    explicit qname(const pivot_tag &trc, const pstring &uri, const pstring &localname);

    explicit qname(const pivot_tag &trc, id_type namespace_id, const pstring &localname);
    
    /**
     * @brief return qualified name.
     * @return returns qualified name if qualified, or localname if not qualified. 
     */
    pstring qualified(const pivot_tag &trc) const;

    pstring qualified(const pivot_tag &trc, const prefix_map &pfmap) const;

    /**
     * @brief get prefix of this name.
     * @param trc transcoder instance.
     * @return prefix string.
     */
    const pstring &prefix(const pivot_tag &trc) const;

    const pstring &prefix(const pivot_tag &trc, const prefix_map &pfs) const;
    
    const id_type get_namespace_id() const { return namespace_id_; }

    /**
     * @brief get local name.
     * @param trc transcoder instance.
     * @return value-copied local name.
     */
    const pstring &local(const pivot_tag &trc) const;


    /** 
     * @brief constructor by using name as std::string. 
     * @param rawname name
     * @exception aka2::error thrown if rawname is in a wrong format, 
     * or namespace prefix not registered.
     */
    explicit qname(const std::string &rawname);

    explicit qname(const std::string &rawname, const prefix_map &pfs);

    /**
     * @brief constructor by using uri and localname.  (Local-code-page version)
     * @param uri namespace URI
     * @param localname local name
     */
    explicit qname(const std::string &uri, const std::string &localname);

    explicit qname(id_type namespace_id, const std::string &localname);

    /**
     * @brief return qualified name.  (Local-code-page version)
     * @return returns qualified name if qualified, or localname if not qualified. 
     */
    std::string qualified() const;

    std::string qualified(const prefix_map &pfmap) const;

    /**
     * @brief get prefix of this name.  (Local-code-page version)
     * @return prefix string.
     */
    std::string prefix() const;

    std::string prefix(const prefix_map &pfs) const;
    
    /**
     * @brief get local name (Local-code-page version)
     * @return value-copied local name.
     */
    const std::string local() const;


    bool operator ==(const qname &rhs) const {
      return (namespace_id_ == rhs.namespace_id_) && (name_ == rhs.name_);
    }
    bool operator !=(const qname &rhs) const {
      return (namespace_id_ != rhs.namespace_id_) || (name_ != rhs.name_);
    }

    /** 
     * @brief empty.
     * @return true if name string is empty, otherwise false.
     */
    bool empty() const;

    /**
     * @brief returns true if qname is qualified.
     * @return true if qualified, otherwise false. 
     */
    bool is_qualified() const { return namespace_id_ != empty_token; }

    bool is_element() const { return name_[0] != '&'; }
  };



  struct qname_less {
    bool operator ()(const qname &lhs, const qname &rhs) const {
      if (lhs.namespace_id_ < rhs.namespace_id_)
        return true;
      else if (lhs.namespace_id_ == rhs.namespace_id_)
        return (lhs.name_ < rhs.name_);
      return false;
    }
  };

  std::string quote(const qname &str);
  std::string quote(const qname &str, const prefix_map &pfs);
  std::string tag(const qname &str);
  std::string tag(const qname &str, const prefix_map &pfs);


} // namespace aka2


#endif
