/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_SEQUENCE_H__
#define AKAXISO2_FRAMEWORK_SEQUENCE_H__

/**
 * @file akaxiso2/framework/sequence.h
 * @brief @ref aka_sequence -related classes/templates
 */

#include <akaxiso2/framework/simpletype.h>
#include <akaxiso2/framework/attribute.h>
#include <akaxiso2/framework/memberdef.h>
#include <akaxiso2/framework/closure.h>
#include <akaxiso2/framework/model_check.h>

namespace aka2 {

  template<class L>
  class sequence_op_dispatcher : public sequence_op {
  public:
    virtual schematype_id get_schematype() const { return sequence_id; }
    virtual const qname &get_typename() const { return L::get_xmltype(); }
    /** creatable */
    virtual bool equals(const void *lhs, const void *rhs) const {
      return L::equals(lhs, rhs);
    }

    virtual void construct(void *e) const {  L::construct(e); }
    virtual void copy_construct(void *e, const void *src) const { L::copy_construct(e, src); }
    virtual void destruct(void *e) const { L::destruct(e); }
    virtual size_t class_size() const { return L::class_size(); }

    /** attribute_info getter. */
    virtual const attribute_types *get_attribute_types() const {
      return &L::attribute_types_;
    }
    /** sequence_info getter. */
    virtual const member_types &get_member_types() const {
      return L::member_types_;
    }
    virtual const attribute_type *get_anyattr_type() const { return L::get_anyattr_type(); }
  };


  template<class L, class T>
  struct sequence_statics {
    static member_types member_types_;
    static sequence_op_dispatcher<L> dispatcher_;
  };

  template<class L, class T>
  member_types sequence_statics<L, T>::member_types_;

  template<class L, class T>
  sequence_op_dispatcher<L> sequence_statics<L, T>::dispatcher_;


  /**
   * @brief template to define @ref aka_sequence leaf class.
   *
   * Leaf classes for @ref aka_sequence are derived from this template class.\n
   * Most of Member-defining classes/functions are implemented in aka2::memberdef<L, T>.\n
   * In addition aka2::sequence<> has special member-defining functions, 
   * enclose() and disclose().\n
   * @see @ref aka_sequence
   *
   * @param T value class type
   * @param L leaf class type
   */
  template<class T, class L=xiso::leaf<T> >
  class sequence : public attributes<L, T>,
		   public sequence_statics<L, T>,
		   public memberdef<L, T> {
  public:
    typedef T value_type;

    virtual ~sequence(){}

    static member_type* register_membertype(const member_type &mtype) {
      L::member_types_.add(mtype);
      return &L::member_types_.back();
    }

    static void initialize() {
      if (!system_type_registry().add(L()))
      	return;
      L::member_types_.clear();
      L::attribute_types_.clear();
      L l; l.model();
      check_emptiable(L::member_types_);
      L::member_types_.make_sorted_cache();
      L::attribute_types_.make_sorted_cache();
    }

    static void uninitialize() {
      L::member_types_.clear();
      L::attribute_types_.clear();
    }


    static void construct(void *e) {
      new (e) T();
      sequence_construct(e, L::dispatcher_);
    }
    static void copy_construct(void *e, const void *src) {
      new (e) T(*static_cast<const T*>(src));
    }
    static size_t class_size() { return sizeof(T); }
    static void destruct(void *elm) { static_cast<T*>(elm)->~T(); }

    static bool equals(const void *lhs, const void *rhs) {
      return sequence_equals(lhs, rhs, L::dispatcher_);
    }

    static element_op* get_attribute_dispatcher() { return &L::dispatcher_; }
    static default_op* create_default_op() { return 0; }

    /**
     * @brief define starting tag.
     *
     * Define beginning tag to create element hieralchy.\n
     * @param tagname tag name
     */
    static void enclose(const std::string &tagname) {
      member_type mtype(new null_getter(), enclose_op::op_, false);
      mtype.set_name(qname(tagname));
      L::register_membertype(mtype);
    }

    /**
     * @brief define ending tag.
     *
     * Define ending tag to close element hieralchy.\n
     * @param tagname tag name
     */
    static void disclose(const std::string &tagname) {
      member_type mtype(new null_getter(), disclose_op::op_, false);
      mtype.set_name(qname(tagname));
      L::register_membertype(mtype);
    }
  };

} // namespace aka2

#endif
