/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_SIMPLETYPE_H__
#define AKAXISO2_FRAMEWORK_SIMPLETYPE_H__

/**
 * @file akaxiso2/framework/simpletype.h
 * @brief @ref aka_simpletype -related classes/templates 
 */

#include <akaxiso2/framework/type_registry.h>
#include <akaxiso2/framework/member.h>
#include <akaxiso2/framework/operators.h>
#include <akaxiso2/framework/xmltype.h>
#include <akaxiso2/framework/entity_complements.h>
#include <akaxiso2/util/sstream.h>
#include <iostream>

namespace aka2 {

  template<class V, class VL>
  struct default_value : public default_op {
    typedef V value_type;

    default_value() : has_default_(false) {}
    virtual ~default_value(){}

    virtual void set_default(const std::string &defval, entity_complements &ecomps) {
      pstring pivot = ecomps.to_pivot(defval);
      VL::read_unicode_text(&default_, pivot, ecomps);
      has_default_ = true;
    }

    virtual bool has_default() const {
      return has_default_;
    }

    virtual const void *value() const {
      assert(has_default_);
      return &default_;
    }

  private:
    V default_;
    bool has_default_;
  };


  template<class L>
  struct simpletype_op_dispatcher : public simpletype_op {
  public:
    virtual schematype_id get_schematype() const { return simpletype_id; }
    virtual const qname &get_typename() const { return L::get_xmltype(); }
    virtual const attribute_types *get_attribute_types() const { return 0; }
    virtual const attribute_type *get_anyattr_type() const { return 0; }

    virtual void write_text(const void* elm,
                            std::ostream &ostm,
                            entity_complements &ecomp) const {
      L::write_text(elm, ostm, ecomp);
    }

    virtual void read_unicode_text(void* elm, const pstring &entity,
				   entity_complements &ecomp) const {
      L::read_unicode_text(elm, entity, ecomp);
    }

    virtual void construct(void *e) const {  L::construct(e); }
    virtual void copy_construct(void *e, const void *src) const { L::copy_construct(e, src); }
    virtual void destruct(void *e) const { L::destruct(e); }
    virtual size_t class_size() const { return L::class_size(); }
    virtual bool equals(const void *lhs, const void *rhs) const { return L::equals(lhs, rhs); }
  };


  template<class L>
  struct simpletype_statics {
    static simpletype_op_dispatcher<L> dispatcher_;
  };

  template<class L>
  simpletype_op_dispatcher<L> simpletype_statics<L>::dispatcher_;

  /**
   * @brief template to define @ref aka_simpletype leaf class.
   *
   * Leaf classes for @ref aka_simpletype are derived from this template class.\n
   * @see @ref aka_simpletype
   * @param T value class type
   * @param L leaf class type
   */
  template<class V, class L=xiso::leaf<V> >
  class simpletype : public simpletype_statics<L>, 
		     public xmltype_statics<L> {
  public:
    virtual ~simpletype(){}
    typedef V value_type;

    static void initialize() {
      system_type_registry().add(L());
      L l; l.model();
    }
    static void uninitialize() {}

    void model() { /* Empty default implementation */  }

    /**
     * @brief write element value to std::ostream
     *
     * Cast elm to const V&, and users get variable to write to std::ostream.
     * preconditionals will provide conditionals.
     * If user need custom-writing logic, override this method.
     * @param elm pointer to value
     * @param ostm std::ostream to output.
     * @param ecomps entity_complements of text values.
     * @exception aka2::error thrown when error occurs.
     */
    static void write_text(const void *elm, std::ostream &ostm,
                           entity_complements &ecomps) {
      const V &value = *reinterpret_cast<const V*>(elm);
      ostm << value;
    }

    /**
     * @brief read element value from aka2::isstream.
     *
     * Cast elm to V&, and users get variable to set read value.
     * preconditionals will provide conditionals.
     * If user need custom-writing logic, override this method.
     * @param elm pointer to value
     * @param entity entity text in UCS-2.
     * @param ecomps complements of text values to parse value.
     * @exception aka2::error thrown when error occurs.
     */
    static void read_unicode_text(void *elm, const pstring &entity,
				  entity_complements &ecomps) {
      std::string lcp = ecomps.to_lcp(entity);
      L::read_text(elm, lcp, ecomps);
    }

    static void read_text(void *elm, const std::string &entity,
			  entity_complements &ecomps) {
      V &value = *reinterpret_cast<V*>(elm);
      isstream istm(entity);
      istm >> value;
      if (istm.fail())
        throw error(std::string("Failed to parse simpletype value, \"")
		    + istm.rdbuf()->str() + "\".",
		    __FILE__, __LINE__);
    }



    static void construct(void *e) { new (e) V(); }
    static void copy_construct(void *e, const void *src) {
      new (e) V(*static_cast<const V*>(src));
    }
    static size_t class_size() { return sizeof(V); }
    static void destruct(void *elm) { static_cast<V*>(elm)->~V(); }

    static bool equals(const void *lhs, const void *rhs) {
      return *static_cast<const V*>(lhs) == *static_cast<const V*>(rhs);
    }

    static default_op* create_default_op() { return new default_value<V, L>; }
    static simpletype_op_dispatcher<L> *get_attribute_dispatcher() {
      return &L::dispatcher_;
    }
  };

} // namespace aka2

#endif
