/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_TYPES_H__
#define AKAXISO2_FRAMEWORK_TYPES_H__

/** 
 * @file akaxiso2/framework/types.h
 * @brief declarations of library-global types and functions
 */

#include <akaxiso2/configuration.h>
#include <akaxiso2/exception.h>
#include <akaxiso2/util/iosfwd.h>
#include <akaxiso2/transcoders/transcoder.h>

/** akaxiso data model namespace */
namespace aka2 {

  /** id for token (mainly namespace URI) */
  typedef long id_type;
  /** id for unregistered token */
  extern const id_type unregistered_token;
  /** id for xml namespace URI */
  extern const id_type xml_ns_token;
  /** id for empty URI */
  extern const id_type empty_token;

  class entity_complements;
  class document_factory;
  class type_registry;
  class qname;


  /**
   * get system global entity_complements.
   * @return system global entity_complements instance.
   */
  entity_complements& system_entity_complements();

  /**
   * clone system global entity_complements.
   * @return cloned instance.
   */
  entity_complements* system_entity_complements_clone(bool copy_prefixes, 
						      transcoder_factory trfact);

  document_factory &system_document_factory();
  type_registry &system_type_registry();

  enum schematype_id {
    simpletype_id = 1,
    simplecontent_id,
    array_id,
    sequence_id,
    choice_id,
    all_id,
    ptrmember_id,
    enclose_id,
    disclose_id, 
    any_id,
    any_array_id,
    wildcard_id,
    wildcard_attribute_id,
    fixed_id
  };

  class attribute_types;
  class attribute_type;
  class qname;


  /** @internal
   * operator for element
   */
  struct element_op {
    element_op() {}
    virtual ~element_op() {}
    virtual schematype_id get_schematype() const = 0;
    virtual const qname &get_typename() const = 0;
    virtual bool equals(const void *lhs, const void *rhs) const = 0;
    virtual const attribute_types *get_attribute_types() const = 0;
    virtual const attribute_type *get_anyattr_type() const = 0;

    virtual void construct(void *e) const = 0;
    virtual void copy_construct(void *e, const void *src) const = 0;
    virtual void destruct(void *e) const = 0;
    virtual size_t class_size() const = 0;

    void *create() const {
      void *dest = ::operator new(class_size());
      construct(dest);
      return dest;
    }

    void *replicate(const void *src) const {
      void *dest = ::operator new(class_size());
      copy_construct(dest, src);
      return dest;
    }

    void destroy(void *e) const {
      destruct(e);
      ::operator delete(e);
    }

    void copy(void *e, const void *src) const {
      destruct(e);
      copy_construct(e, src);
    }

  };


  struct default_op {
    virtual ~default_op() { }
    virtual void set_default(const std::string &defval, entity_complements &ecomps) = 0;
    virtual bool has_default() const = 0;
    virtual const void *value() const = 0;
  };

} // namespace aka2


namespace xiso {
  /** forwarded declaration of leaf<> template. */
  template<class T> struct leaf;
}

namespace aka = aka2;

#endif
