#include "encoding_name.h"
#include "../exception.h"
#include <ctype.h>

using namespace aka2;

namespace {
  static const unsigned long endian = 0x12345678;
  bool is_big_endian() {
    return *reinterpret_cast<const char*>(&endian) == 0x12;
  }

  /**
   * Encoding name entries.
   */
  const char* utf16_name    = "UTF16";
  const char* ucs2_name     = "UCS2";
  const char* ucs2be_name   = "UCS2BE";
  const char* ucs2le_name   = "UCS2LE";
  const char* utf16be_name  = "UTF16BE";
  const char* utf16le_name  = "UTF16LE";
  const char* utf8_name     = "UTF8";
  const char* utf7_name     = "UTF7";

  bool encoding_name_eq(const char *name, const std::string &key) {
    return strcmp(name, key.c_str()) == 0;
  }
}


const struct char_mark aka2::utf7_bom    = { "\x2b\x2f\x76\x38\x2d\x0", 5 };
const struct char_mark aka2::utf8_bom    = { "\xef\xbb\xbf", 3 };

const struct char_mark aka2::ucs2be_bom  = { "\xfe\xff", 2 };
const struct char_mark aka2::ucs2le_bom  = { "\xff\xfe", 2 };
const struct char_mark aka2::ucs4be_bom  = { "\x00\x00\xfe\xff", 4 }; 
const struct char_mark aka2::ucs4le_bom  = { "\xff\xfe\x00\x00", 4 };


std::string aka2::create_encoding_key(const std::string &encoding) {
  std::string key;
  for (std::string::size_type pos = 0; pos < encoding.size(); ++pos) {
    char ch = encoding[pos];
    if ((ch == '_') || (ch == '-'))
      continue;
    ch = toupper(ch);
    key.append(&ch, 1);
  }
  return key;
}

  
bool aka2::is_binary_encoding(const std::string &encoding) {
  std::string key = create_encoding_key(encoding);

  if (encoding_name_eq(utf16_name, key))
    return true;
  if (encoding_name_eq(utf16be_name, key))
    return true;
  if (encoding_name_eq(utf16le_name, key))
    return true;
  if (encoding_name_eq(ucs2_name, key))
    return true;
  if (encoding_name_eq(ucs2be_name, key))
    return true;
  if (encoding_name_eq(ucs2le_name, key))
    return true;
  return false;
}
  

bool aka2::is_ucs2(const std::string &encoding) {
  std::string key = create_encoding_key(encoding);

  if (encoding_name_eq(utf16_name, key))
    return true;
  if (encoding_name_eq(ucs2_name, key))
    return true;
  return false;
}

bool aka2::is_utf8(const std::string &encoding) {
  std::string key = create_encoding_key(encoding);
  if (encoding_name_eq(utf8_name, key))
    return true;
  return false;
}

const struct char_mark *aka2::get_unicode_bom(const std::string &encoding) {
  std::string key = create_encoding_key(encoding);
  if (encoding_name_eq(utf7_name, key))
    return &utf7_bom;
  if (encoding_name_eq(utf8_name, key))
    return &utf8_bom;
  if (encoding_name_eq(ucs2_name, key)) {
    if (is_big_endian())
      return &ucs2be_bom;
    else
      return &ucs2le_bom;
  }
  if (encoding_name_eq(utf16_name, key)) {
    if (is_big_endian())
      return &ucs2be_bom;
    else
      return &ucs2le_bom;
  }
  return 0;
}
