#include "transcoder.h"
#include "../configuration.h"
#include "../unicode/uniconv.h"
#include "builtin_transcoders.h"
#include "encoding_name.h"
#include <assert.h>
#include <stdio.h>

using namespace aka2;


void transcoder::set_encoding_types(const std::string &tocode, const std::string &fromcode) {
  if (is_utf8(tocode))
    to_type_ = utf8_type;
  else if (is_ucs2(tocode))
    to_type_ = ucs2_type;
  else
    to_type_ = other_types;

  if (is_utf8(fromcode))
    from_type_ = utf8_type;
  else if (is_ucs2(fromcode))
    from_type_ = ucs2_type;
  else
    from_type_ = other_types;
}


size_t transcoder::utf8_escape(const char *utf8, const char *utf8end) {
  aka2::uchar_t uch;
  size_t length = uniconv::utf8_to_ucs2(utf8, utf8end, &uch);
  assert(length != 0);
  ucs2_escape(uch);
  return length;
}


void transcoder::ucs2_escape(const aka2::uchar_t uch) {
  char *outbuf = outbuf_.get_end_ptr(16);
  int size = sprintf(outbuf, "&#%d;", uch);
  outbuf_.commit_additional_length(size);
}


bool transcoder::clear_bom(const struct char_mark *bom) {
  if (outbuf_.get_content_length() == 0)
    return false;
  if (strncmp(bom->mark_, 
	      outbuf_.get_ptr(), 
	      bom->length_) == 0) {
    const char *real_beginning = outbuf_.get_ptr() + bom->length_;
    outbuf_.erase_by(real_beginning);
  }
  return true;
}


bool transcoder::write_bom(const struct char_mark *bom) {
  if (outbuf_.empty())
    return false;

  if (strncmp(bom->mark_, outbuf_.get_ptr(), 
	      bom->length_) != 0) {
    outbuf_.insert_front(bom->mark_, bom->length_);
  }
  return true;
}


void transcoder::clear_transcoded_buffer() {
  outbuf_.clear();
}


/** ubuffered_transcoder */

ubuffered_transcoder::~ubuffered_transcoder() { }


bool ubuffered_transcoder::transcode(const char *to_convert, size_t length) {

  bool inres = false;
  bool outres = false;

  inbuf_.append(to_convert, length);
  
  switch (from_type_) {
  case utf8_type: {
    size_t length = uniconv::utf8_to_ucs2(ubuffer_, 
					  inbuf_.get_ptr(), inbuf_.get_content_length());
    inbuf_.erase_front(length);
    break;
  }
  case ucs2_type: {
    const aka2::uchar_t *uinbuf = reinterpret_cast<const aka2::uchar_t*>(inbuf_.get_ptr());
    size_t ulength = inbuf_.get_content_length() >> 1;
    ubuffer_.append(uinbuf, ulength);
    inbuf_.erase_front(ulength << 1);
    break;
  }
  case other_types:
    inres = from_inbuf();
    if (!inres)
      throw error("ill sequence.", __FILE__, __LINE__);
  }

  switch (to_type_) {
  case utf8_type: {
    uniconv::ucs2_to_utf8(outbuf_, 
			  ubuffer_.get_ptr(), ubuffer_.get_content_length());
    ubuffer_.clear();
    break;
  }
  case ucs2_type: {
    const char *inbuf = reinterpret_cast<const char*>(ubuffer_.get_ptr());
    size_t length = ubuffer_.get_content_length() << 1;
    outbuf_.append(inbuf, length);
    ubuffer_.clear();
    break;
  }
  case other_types: {
    if (ubuffer_.empty())
      break;
    aka2::uchar_t *pivot_current = ubuffer_.get_ptr();
    while (true) {
      outres = to_outbuf(pivot_current);
      if (!outres) {
        ucs2_escape(*pivot_current);
        ++pivot_current;
      }
      if (pivot_current == ubuffer_.get_end_ptr())
	      break;
    }
  }
  }
  return inbuf_.empty();
}



transcoder *aka2::create_transcoder(const std::string &tocode, 
				    const std::string &fromcode,
				    transcoder_factory tf) {

  std::string to_key = create_encoding_key(tocode);
  std::string from_key = create_encoding_key(fromcode);

  transcoder *tr = builtin_transcoder::create_no_throw(tocode, fromcode);
  if (tr != 0)
    return tr;

  /** delegate to given transcoder factory. */
  if (tf != 0)
    return tf(tocode, fromcode);

  /** default behavior. */
  return default_transcoder::create(tocode, fromcode);
}

