/* -*- c++ -*- */
#ifndef AKAXISO2_UNICODE_UNICONV_H__
#define AKAXISO2_UNICODE_UNICONV_H__

/**
 * @file akaxiso2/unicode/uniconv.h
 * @brief encoding conversion between UTF-8 and UCS-2. 
 */

#include <akaxiso2/configuration.h>
#include <akaxiso2/exception.h>
#include <akaxiso2/unicode/ustring.h>
#include <akaxiso2/unicode/string_buffer.h>
#include <assert.h>

namespace aka2 {


  struct uniconv {

    static std::string ucs2_to_utf8(const ustring &ucs2) {
      return ucs2_to_utf8(ucs2.data(), ucs2.size());
    }

    static ustring utf8_to_ucs2(const std::string &utf8);


    /**
     * Convert UCS2 string to UTF-8 string.
     * @param buffer UCS2 string.
     * @param source of UCS2 string.
     * @param length byte sequence length of source UTF-8 string.  
     *        If -1 is specified, ch should be null-terminated.
     * @return converted UTF-8 string length.
     */
    static int ucs2_to_utf8(string_buffer &buffer, 
			    const aka2::uchar_t *source, int length = -1);

    static std::string ucs2_to_utf8(const aka2::uchar_t *source, int length = -1);

    /**
     * @brief convert UCS2 character to UTF-8 character.
     * @param wc UCS2 character.
     * @param converted converted UTF-8 character. (length is from 1 to 3.)
     */
    static size_t ucs2_to_utf8(const aka2::uchar_t wc, char *converted);


    /**
     * Convert UTF-8 string to UCS-2 string.
     * @param buffer buffer to get converted byte sequence.
     * @param utf8str UTF-8 byte sequence to be converted.
     * @param length length of UTF-8 byte sequence. If -1 is specified, ch should be null-terminated.
     * @return length of converted UCS-2 sequence.
     */
    static int utf8_to_ucs2(string_buffer &buffer, 
			    const char *utf8str, int length = -1);
    
    /**
     * Convert UTF-8 string to UCS-2 string.
     * @param buffer buffer to get converted uchar_t sequence.
     * @param utf8str UTF-8 byte sequence to be converted.
     * @param length length of UTF-8 byte sequence. If -1 is specified, ch should be null-terminated.
     * @return length of converted UCS-2 sequence.
     */
    static int utf8_to_ucs2(ustring_buffer &buffer, 
			    const char *utf8str, int length = -1);
    
    
    /**
     * @brief convert utf8 character 
     * @param utf8str utf8 character to be converted.
     * @param utf8strend position of next character.
     * @param uch pointer to uchar_t to get converted UCS-2 codepoint.
     * @return consumed byte length of UTF-8 buffer.
     * @exception aka2::error thrown when UTF-8 character is not in range of UCS2.
     */
    static size_t utf8_to_ucs2(const char *utf8str, const char *utf8strend, 
			       aka2::uchar_t *uch);
  };
}

#endif
