/* -*- c++ -*- */
#ifndef AKAXISO2_UTIL_SHARED_PTR_H__
#define AKAXISO2_UTIL_SHARED_PTR_H__

/**
 * @file akaxiso2/util/shared_ptr.h
 * @brief shared pointer.
 * @author Tetsu Yoshioka (yoshioka@users.sourceforge.jp)
 *
 * Some extensions ( reset(), null refcnt_... ) added by smorino@users.sourceforge.jp
 */

#include <assert.h>

namespace aka2 {

  template <class T> class shared_ptr {
  public:
    typedef shared_ptr<T> this_type;
    explicit shared_ptr(T* ptr = 0) : ptr_(ptr), refcnt_(0) {
      refcnt_ = ptr ? new int(1) : 0;
    }
    shared_ptr(const shared_ptr<T>& rhs) : ptr_(rhs.ptr_), refcnt_(rhs.refcnt_) {
      inc_count();
    }
    ~shared_ptr() {
      dec_count();
    }

    shared_ptr<T>& operator =(const shared_ptr<T>& rhs) {
      if (ptr_ == rhs.ptr_)
	return *this;

      dec_count();

      ptr_ = rhs.ptr_;
      if (rhs.ptr_) {
	refcnt_ = rhs.refcnt_;
	inc_count();
      }
      else
	refcnt_ = 0;

      return *this;
    }

    bool operator ==(const T *rhs) const { return ptr_ == rhs; }
    bool operator !=(const T *rhs) const { return ptr_ != rhs; }

    bool operator ==(const this_type &rhs) const { 
      if (refcnt_ != rhs.refcnt_) {
	assert(ptr_ != rhs.ptr_);
	return false;
      }
      else {
	assert(ptr_ == rhs.ptr_);
	return true;
      }
    }
    bool operator !=(const this_type &rhs) const { return !(*this == rhs); }

    void reset(T *p = 0) {
      if (ptr_ == p)
	return;

      dec_count();

      ptr_ = p;
      refcnt_ = p ? new int(1) : 0;
    }

    T* get() const { return ptr_; }

    T& operator *() const { return *ptr_; }
    T* operator ->() const { return ptr_; }

  private:
    T* ptr_;
    int* refcnt_;

    void inc_count() {
      if (refcnt_)
	++ *refcnt_;
    }

    void dec_count() {
      if (refcnt_ == 0)
	return;
      if (--*refcnt_ == 0) {
	delete ptr_;
	delete refcnt_;
	ptr_ = 0;
	refcnt_ = 0;
      }
    }
  };

} // namespace aka2

#endif
