#ifdef _MSC_VER
# pragma warning( disable : 4786 )
#endif

#if defined(__unix__) || defined(__linux__) || defined(__APPLE_CC__)
# include <unistd.h>
#else
  extern "C" int getopt (int argc, char *const *argv, const char *shortopts);
  extern "C" int optind;
  extern "C" char	*optarg;
#endif

#include <akaxiso2/akaxiso2.h>
#include <akaxiso2/util/string_funcs.h>
#include <akaxiso2/util/sstream.h>

#include <iostream>
#include <fstream>

#include "osixaka2.h"


static void usage() {

  std::cerr << "osixaka (" << AKAXISO2_PACKAGE << "-" << AKAXISO2_VERSION << ")" << std::endl
	    << " XMLSchema compiler for akaxiso2." << std::endl
	    << " usage  osixaka2 <options> <Schema files...>" << std::endl
	    << "  -d: dump parsed document." << std::endl
	    << "  -p: just parse XMLSchema document." << std::endl
	    << "  -o: specify output directory to write generated files." << std::endl
	    << "  -b: specify base directory for include files." << std::endl
	    << "  -v: verbose mode." << std::endl
	    << "  -n: generate class declarations to seperate files for every namespace." << std::endl
	    << "  -f: generate global function declarations to a specified file." << std::endl
	    << " Preference" << std::endl
	    << "  -T: generate preference template file." << std::endl 
	    << "  -S: save preference to a specified file." << std::endl
	    << "  -L: load preference from a specified file." << std::endl 
	    << "  -1: use osixaka1 preference." << std::endl
	    << "  -M: use MFC-style member name." << std::endl
// 	    << "  -g: generate dot file." << std::endl
	    << " Optimization" << std::endl
	    << "  -a[0-1]: Akaxisonization level. (0: None, 1: Normal.)" << std::endl
	    << "  -i: allow impcomplete types for array element types" << std::endl
	    << "  -x: specify xiso-instancing function name" << std::endl
#ifdef USE_EXPAT
	    << " DTD options" << std::endl
 	    << "  -D: convert DTD to XML-Schema." << std::endl
#endif
	    << "  -h: show this message." << std::endl;

}


int main(int argc, char* argv[]){

  aka::initialize();
  osx::osixaka2::initialize();

  std::ostream &logstm = std::cerr;
  std::string outputdir = "./";
  std::string basedir;
  std::string function_filename;
  std::string load_preference_path;
  std::string save_preference_path;
  std::string instantiate_xiso_func_name;
  bool dtd_ = false;
  bool parse_only = false;
  bool do_dump = false;
  bool verbose = false;
  bool save_preference = false;
  bool generate_template = false;
  bool use_osixaka1_style = false;
  bool use_MFC_style = false;
  bool each_ns = false;
  bool generate_dot = false;
  bool use_soft_array = false;
  bool akaxisonize_level = 1;

  char c;
  while ((c = getopt(argc, argv, "db:po:nf:vS:L:T:Dh1Mga:ix:")) != -1) {
    switch (c) {
    case 'd': {
      do_dump = true;
      break;
    }
    case 'b': {
      basedir = optarg;
      if (basedir.at(basedir.size() - 1) != '/')
	basedir += "/";
      break;
    }
    case 'p': {
      //do_dump = true;
      parse_only = true;
      break;
    }
    case 'o': {
      outputdir = optarg;
      if (outputdir.at(outputdir.size() - 1) != '/')
	outputdir += "/";
      break;
    }
    case 'a': {
      aka::isstream istm(optarg);
      istm >> akaxisonize_level;
      if (istm.fail()) {
	logstm << "Failed to read akaxisonize level." << std::endl;
	usage();
	exit(1);
      }
      if ((akaxisonize_level != 0) &&
	  (akaxisonize_level != 1)) {
	logstm << "Wrong akaxisonize level." << std::endl;
	usage();
	exit(1);
      }
      break;
    }
    case 'n': {
      each_ns = true;
      break;
    }
    case 'f': {
      function_filename = optarg;
      break;
    }
    case 'v': {
      verbose = true;
      break;
    }
    case 'S' : {
      save_preference_path = optarg;
      save_preference = true;
      break;
    }
    case 'L' : {
      load_preference_path = optarg;
      break;
    }
    case 'T' : {
      save_preference_path = optarg;
      generate_template = true;
      break;
    }
    case 'D': {
      dtd_ = true;
      parse_only = true;
      do_dump = true;
      break;
    }
    case '1' : {
      use_osixaka1_style = true;
      break;
    }
    case 'M' : {
      use_MFC_style = true;
      break;
    }
    case 'g': {
      generate_dot = true;
      break;
    }
    case 'h' : {
      usage();
      exit(0);
    }
    case 'i': {
      use_soft_array = true;
      break;
    }
    case 'x': {
      instantiate_xiso_func_name = optarg;
      break;
    }
    default: {
      logstm << "Error : Unknown option." << std::endl;
      exit(1);
    }
    }
  }

  osx::osixaka2 osixaka(std::cerr, use_soft_array, verbose);

  osixaka.set_basedir(basedir);
  osixaka.set_outputdir(outputdir);

  if (!load_preference_path.empty()) {
    /** loading preference. */
    try {
      osixaka.load_preference(load_preference_path);
    }
    catch (const std::exception &e) {
      std::cerr << e.what() << std::endl
                << "Error : Failed to load preference." << std::endl;
      exit(1);
    }
  }
  else if (use_osixaka1_style)
    osixaka.load_osixaka1_defaults();
  else
    osixaka.load_defaults();

  if (use_MFC_style)
    osixaka.use_MFC_member_style();

  if (generate_template) {
    std::ofstream file(save_preference_path.c_str());
    if (!file) {
      std::cerr << "Error : Failed to open preference file." << std::endl;
      exit(1);
    }
    osixaka.generate_template(file);
    file.close();
    exit(0);
  }
  else if (save_preference) {
    std::ofstream file(save_preference_path.c_str());
    if (!file) {
      std::cerr << "Error : Failed to open preference file." << std::endl;
      exit(1);
    }
    osixaka.save_preference(file);
    file.close();
    exit(0);
  }
  
  osixaka.commit_preference();

  if (!instantiate_xiso_func_name.empty()) {
    osixaka.set_instantiate_xiso_name(instantiate_xiso_func_name);
  }

  if (optind == argc) {
    logstm << "Error : No document for generation." << std::endl;
    usage();
    exit(1);
  }

  // process DTD, and exit.
  if (dtd_) {
    if (optind != argc - 1) {
      std::cerr << "Error : On DTD-conversion, only one DTD document must be specified." 
		<< std::endl;
      exit(1);
    }
    if (osixaka.process_dtd(argv[optind]) != 0) {
      logstm << "Error : Failed to process DTD declaration, " << argv[optind] << "." << std::endl;
      exit(1);
    }
    return 0;
  }

  if (function_filename.empty() && (argc - optind != 1)) {
    if (argc - optind != 1) {
      logstm << "Error : Just one schema file should be specified when -f not specified." 
	     << std::endl;
      usage();
      exit(1);
    }
  }

  for (int index = optind; index < argc; ++index) {
    if (osixaka.preprocess(argv[index]) != 0)
      exit(1);
  }    

  if (do_dump) {
    if (osixaka.dump_schemas(std::cout) != 0)
      return 0;
  }
    
  if (parse_only)
    return 0;

  if (osixaka.process() != 0)
    return 0;

  if (osixaka.akaxisonize(akaxisonize_level) != 0)
    return 1;

  if (generate_dot) {
    if (osixaka.generate_dot())
      exit(1);
  }
  else if (!function_filename.empty()) {
    if (osixaka.generate_functions(function_filename)) {
      exit(1);
    }
  }
  else {
    if (osixaka.generate(each_ns) != 0) {
      exit(1);
    }
  }

  std::string target_schema_document;
  if (function_filename.empty())
    target_schema_document = argv[argc - 1];
  else
    target_schema_document = function_filename;

  logstm << target_schema_document << ": Generation completed." << std::endl;
    
  // Uninitializers.
  aka::uninitialize();
  return 0;
}
