#include <akaxiso2/akaxiso2.h>

#include "osixaka2.h"
#include "osx_xiso.h"
#include "registry.h"
#include "imported_types.h"

namespace {
  
  struct value_type {
    const char *cpp_;
    const char *leaf_;
    osx::import_id id_;
  };

  struct array_type {
    const char *cpp_;
    const char *typedef_;
    const char *leaf_;
  };

  struct predefined {
    const char *name_;
    value_type type_;
    array_type array_;
  };

  const predefined default_builtins[] = {
    {"char",             {0, 0, osx::import_intrinsic}, {"aka2:char_array",      "std::vector<char>",           "aka::char_array_leaf"     }},
    {"unsigned char",    {0, 0, osx::import_intrinsic}, {"aka2:uchar_array",     "std::vector<unsigned char>",  "aka::uchar_array_leaf"    }},
    {"short",            {0, 0, osx::import_intrinsic}, {"aka2:short_array",     "std::vector<short>",          "aka::short_array_leaf"    }},
    {"unsigned short",   {0, 0, osx::import_intrinsic}, {"aka2:ushort_array",    "std::vector<unsigned short>", "aka::uchar_array_leaf"    }},
    {"long",             {0, 0, osx::import_intrinsic}, {"aka2:long_array",      "std::vector<short>",          "aka::long_array_leaf"     }},
    {"unsigned long",    {0, 0, osx::import_intrinsic}, {"aka2:ulong_array",     "std::vector<unsigned long>",  "aka::ulong_array_leaf"    }},
    {"LONGLONG",         {0, 0, osx::import_intrinsic}, {"aka2:longlong_array",  "std::vector<LONGLONG>",       "aka::longlong_array_leaf" }},
    {"ULONGLONG",        {0, 0, osx::import_intrinsic}, {"aka2:ulonglong_array", "std::vector<ULONGLONG>",      "aka::ulonglong_array_leaf"}},
    {"int",              {0, 0, osx::import_intrinsic}, {"aka2:int_array",       "std::vector<int>",            "aka::int_array_leaf"      }},
    {"unsigned int",     {0, 0, osx::import_intrinsic}, {"aka2:uint_array",      "std::vector<unsigned int>",   "aka::uint_array_leaf"     }},
    {"bool",             {0, 0, osx::import_intrinsic}, {"aka2:bool_array",      "std::vector<bool>",           "aka::bool_array_leaf"     }},
    {"float",            {0, 0, osx::import_intrinsic}, {"aka2:float_array",     "std::vector<float>",          "aka::float_array_leaf"    }},
    {"double",           {0, 0, osx::import_intrinsic}, {"aka2:double_array",    "std::vector<double>",         "aka::double_array_leaf"   }},
    {"std:string",       {0, 0, osx::import_intrinsic}, {"aka2:string_array",    "std::vector<std::string>",    "aka::string_array_leaf"   }},
    {"aka2:nill",        {0, 0, osx::import_intrinsic}, {"aka2:nill_array",      "std::vector<aka::nill>",      "aka::nill_array_leaf"     }},
    {0}
  };

  const predefined default_imports[] = {
    /** alias builtin names */
    {"xs:byte",               
     {"char",           "xs::byte_leaf", osx::import_weak_resolve}, 
     {"xs:byte_array", "std::vector<char>", "xs::byte_array_leaf"}
    },
    {"xs:unsignedByte",       
     {"unsigned char",  "xs::unsignedByte_leaf", osx::import_weak_resolve}, 
     {"xs:unsignedByte_array", "std::vector<unsigned char>", "xs::uchar_array"}
    },
    {"xs:short",              
     {"short",          "xs::short_leaf", osx::import_weak_resolve}, 
     {"xs:short_array", "std::vector<short>", "xs::short_array_leaf"}
    },
    {"xs:unsignedShort",      
     {"unsigned short", "xs::unsignedShort_leaf",  osx::import_weak_resolve}, 
     {"xs:unsignedShort_array", "std::vector<unsigned short>", "xs::unsignedShort_array_leaf"},
    },
    {"xs:int",                
     {"long",           "xs::long_leaf",  osx::import_weak_resolve}, 
     {"xs:long_array", "std::vector<long>", "xs::long_array_leaf"}
    },
    {"xs:unsignedInt",        
     {"unsigned long",      "xs::unsignedInt_leaf",  osx::import_weak_resolve}, 
     {"xs:unsignedInt_array", "std::vector<ULONGLONG>", "xs::unsignedInt_array"}
    },
    {"xs:long",               
     {"LONGLONG",       "xs::long_leaf",  osx::import_weak_resolve}, 
     {"xs:long_array", "std::vector<LONGLONG>", "xs::long_array_leaf"}
    },
    {"xs:unsignedLong",       
     {"ULONGLONG",      "xs::unsignedLong_leaf",  osx::import_weak_resolve}, 
     {"xs:unsignedLong_array", "std::vector<ULONGLONG>", "xs::unsignedLong_array_leaf"}
    },
    {"xs:integer",            
     {"long",           "xs::integer_leaf",  osx::import_weak_resolve}, 
     {"xs:integer_array", "std::vector<long>", "xs::integer_array_leaf"}
    },
    {"xs:boolean",            
     {"bool",           "xs::boolean_leaf",  osx::import_weak_resolve}, 
     {"xs:boolean_array", "std::vector<bool>", "xs::boolean_array_leaf"}
    },
    {"xs:float",              
     {"float",          "xs::float_leaf",  osx::import_weak_resolve}, 
     {"xs:float_array", "std::vector<float>", "xs::float_array_leaf"}
    },
    {"xs:double",             
     {"double",         "xs::double_leaf",  osx::import_weak_resolve}, 
     {"xs:double_array", "std::vector<double>", "xs::double_array_leaf"}
    },
    {"xs:string",             
     {"std::string",     "xs::string_leaf",  osx::import_weak_resolve}, 
     {"xs:string_array", "std::vector<bool>", "xs::string_array_leaf"}
    },

    /* osx:type */
    {"xs:normalizedString", 
     {"std::string", "xs::normalizedString_leaf", osx::import_user},
     {"normalizedString_array", "std::vector<std::string>", "xs::normalizedString_array_leaf"},
    },
    {"xs:token", 
     {"std::string", "xs::token_leaf", osx::import_user},
     {"token_array", "std::vector<std::string>", "xs::token_array_leaf"},
    },
    {"xs:base64Binary", 
     {"std::string", "xs::base64Binary_leaf", osx::import_user},
     {"base64Binary_array", "std::vector<std::string>", "xs::base64Binary_array_leaf"},
    },
    {"xs:hexBinary", 
     {"std::string", "xs::hexBinary_leaf", osx::import_user},
     {"hexBinary_array", "std::vector<std::string>", "xs::hexBinary_array_leaf"},
    },
    {"xs:positiveInteger", 
     {"long", "xs::positiveInteger_leaf", osx::import_user},
     {"positiveInteger_array", "std::vector<long>", "xs::positiveInteger_array_leaf"},
    },
    {"xs:negativeInteger", 
     {"long", "xs::negativeInteger_leaf", osx::import_user},
     {"negativeInteger_array", "std::vector<long>", "xs::negativeInteger_array_leaf"},
    },
    {"xs:nonNegativeInteger", 
     {"long", "xs::nonNegativeInteger_leaf", osx::import_user},
     {"nonNegativeInteger_array", "std::vector<long>", "xs::nonNegativeInteger_array_leaf"},
    },
    {"xs:nonPositiveInteger", 
     {"long", "xs::nonPositiveInteger_leaf", osx::import_user},
     {"nonPositiveInteger_array", "std::vector<long>", "xs::nonPositiveInteger_array_leaf"},
    },
    {"xs:decimal", 
     {"long", "xs::decimal_leaf", osx::import_user},
     {"decimal_array", "std::vector<long>", "xs::decimal_array_leaf"},
    },
    {"xs:time", 
     {"std::string", "xs::time_leaf", osx::import_user},
     {"time_array", "std::vector<std::string>", "xs::time_array_leaf"},
    },
    {"xs:dateTime", 
     {"std::string", "xs::dateTime_leaf", osx::import_user},
     {"dateTime_array", "std::vector<std::string>", "xs::dateTime_array_leaf"},
    },
    {"xs:duration", 
     {"std::string", "xs::duration_leaf", osx::import_user},
     {"duration_array", "std::vector<std::string>", "xs::duration_array_leaf"},
    },
    {"xs:date", 
     {"std::string", "xs::date_leaf", osx::import_user},
     {"date_array", "std::vector<std::string>", "xs::date_array_leaf"},
    },
    {"xs:gMonth", 
     {"std::string", "xs::gMonth_leaf", osx::import_user},
     {"gMonth_array", "std::vector<std::string>", "xs::gMonth_array_leaf"},
    },
    {"xs:gYear", 
     {"std::string", "xs::gYear_leaf", osx::import_user},
     {"gYear_array", "std::vector<std::string>", "xs::gYear_array_leaf"},
    },
    {"xs:gYearMonth", 
     {"std::string", "xs::gYearMonth_leaf", osx::import_user},
     {"gYearMonth_array", "std::vector<std::string>", "xs::gYearMonth_array_leaf"},
    },
    {"xs:gDay", 
     {"std::string", "xs::gDay_leaf", osx::import_user},
     {"gDay_array", "std::vector<std::string>", "xs::gDay_array_leaf"},
    },
    {"xs:gMonthDay", 
     {"std::string", "xs::gMonthDay_leaf", osx::import_user},
     {"gMonthDay_array", "std::vector<std::string>", "xs::gMonthDay_array_leaf"},
    },
    {"xs:Name", 
     {"std::string", "xs::Name_leaf", osx::import_user},
     {"Name_array", "std::vector<std::string>", "xs::Name_array_leaf"},
    },
    {"xs:QName", 
     {"aka2:qname", "xs::QName_leaf", osx::import_user},
     {"QName_array", "std::vector<aka:qname>", "xs::QName_array_leaf"},
    },
    {"xs:NCName", 
     {"std::string", "xs::NCName_leaf", osx::import_user},
     {"NCName_array", "std::vector<std::string>", "xs::NCName_array_leaf"},
    },
    {"xs:anyURI", 
     {"std::string", "xs::anyURI_leaf", osx::import_user},
     {"anyURI_array", "std::vector<std::string>", "xs::anyURI_array_leaf"},
    },
    {"xs:language", 
     {"std::string", "xs::language_leaf", osx::import_user},
     {"language_array", "std::vector<std::string>", "xs::language_array_leaf"},
    },
    {"xs:ID", 
     {"std::string", "xs::ID_leaf", osx::import_user},
     {"ID_array", "std::vector<std::string>", "xs::ID_array_leaf"},
    },
    {"xs:IDREF", 
     {"std::string", "xs::IDREF_leaf", osx::import_user},
     {"IDREF_array", "std::vector<std::string>", "xs::IDREF_array_leaf"},
    },
    {"xs:IDREFS", 
     {"std::string", "xs::IDREFS_leaf", osx::import_user},
     {"IDREFS_array", "std::vector<std::string>", "xs::IDREFS_array_leaf"},
    },
    {"xs:ENTITY", 
     {"std::string", "xs::ENTITY_leaf", osx::import_user},
     {"ENTITY_array", "std::vector<std::string>", "xs::ENTITY_array_leaf"},
    },
    {"xs:ENTITIES", 
     {"std::string", "xs::ENTITIES_leaf", osx::import_user},
     {"ENTITIES_array", "std::vector<std::string>", "xs::ENTITIES_array_leaf"},
    },
    {"xs:NOTATION", 
     {"std::string", "xs::NOTATION_leaf", osx::import_user},
     {"NOTATION_array", "std::vector<std::string>", "xs::NOTATION_array_leaf"},
    },
    {"xs:NMTOKEN", 
     {"std::string", "xs::NMTOKEN_leaf", osx::import_user},
     {"NMTOKEN_array", "std::vector<std::string>", "xs::NMTOKEN_array_leaf"},
    },
    {"xs:NMTOKENS", 
     {"std::string", "xs::NMTOKENS_leaf", osx::import_user},
     {"NMTOKENS_array", "std::vector<std::string>", "xs::NMTOKENS_array_leaf"},
    },
    {0}
  };


  const predefined internal_builtins[] = {
    /* aka:any (xs:anyType) is processed in the special way. */
    {"aka2:any",      {0, 0, osx::import_intrinsic}, {"aka2:any_array", "std::vector<aka::any>", 0}},
    /* wildcard appears in the document node.  Array type is not required. */
    {"aka2:wildcard", {0, 0, osx::import_intrinsic}, {0,               0,                       0}},
    {0}
  };

  const predefined internal_imports[] = {
    /** ur-type.  Handled in special way. */ 
    {"aka2:nill",             {"aka2:nill",      0, osx::import_resolve }, {0}},
    {"xs:anyType",            {"aka2:any",       0, osx::import_resolve }, {0}},
    {"xs:anySimpleType",      {"std:string",     0, osx::import_resolve }, {0}},
    {0}
  };

  struct default_substitution {
    const char *original_;
    const char *substituted_;
  };

  const char *cpp_keywords[] = {
    "and", "bool", "compl", "do", "export", "goto", "namespace", "or_eq",
    "return", "struct", "try", "using", "xor", "and_eq", "break", "const",
    "double", "extern", "if", "new", "private", "short", "switch", "typedef", 
    "virtual", "xor_eq", "asm", "case", "const_cast", "dynamic_cast", "false",
    "inline", "not", "protected", "signed", "template", "typeid" ,"void",
    "auto", "catch", "continue", "else", "float", "int", "not_eq", "public", 
    "sizeof", "this", "typename", "volatile", "bitand", "char", "default",
    "enum", "for", "long", "operator", "register", "static", "throw", "union",
    "wchar_t", "bitor", "class", "delete", "explicit", "friend", "mutable",
    "or", "reinterpret_cast", "static_cast", "true", "unsigned", "while", 0
  };
  
  struct default_escape {
    const char *to_escape_;
    const char *escaped_; 
  };

  const default_escape default_escapes[] = {
    {"-", "_"},
    {".", "_"},
    {0}
  };

  bool builtin_exists(const osx::type_array &types, const std::string &name) {
    for (osx::type_array::const_iterator it = types.begin();
	 it != types.end(); ++it) {
      if (it->name_ == name)
	return true;
    }
    return false;
  }

  bool import_exists(const osx::imports_choice &types, const std::string &name) {
    for (osx::imports_choice::const_iterator it = types.begin();
	 it != types.end(); ++it) {
      if (aka::item_of(*it, name))
	return true;
    }
    return false;
  }

  bool substitution_exists(const osx::substitution_array &subs, const std::string &original) {
    for (osx::substitution_array::const_iterator it = subs.begin();
	 it != subs.end(); ++it) {
      if (it->original_ == original)
	return true;
    }
    return false;
  }

  bool escape_exists(const osx::escape_array &ess, const std::string &to_escape) {
    for (osx::escape_array::const_iterator it = ess.begin();
	 it != ess.end(); ++it) {
      if (it->to_escape_ == to_escape)
	return true;
    }
    return false;
  }
  
  osx::type construct_type(const predefined *predef, bool has_value_type) {
    
    osx::type type;
    
    type.name_ = predef->name_;
    if (predef->type_.cpp_ != 0) {
      if (has_value_type)
	type.primitive_.value_type_.reset(new std::string(predef->type_.cpp_));
      else
	type.primitive_.typedef_of_.reset(new std::string(predef->type_.cpp_));
    }
    if (predef->type_.leaf_ != 0)
      type.primitive_.leaf_.reset(new std::string(predef->type_.leaf_));
    
    if (predef->array_.cpp_ != 0) {
      osx::builtin_array array;
      array.name_ = predef->array_.cpp_;
      if (predef->array_.leaf_ != 0)
	array.primitive_.leaf_.reset(new std::string(predef->array_.leaf_));
      if (predef->array_.typedef_ != 0)
	array.primitive_.typedef_of_.reset(new std::string(predef->array_.typedef_));
      type.array_.reset(new osx::builtin_array(array));
    }
    
    return type;
  }
  
  bool namespace_exists(const osx::directives_choice &dirs, const std::string &prefix) {
    for (osx::directives_choice::const_iterator it = dirs.begin();
	 it != dirs.end(); ++it) {
      if (!aka::item_of(*it, prefix))
	continue;
      const osx::_namespace &ns = aka::item_cast<osx::_namespace>(*it);
      if (ns.prefix_ == prefix)
	return true;
    }
    return false;
  }
  
}

using namespace osx;

void osixaka2::create_default_preference() {

  pref_.reset(new preference());
  aka::construct_element(*pref_, osx::preference_leaf());

  // member
  pref_->format_.member_.prefix_ = "";
  pref_->format_.member_.suffix_ = "_";

  // array
  pref_->format_.array_.prefix_ = "";
  pref_->format_.array_.suffix_ = "_array";
  pref_->format_.array_.container_ = "std::list";

  // choice.
  pref_->format_.choice_.container_ = "std::list";
  // simplecontent
  pref_->format_.simplecontent_.valuename_ = "value";

  // particle array
  pref_->format_.particle_item_.prefix_ = "";
  pref_->format_.particle_item_.suffix_ = "_item";

  merge_defaults();

}

void osixaka2::load_defaults() {
  create_default_preference();
}

void osixaka2::load_osixaka1_defaults() {
  create_default_preference();
  pref_->format_.choice_.container_ = "std::vector";
  pref_->format_.array_.container_ = "std::vector";
  pref_->format_.array_.prefix_ = "";
  pref_->format_.array_.suffix_ = "s_type";
}

void osixaka2::use_MFC_member_style() {
  assert(pref_.get() != 0);
  pref_->format_.member_.prefix_ = "m_";
  pref_->format_.member_.suffix_ = "";
}

void osixaka2::generate_template(std::ostream &file) {
  osx::preference pref = *pref_;
  pref.escapes_.escape_.clear();
  pref.substitutions_.substitution_.clear();
  pref.directives_.c0_.clear();
  pref.internal_.reset(0);

  osx::escape es;
  es.to_escape_ = "character_to_escape";
  es.escaped_ = "escaped_character";
  pref.escapes_.escape_.push_back(es);

  osx::substitution sub;
  sub.original_ = "token_to_substitute";
  sub.substituted_ = "substituted_token";
  pref.substitutions_.substitution_.push_back(sub);

  aka::xml_serializer ser;
  ser.default_ns_prefix("osx");
  ser.serialize(pref, "osx:preference", file);
}

void osixaka2::load_preference(const std::string &filename) {
  aka::document doc = aka::deserialize_file(filename);
  pref_.reset(aka::adopt_root<osx::preference>(doc));
  merge_defaults();
}

void osixaka2::save_preference(std::ostream &file) {
  aka::xml_serializer ser;
  ser.default_ns_prefix("osx");
  ser.serialize(*pref_, "osx:preference", file);
}



namespace {

  void setup_preferenced_type(osx::internal &internal, 
			      const predefined *builtins, const predefined *imports) {

    // setup builtin types.

    const predefined *predef;

    for (predef = builtins; predef->name_ != 0; ++predef) {
      if (builtin_exists(internal.builtins_.type_, predef->name_))
	continue;
      osx::type type = construct_type(predef, false);
      internal.builtins_.type_.push_back(type);
    }

    osx::imports_choice_leaf::binder binder(internal.imports_.c0_);
    for (predef = imports; predef->name_ != 0; ++predef) {
      if (import_exists(internal.imports_.c0_, predef->name_))
	continue;

      switch (predef->type_.id_) {
      case import_weak_resolve: {
	osx::type type = construct_type(predef, true);
	binder.push_back(type, "osx:type");
	break;
      }
      case import_user: {
	osx::type type = construct_type(predef, false);
	binder.push_back(type, "osx:type");
	break;
      }
      case import_resolve: {
	osx::resolve resolve;
	resolve.name_ = predef->name_;
	resolve.value_ = predef->type_.cpp_;
	binder.push_back(resolve, "osx:resolve");
	break;
      }
      case import_array:
      case import_na:
      case import_intrinsic:
	assert(!"Must not reach here.");
      }
    }
  }

}


void osixaka2::merge_defaults() {

  // add internal type.
  if (pref_->internal_.get() == 0) {
    pref_->internal_.reset(new osx::internal);
  }


  osx::directives_choice_leaf::binder dirbind(pref_->directives_.c0_);

  osx::_namespace ns;
  if (!namespace_exists(pref_->directives_.c0_, "aka2")) {
    ns.prefix_ = "aka2";
    ns.uri_ = aka::get_namespace_uri("aka2");
    dirbind.push_back(ns, "osx:namespace");
  }
  if (!namespace_exists(pref_->directives_.c0_, "std")) {
    ns.prefix_ = "std";
    ns.uri_ = aka::get_namespace_uri("std");
    dirbind.push_back(ns, "osx:namespace");
  }
  if (!namespace_exists(pref_->directives_.c0_, "osx")) {
    ns.prefix_ = "osx";
    ns.uri_ = aka::get_namespace_uri("osx");
    dirbind.push_back(ns, "osx:namespace");
  }

  osx::include include;
  include.system_ = true;
  include.value_ = "akaxiso2/builtin/builtin.h";
  dirbind.push_back(include, "osx:include");

  include.value_ = "akaxiso2/builtin/schema_builtin.h";
  dirbind.push_back(include, "osx:include");

  include.value_ = "akaxiso2/content_model.h";
  dirbind.push_back(include, "osx:xiso_include");

  include.value_ = "akaxiso2/akaxiso2.h";
  dirbind.push_back(include, "osx:ximpl_include");


  osx::internal &internal = *pref_->internal_;


  osx::substitutions &sub_elms = pref_->substitutions_;
  for (const char **sub = cpp_keywords; *sub != 0; ++sub) {
    if (substitution_exists(pref_->substitutions_.substitution_, *sub))
      continue;
    osx::substitution sub_elm;
    sub_elm.original_ = *sub;
    sub_elm.substituted_ = std::string("_") + *sub;
    sub_elms.substitution_.push_back(sub_elm);
  }

  osx::escapes &escapes = pref_->escapes_;
  for (const default_escape *es = default_escapes; es->to_escape_ != 0; ++es) {
    if (escape_exists(pref_->escapes_.escape_, es->to_escape_))
      continue;
    osx::escape es_elm;
    es_elm.to_escape_ = es->to_escape_;
    es_elm.escaped_ = es->escaped_;
    escapes.escape_.push_back(es_elm);
  }

  setup_preferenced_type(internal, default_builtins, default_imports);
}

void osixaka2::commit_preference() {
  assert(pref_.get() != 0);

  osx::internal &internal = *pref_->internal_;

  const predefined *predef;
  
  for (predef = internal_builtins; predef->name_ != 0; ++predef) {
    if (builtin_exists(internal.builtins_.type_, predef->name_)) {
      std::string message = std::string("builtin type, ") + aka::quote(predef->name_) 
	+ ", should not be in preference.";
      throw aka::error(message, __FILE__, __LINE__);
    }
  }
  
  for (predef = internal_imports; predef->name_ != 0; ++predef) {
    if (import_exists(internal.imports_.c0_, predef->name_)) {
      std::string message = std::string("imported type, ") + aka::quote(predef->name_) 
	+ ", should not be in preference.";
      throw aka::error(message, __FILE__, __LINE__);
    }
  }
  setup_preferenced_type(internal, internal_builtins, internal_imports);
  registry_.set_preference(*pref_);
}
