#include "preprocessor.h"
#include "xschemadeserializer.h"
#include <akaxiso2/util/filepath.h>


using namespace osx;

xs::schema* preprocessor::deserialize(const std::string &filename) {

  xschemadeserializer deserializer(ostm_, verbose_);
  xs::schema *target_schema = 
    deserializer.deserialize(aka::replace_path_seperator(filename));
  
  if (target_schema == 0)
    throw aka::error(aka::quote(filename) + " : Failed to parse.", 
		     __FILE__, __LINE__);
  return target_schema;
}

void preprocessor::preprocess(const std::string &filepath) {
  if (preprocess_recursive(filepath))
    registry_.add_unit(unit_);
}

bool preprocessor::preprocess_recursive(const std::string &filepath) {

  std::string simplified_path = aka::simplify_path(filepath);
  if (is_processed(simplified_path))
    return false;

  fns_.insert(simplified_path);

  basepath_ = aka::get_dirpath(simplified_path);
  xs::schema *schema = deserialize(simplified_path);

  /** targetNamespace */
  target_ns_id_ = aka::get_namespace_id(schema->targetNamespace_);

  std::string basename = aka::get_basename(filepath);
  unit_ = registry_unit(schema, target_ns_id_, basename, simplified_path);

  const xs::directives &dirs = schema->directives_;
  for (xs::directives::const_iterator it = dirs.begin();
       it != dirs.end(); ++it) {
    const aka::item &item = *it;

    if (aka::item_of(item, "xs:include")) {
      const xs::include &inc = aka::item_cast<const xs::include>(item);
      process_include(inc);
    }
    else if (aka::item_of(item, "xs:import")) {
      process_import(aka::item_cast<xs::import>(item));
    }
    else if (aka::item_of(item, "xs:redefine")) {
      process_redefine(aka::item_cast<xs::redefine>(item));
    }
    else if (aka::item_of(item, "xs:annotation")) {
      process_annotation(aka::item_cast<xs::annotation>(item));
    }
  }
  // finalDefault, blockDefault, id, xml:lang, version is not processed. !!!!!

  return true;
}

void preprocessor::process_include(const xs::include &include) {

  if (include.schemaLocation_.empty()) {
    ostm_ << "xs:include@schemaLocation is empty.  Ignore xs:include." << std::endl;
    return;
  }

  preprocessor pre(registry_, 
		   fns_,
		   ostm_,
		   verbose_);

  if (pre.preprocess_recursive(basepath_ + include.schemaLocation_)) {
    if (pre.get_target_ns_id() != target_ns_id_)
      throw aka::error(include.schemaLocation_ + ": Included document has different namespace.",
		       __FILE__, __LINE__); // The included NS differs from the target one.
    
    // merge include and other preference. !!!!!!!!
    merge_schema(*unit_.schema_, *pre.get_unit().schema_);
  }
}

void preprocessor::process_import(const xs::import &import) {

  if (import.schemaLocation_.empty()) {
    ostm_ << "xs:import@schemaLocation is empty.  Ignore xs:import." << std::endl;
    return;
  }

  preprocessor pre(registry_, 
		   fns_,
		   ostm_, 
		   verbose_);
  if (pre.preprocess_recursive(basepath_ + import.schemaLocation_)) {
    if (pre.get_target_ns_id() == target_ns_id_) {
      // Importing NS should be the same as the target schema doc. 
      throw aka::error(aka::quote(import.schemaLocation_) + 
		       " : namespace of imported document should not be the same.", 
		       __FILE__, __LINE__); 
    }
    registry_.add_unit(pre.get_unit());
  }
}

void preprocessor::process_redefine(const xs::redefine &redefine) {
  throw not_supported("xs:redefine is not supported.", __FILE__, __LINE__);
}


void preprocessor::process_annotation(const xs::annotation &ann) {
  
  for (xs::annotation_choice::const_iterator it = ann.c0_.begin();
       it != ann.c0_.end(); ++it) {
    if (!aka::item_of(*it, "xs:appinfo"))
      continue;
    
    const xs::appinfo &ainfo = aka::item_cast<xs::appinfo>(*it);
    for (aka::any_array::const_iterator ait = ainfo.any_.begin();
	 ait != ainfo.any_.end(); ++ait) {
      process_osx_element(*ait);
    }
  }
}

void preprocessor::process_osx_element(const aka::any &any) {
  
  if (!any.has_document())
    return;

  const aka::document &doc = any.document_;
  
  if (aka::document_of(doc, "osx:include")) {
    const osx::include *root = aka::root_cast<osx::include>(doc);
    unit_.includes_.push_back(*root);
  }
  else if (aka::document_of(doc, "osx:xiso_include")) {
    const osx::include *root = aka::root_cast<osx::include>(doc);
    unit_.xiso_includes_.push_back(*root);
  }
  else if (aka::document_of(doc, "osx:ximpl_include")) {
    const osx::include *root = aka::root_cast<osx::include>(doc);
    unit_.ximpl_includes_.push_back(*root);
  }
  else if (aka::document_of(doc, "osx:namespace")) {
    const osx::_namespace *root = aka::root_cast<osx::_namespace>(doc);
    osx::_namespace ns;
    ns = *root;
    if (ns.uri_.empty())
      ns.uri_ = aka::get_namespace_uri(target_ns_id_);
    unit_.namespaces_.push_back(ns);
  }
  else {
    ostm_ << "Document name " << aka::document_name(doc) << std::endl;
    assert(!"Mut not reach here.");
  }
}

void preprocessor::merge_schema(xs::schema &schema, const xs::schema &to_merge) {
  schema.schemaTops_.insert(schema.schemaTops_.end(),
			    to_merge.schemaTops_.begin(),
			    to_merge.schemaTops_.end());
}


bool preprocessor::is_processed(const std::string &filename) const {
  units uts = registry_.get_units();
  for (units::const_iterator it = uts.begin(); it != uts.end(); ++it)
    if ((*it)->filename_ == filename)
      return true;
  return filenames::const_iterator(fns_.find(filename)) != fns_.end();
}

const registry_unit &preprocessor::get_unit() const {
  return unit_;
}
