#include "schema_generator.h"

#include <akaxiso2/akaxiso2.h>
#include <akaxiso2/util/string_funcs.h>
#include <akaxiso2/framework/document_factory.h>
#include <sstream>

using namespace aka2;
using namespace osx;

void osx::generate_schema(const std::string &docname, std::ostream &ostm) {
  schema_generator gen;
  gen.add_docname(docname);
  gen.generate_schema(ostm);
}

schema_generator::schema_generator() : serializer(2, "", aka::default_transcoder::create) {
}

schema_generator::schema_generator(const std::string &encoding) 
  : serializer(2, encoding, aka::default_transcoder::create) {
}

schema_generator::~schema_generator() {
}

void schema_generator::generate_schema(std::ostream &ostm) {
  if (docnames_.empty()) {
    throw error("Nothing to generate.", __FILE__, __LINE__);
  }

  target_ns_ = docnames_.front().get_namespace_id();
  qnames::const_iterator it;
  for (it = docnames_.begin(); it != docnames_.end(); ++it) {
    if (target_ns_ != it->get_namespace_id()) {
      throw error("documents belong to some namespaces.", __FILE__, __LINE__);
    }
  }

  prepare(ostm);
  write_schema();
  finish();
}

void schema_generator::write_schema() {

  /* Write schema header */
  std::ostringstream ostm;
  ostm << "<?xml version=\"1.0\" encoding=" 
       << quote(get_default_encoding()) << "?>" << std::endl
       << std::endl
       << "<xs:schema xmlns:xs=\"http://www.w3.org/2001/XMLSchema\"" << std::endl;
  if (target_ns_ != empty_token) {
    ostm << " targetNamespace=" << quote(get_namespace_uri(target_ns_));
  }
  ostm << ">" << std::endl;

  formatter_.write_uni(to_pivot(ostm.str()));

  inc_indent_level();

  qnames::const_iterator it;
  for (it = docnames_.begin(); 
       it != docnames_.end(); ++it) {
    const element_props &props = system_document_factory().get_props(*it);
    write_element_declaration(*it, props);
  }

  for (it = docnames_.begin(); it != docnames_.end(); ++it) {
    const element_props &props = system_document_factory().get_props(*it);
    write_definition(props.op());
  }

  bool finished;
  do {
    finished = true;
    for (opset::const_iterator it = to_generate_.begin();
	 it != to_generate_.end(); ++it) {
      if (generated_.count(*it) == 0) {
	finished = false;
	write_definition(**it);
	generated_.insert(*it);
      }
    }
  } while (!finished);

  dec_indent_level();
  new_line();
  write("</xs:schema>\n");
}

void schema_generator::add_docname(const std::string &docname) {
  docnames_.push_back(qname(docname));
}

void schema_generator::write_element_declaration(const aka::qname &docname, 
						 const element_props &props) {
  std::ostringstream ostm;
  ostm << "<xs:element name=" << quote(docname.local()) 
       << " type=" << quote(props.op().get_typename()) << "/>" << std::endl
       << std::endl;
}


void schema_generator::write_definition(const element_op &op) {

  switch (op.get_schematype()) {
  case simpletype_id:
    write_simpleType_definition(op);
    break;
  case sequence_id:
  case choice_id:
  case all_id:
  case simplecontent_id:
    write_complexType_definition(op);
    break;
  case array_id: {
    const array_op &aop = static_cast<const array_op&>(op);
    write_definition(aop.get_item_op());
    break;
  }
  case ptrmember_id: {
    const ptrmember_op &pop = static_cast<const ptrmember_op&>(op);
    write_definition(pop.get_value_op());
    break;
  }
  case fixed_id: {
    const fixed_op &fop = static_cast<const fixed_op&>(op);
    write_definition(fop.get_value_op());
    break;
  }
  case any_id:
  case any_array_id:
  case wildcard_id:
    break;
  case enclose_id:
  case disclose_id:
    assert(!"Not implemented.");
  case wildcard_attribute_id:
    assert(!"Must not reach here.");
  }
}


void schema_generator::write_simpleType_definition(const element_op &op) {

  assert(op.get_schematype() == simpletype_id);
  new_line();
  write("<xs:simpleType name=" + quote(op.get_typename()) + ">");
  inc_indent_level();
  new_line();
  write("<xs:restriction base=" + quote("xs:string") + "/>");
  dec_indent_level();
  new_line();
  write("</xs:simpleType>\n");
}


void schema_generator::write_complexType_definition(const element_op &op) {

  new_line();
  write("<xs:complexType name=" + quote(op.get_typename()) + ">");

  inc_indent_level();

  switch (op.get_schematype()) {
  case sequence_id:
    write_sequence(op);
    break;
  case choice_id:
    write_choice(op);
    break;
  case all_id:
    write_all(op);
    break;
  case simplecontent_id:
    write_simpleContent(op);
    break;
  case enclose_id:
  case disclose_id:
    assert(!"Not implemented.");
  case array_id:
  case ptrmember_id :
  case any_id:
  case any_array_id:
  case wildcard_id:
  case fixed_id:
  case simpletype_id:
  case wildcard_attribute_id:
    assert(!"Must not reach here.");
  }

  dec_indent_level();
  new_line();
  write("</xs:complexType>\n");

}

void schema_generator::write_simpleContent(const element_op &props) {
  
}

void schema_generator::write_sequence(const element_op &op) {

  const sequence_op &sop = static_cast<const sequence_op&>(op);
  const member_types &mtypes = sop.get_member_types();
  
  if (mtypes.empty()) {
    new_line();
    write("<xs:sequence/>");
  }
  else {
    new_line();
    write("<xs:sequence>");
    inc_indent_level();
    
    for (member_types::const_iterator it = mtypes.begin(); it != mtypes.end(); ++it) {
      new_line();
      write_child(*it);
      to_generate_.insert(&it->op());
    }

    dec_indent_level();
    new_line();
    write("</xs:sequence>");
  }
}

void schema_generator::write_choice(const element_op &op) {

  const choice_op &cop = static_cast<const choice_op&>(op);
  const item_types &itypes = cop.get_item_types();
  
  if (itypes.empty()) {
    new_line();
    write("<xs:choice/>");
  }
  else {
    new_line();
    write("<xs:choice>");
    inc_indent_level();
    
    for (item_types::const_iterator it = itypes.begin(); it != itypes.end(); ++it) {
      new_line();
      write_child(*it);
      to_generate_.insert(&it->op());
    }

    dec_indent_level();
    new_line();
    write("</xs:choice>");
  }
}

void schema_generator::write_all(const element_op &op) {

  const sequence_op &aop = static_cast<const all_op&>(op);
  const member_types &mtypes = aop.get_member_types();
  
  if (mtypes.empty()) {
    new_line();
    write("<xs:all/>");
  }
  else {
    new_line();
    write("<xs:sequence>");
    inc_indent_level();
    
    for (member_types::const_iterator it = mtypes.begin(); it != mtypes.end(); ++it) {
      new_line();
      write_child(*it);
      to_generate_.insert(&it->op());
    }

    dec_indent_level();
    new_line();
    write("</xs:sequence>");
  }
}

void schema_generator::write_child(const element_props &props) {

  const occurrence &occ = props.get_occurrence();
  if (props.is_element()) {
    write("<xs:element name=" + quote(props.get_name()) 
	  + " type=" + quote(props.op().get_typename()));
    if (occ.is_array())
      write_occurrence(occ);
    write("/>");
  }
  else {
    write("<xs:group ref=" + quote(props.op().get_typename()));
    if (occ.is_array())
      write_occurrence(occ);
    write("/>");
  }
}

void schema_generator::write_occurrence(const occurrence &occ) {
  assert(occ.is_array());
  std::ostringstream ostm;
  ostm << " minOccurs=\"" << occ.minOccurs_ << "\"";
  if (occ.maxOccurs_ == aka::unbounded)
    ostm << " maxOccurs=\"unbounded\"";
  else
    ostm << " maxOccurs=\"" << occ.maxOccurs_ << "\"";
  write(ostm.str());
}


void schema_generator::write(const std::string &text) {
  formatter_.write_uni(aka::to_pivot(text));
}
