/* -*- c++ -*- */
#ifndef OSIXAKA2_USERDEFS_H__
#define OSIXAKA2_USERDEFS_H__

#include <akaxiso2/configuration.h>
#include <osixaka2/simple_map.h>
#include <osixaka2/schema_st.h>
#include <set>

namespace osx {

  inline void occurrence_valid(const aka::occurrence &occ) {
    if (occ.maxOccurs_ == aka::unbounded)
      return;
    if (occ.minOccurs_ > occ.maxOccurs_) {
      throw aka::error("wrong occurrence found", __FILE__, __LINE__);
    }
  }


  typedef std::set<aka::qname, aka::qname_less> qname_set;
  
  class imported_type;
  class user_class;
  typedef std::vector<const user_class*> const_user_classes;
  
  struct type_ref {
    type_ref() : imported_(0), us_(0), is_local_(false), is_element_(false) { }
    explicit type_ref(const imported_type *imported) 
      : imported_(imported), us_(0), is_local_(false), is_element_(false) { 
      assert(imported != 0);
    }
    explicit type_ref(const user_class *us)
      : imported_(0), us_(us), is_local_(false), is_element_(false) { 
      assert(us != 0);
    }
    explicit type_ref(const aka::qname &name, bool is_local, bool is_element)
      : imported_(0), us_(0), referred_name_(name), 
	is_local_(is_local), is_element_(is_element) { 
      assert(!name.empty());
    }
    
    const imported_type *imported_;
    const user_class *us_;
    // reference.
    aka::qname referred_name_;
    bool is_local_;
    bool is_element_;
    
    bool empty() const { return (imported_ == 0) && (us_ == 0) && referred_name_.empty(); }
    bool is_imported() const { return imported_ != 0; }
    bool is_reference() const { return !referred_name_.empty(); }
    const aka::qname &get_name() const;
    const aka::qname &get_classname() const;
  };
  
  
  
  /**
   * classes for user class definition.
   */

  /* element */
  class element_type {
    aka::occurrence occ_;
  public:
    element_type() 
      : is_fixed_(false), is_group_(false), is_ptrmember_(false), occ_required_(false) {}
    aka::qname name_; 
    aka::qname member_name_;
    std::string default_;
    xs::namespaceList namespace_list_;
    type_ref type_;

    bool is_fixed_;
    bool is_group_;
    bool is_ptrmember_;
    bool occ_required_;
    bool empty() const { return name_.empty() && type_.empty(); }

    void set_occurrence(const aka::occurrence &occ) {
      occ_ = occ;
      occurrence_valid(occ_);
    }
    const aka::occurrence &get_occurrence() const {
      return occ_;
    }
  };

  typedef std::list<element_type> element_types;

  /* attribute */
  struct attribute_type {
    attribute_type() : required_(false), is_fixed_(false), is_ptrmember_(false) {}
    aka::qname name_;
    type_ref type_;
    bool required_;
    bool is_fixed_;
    std::string default_;
    bool is_ptrmember_;
  };

  typedef std::list<attribute_type> attribute_types;

  struct attribute_defs {
    attribute_defs() : has_anyAttribute_(false) {}
    
    attribute_types attributes_;
    bool has_anyAttribute_;
    xs::namespaceList namespace_list_;

    attribute_type *get(const aka::qname &name);
    void erase(const aka::qname &name);

    bool has_attributes() const {
      return !attributes_.empty() || has_anyAttribute_;
    }
  };



  enum def_type {
    simpletype_id = 1,
    simplecontent_id,
    array_id,
    sequence_id,
    choice_id,
    all_id,
    //   enclose_id,
    //   disclose_id,
    //   any_id,
    //   any_array_id,
    //   any_attribute_id,
    //   fixed_id
  };


  class user_classes {
    typedef std::map<aka::qname, user_class*, aka::qname_less> classmap;
    classmap map_;
  public:
    ~user_classes();

    typedef classmap::iterator iterator;
    typedef classmap::const_iterator const_iterator;
    iterator begin() { return map_.begin(); }
    iterator end() { return map_.end(); }
    
    void insert(const aka::qname &name, user_class *us) {
      std::pair<iterator, bool> itres = map_.insert(classmap::value_type(name, us));
      if (!itres.second) {
	raise_name_error("type", name, "already registered.",
			 __FILE__, __LINE__);
      }
    }

    user_class *get(const aka::qname &name) {
      iterator it = map_.find(name);
      if (it != map_.end())
	return it->second;
      return 0;
    }

    const user_class *get(const aka::qname &name) const {
      const_iterator it = map_.find(name);
      if (it != map_.end())
	return it->second;
      return 0;
    }

    void erase(const aka::qname &name) {
      const_iterator it = map_.find(name);
      assert(it != map_.end());
      map_.erase(it->first);
    }

  };

  class user_class {
    type_ref value_type_;
    aka::qname leaf_type_;
    aka::qname name_;
    aka::qname classname_;
    user_class(const aka::qname &name, const aka::qname &classname,
	       const aka::occurrence &occ, def_type id)
      : name_(name), classname_(classname), occ_(occ), id_(id) {
      assert(!name.empty());
      occurrence_valid(occ_);
    }
    user_class *derive(const aka::qname &name, const aka::qname &classname) const;
    aka::occurrence occ_;
  public:
    void set_occurrence(const aka::occurrence &occ) {
      occ_ = occ;
      occurrence_valid(occ_);
    }
    const aka::occurrence &get_occurrence() const {
      return occ_;
    }


    def_type id_;
    element_types etypes_;
    attribute_defs adefs_;

    void set_value_type(const type_ref &value_type) { 
      value_type_ = value_type; 
      assert(!value_type_.empty());
    }
    const type_ref &get_value_type() const { 
      assert(!value_type_.empty());
      return value_type_; 
    }
    const aka::qname &get_name() const { 
      assert(!name_.empty());
      return name_; 
    }

    const aka::qname &get_classname() const;

    void set_leaf_type(const aka::qname &leaf_type) {
      leaf_type_ = leaf_type;
    }
    const aka::qname &get_leaf_type() const {
      return leaf_type_;
    }

    void check_dup_elements() const;
    void check_dup_attributes() const;

    static user_class *user_sequence(const aka::qname &name, const aka::qname &classname,
				     user_classes &uss) {
      user_class *us = new user_class(name, classname, 
				      aka::occurrence(), sequence_id);
      uss.insert(name, us);
      return us;
    }

    static user_class *user_all(const aka::qname &name, const aka::qname &classname,
				user_classes &uss) {
      user_class *us = new user_class(name, classname,
				      aka::occurrence(), all_id);
      uss.insert(name, us);
      return us;
    }

    static user_class *user_choice(const aka::qname &name, const aka::qname &classname, 
				   const aka::occurrence &occ, user_classes &uss) {
      user_class *us = new user_class(name, classname,
				      occ, choice_id);
      uss.insert(name, us);
      return us;
    }

    static user_class *user_simplecontent(const aka::qname &name, 
					  const aka::qname &classname,
					  const type_ref &value_type,
					  user_classes &uss) {
      user_class *us = new user_class(name, classname,
				      aka::occurrence(), simplecontent_id);
      us->set_value_type(value_type);
      uss.insert(name, us);
      return us;
    }

    static user_class *user_simpletype(const aka::qname &name, 
				       const aka::qname &classname,
				       const type_ref &type,
				       user_classes &uss) { 
      user_class *us = new user_class(name, classname,
				      aka::occurrence(), simpletype_id);
      us->set_value_type(type);
      uss.insert(name, us);
      return us;
    }

    static user_class *user_array(const aka::qname &name, 
				  const aka::qname &classname,
				  const aka::occurrence &occ,
				  const type_ref &type,
				  user_classes &uss) {
      user_class *us = new user_class(name, classname,
				      occ, array_id);
      us->set_value_type(type);
      uss.insert(name, us);
      return us;
    }

    static user_class *derive(const user_class *us,
			      const aka::qname &name, 
			      const aka::qname &classname,
			      user_classes &uss) {
      user_class *derived = us->derive(name, classname);
      uss.insert(name, derived);
      return derived;
    }
  };


  /**
   * classes below have name-declarations, don't have real entity.
   */
  
  /* element */
  class element_def {
  public:
    element_def() 
      : is_fixed_(false), abstract_(false) {}
    aka::qname name_;
    type_ref type_;
    std::string default_;
    bool is_fixed_;
    bool abstract_;
  };
  
  
  /* global attribute */
  struct gattribute_def {
    gattribute_def() : is_fixed_(false) {}
    aka::qname name_;
    type_ref type_;
    std::string default_;
    bool is_fixed_;
  };
  
  class attributeGroup_def : public attribute_defs {
  public:
    aka::qname name_;
  };
  
  typedef simple_map_T<element_def> element_defs;
  typedef simple_map_T<attributeGroup_def> attrGroup_defs;
  typedef simple_map_T<gattribute_def> gattribute_defs;

}

#endif /* CLASSDEFS_H__ */
