#include <osixaka2/osixaka2.h>
#include <iostream>
#include <stdlib.h> /* {BCB6}:exit */


/** Must be in anonymous namespace to prevent name conflict. */
namespace {
  void usage(const std::string &progname) {
    std::cerr << "usage: " << progname << " [option(s)] file1 file2 ... filen" << std::endl
	      << "options" << std::endl
	      << " -n : show document name." << std::endl
	      << " -q : quiet mode (Don't show error messages.)" << std::endl
	      << " -s : do serialization for deserialized document." << std::endl
	      << " -h : show this message." << std::endl;
  }
}



/**
 * public functions
 */

#if defined(__unix__) || defined(__linux__) || defined(__APPLE_CC__)
# include <unistd.h>
# include <libgen.h>
#else
  extern "C" int getopt (int argc, char *const *argv, const char *shortopts);
  extern "C" int optind;
  extern "C" char *optarg;
#endif

/* instantiate_xiso() must be implemented in XISO objects. */
void instantiate_xiso();

int main(int argc, char* argv[]){
  
  bool do_serialize = false;
  bool quiet = false;
  bool show_document_name = false;
  int res = 0;
  std::string progname = basename(argv[0]);

  // initialize akaxiso.
  aka::initialize();
  // initialize XML typeinfo classes.
  instantiate_xiso();

  int c;
  while ((c = getopt(argc, argv, "nqsh")) != -1) {
    switch (c) {
    case 'n':
      show_document_name = true;
      break;
    case 'q':
      quiet = true;
      break;
    case 's':
      do_serialize = true;
      break;
    case 'h':
      usage(progname);
      exit(0);
    case '?':
    default:
      usage(progname);
      exit(1);
    }
  }

  for (int index = optind; index < argc; ++index) {
    aka::document doc;
    std::string errstr;
    std::string filename = argv[index];

    try {
      doc = aka::deserialize_file(filename);
    }
    catch (const aka::error &e) {
      errstr = e.what();
      errstr += 
	"\n"
	"Parse error detected.  Abort immediately.\n";
    }
    catch ( ... ) {
      errstr = "Unhandled parse error detected.  Abort immediatedly.\n";
    }


    if (errstr.empty()) {

      if (show_document_name) 
	std::cerr << filename << " : " << aka::document_name(doc) << std::endl;
      
      if (do_serialize) {
	aka::xml_serializer ser;
	ser.serialize(doc, std::cout);
      }
    }
    else { // Parse error occured.
      res = 1; // returns error.
      if (!quiet)
	std::cerr << errstr;
    }
  }
  
  // Uninitializers.
  aka::uninitialize();

  return res;
}
