/* -*- c++ -*- */
#include "formatter.h"
#include "../../exception.h"
#include "../../framework/qname.h"
#include "../../unicode/utf8_traits.h"
#include "../../transcoders/encoding_name.h"
#include <iostream>

using namespace aka2;

namespace {
  struct escape_entry {
    pchar_t char_;
    pchar_t escaped_[16];
    size_t size_;
  };

  const escape_entry escape_table[] = {
    {'"', {'&', 'q', 'u', 'o', 't', ';'}, 6},
    {'&', {'&', 'a', 'm', 'p', ';',    }, 5},
    {'<', {'&', 'l', 't', ';',         }, 4},
    {0,   {'0',                        }, 0}
  };
  
} // namespace


#ifdef _WIN32

#include <stdio.h>
#include <io.h>
#include <fcntl.h> 
#include <iostream>

#ifdef __BORLANDC__
# define _setmode setmode
#endif


void formatter::set_binary_mode_for_stdio() {
  if (ostm_ == static_cast<std::ostream*>(&std::cout))
    fd_ = _fileno(stdout);
  else if (ostm_ == static_cast<std::ostream*>(&std::cerr))
    fd_= _fileno(stderr);
  else if (ostm_ == static_cast<std::ostream*>(&std::clog))
    fd_ = _fileno(stderr);
  if (fd_ != -1) {
    *ostm_ << std::flush;
    saved_stream_mode_ = _setmode(fd_, _O_BINARY);
  }
}

void formatter::revert_stream_mode() {
  if (ostm_ == 0)
    return;

  if (fd_ != -1) {
    *ostm_ << std::flush;
    _setmode(fd_, saved_stream_mode_);
  }
}

#else

void formatter::set_binary_mode_for_stdio() {}
void formatter::revert_stream_mode() {}

#endif


formatter::formatter(transcoder_factory create) 
  : create_(create), ostm_(0), fd_(-1), to_out_(0){
}


formatter::~formatter() {
  assert(to_out_ == 0);
}


void formatter::prepare(std::ostream &ostm, const std::string &encoding) {
  ostm_ = &ostm;
  if (is_binary_encoding(encoding))
    set_binary_mode_for_stdio();

  assert(to_out_ == 0);
  to_out_ = stateful_transcoder::create(encoding, pchar_traits::encoding(), true, create_);
  /**  Don't have to reset explicitly because to_out_ have been created just now.
   *  to_out_->reset(true);
   */
}

void formatter::finish() {
  revert_stream_mode();
  ostm_ = 0;

  assert(to_out_ != 0);
  delete to_out_;
  to_out_ = 0;
}

void formatter::write_entity(const pchar_t *entity, size_t size, const int escape_index) {

  escaped_entity_.clear();

  for (pstring::size_type pos = 0; pos < size; ) {
    size_t length = pchar_traits::char_length(entity + pos);
    if (length == 1) {
      pchar_t ch = entity[pos];
      const escape_entry *entry = &escape_table[escape_index];
      while (entry->char_ != 0) {
        if (entry->char_ == ch) {
	  escaped_entity_.append(entry->escaped_, entry->size_);
	  break;
        }
	++entry;
      }
      if (entry->char_ == 0)
        escaped_entity_.append(&ch, 1);
      ++pos;
    }
    else {
      escaped_entity_.append(entity + pos, length);
      pos += length;
    }
  }    
  write_uni(escaped_entity_.get_ptr(), 
	    escaped_entity_.get_content_length());
}

void formatter::write_uni(const pchar_t *entity) {
  write_uni(entity, pchar_traits::length(entity));
}

void formatter::write_uni(const pchar_t *entity, size_t length) {
  
  if (!to_out_->transcode(reinterpret_cast<const char *>(entity), length * sizeof(pchar_t)))
    throw error("transcode failed.", __FILE__, __LINE__);
  
  ostm_->write(to_out_->get_buffer().get_ptr(), 
	       to_out_->get_buffer().get_content_length());
  
  to_out_->clear_transcoded_buffer();
}

void formatter::write_pchar(const pchar_t pch) {
  if (!to_out_->transcode(reinterpret_cast<const char *>(&pch), sizeof(pchar_t)))
    throw error("transcode failed.", __FILE__, __LINE__);
}
