#include "xerces_deserializer.h"
#ifdef AKAXISO2_USE_XERCESC

#include "../../util/scoped_ptr.h"
#include "../../util/sstream.h"
#include "../../framework/document.h"
#include "../../framework/entity_complements.h"
#include "../../util/platform.h"
#include "../../util/string_funcs.h"
#include "../../transcoders/builtin_transcoders.h"
#include "xerces_handler.h"

#include <xercesc/sax2/XMLReaderFactory.hpp>

#include <iostream>
#include <fstream>


using namespace aka2;

xerces_deserializer::xerces_deserializer(transcoder_factory trfact) 
  : trfact_(trfact), parser_(0), ecomps_(0) {
  if (trfact_ == 0)
    trfact_ = default_transcoder::create;
}

xerces_deserializer::~xerces_deserializer() {
  if (ecomps_ != 0)
    delete ecomps_;
  if (parser_ != 0)
    delete parser_;
}

entity_complements &xerces_deserializer::get_entity_complements() {
  if (ecomps_ == 0)
    ecomps_ = system_entity_complements_clone(false, trfact_);
  return *ecomps_;
}

void xerces_deserializer::create_parser() {

  if (parser_ != 0)
    return;

  bool do_namespaces         = true;
  bool do_namespace_prefixes = false;
  bool do_schema             = false;
  bool do_full_checking      = false;
  bool load_external_dtd     = false;

  xercesc::SAX2XMLReader::ValSchemes valschemes = xercesc::SAX2XMLReader::Val_Never;

  parser_ = xercesc::XMLReaderFactory::createXMLReader();

  parser_->setFeature(xercesc::XMLUni::fgSAX2CoreNameSpaces,        do_namespaces);
  parser_->setFeature(xercesc::XMLUni::fgSAX2CoreNameSpacePrefixes, do_namespace_prefixes);

  parser_->setFeature(xercesc::XMLUni::fgXercesSchema,              do_schema);
  parser_->setFeature(xercesc::XMLUni::fgXercesSchemaFullChecking,  do_full_checking);
  parser_->setFeature(xercesc::XMLUni::fgXercesLoadExternalDTD,     load_external_dtd);


  if (valschemes == xercesc::SAX2XMLReader::Val_Auto)  {
      parser_->setFeature(xercesc::XMLUni::fgSAX2CoreValidation, true);
      parser_->setFeature(xercesc::XMLUni::fgXercesDynamic,      true);
  }
  if (valschemes == xercesc::SAX2XMLReader::Val_Never)  {
      parser_->setFeature(xercesc::XMLUni::fgSAX2CoreValidation, false);
  }
  if (valschemes == xercesc::SAX2XMLReader::Val_Always)  {
      parser_->setFeature(xercesc::XMLUni::fgSAX2CoreValidation, true);
      parser_->setFeature(xercesc::XMLUni::fgXercesDynamic,      false);
  }
}

xercesc::SAX2XMLReader *xerces_deserializer::get_parser() {
  create_parser();
  return parser_;
}

document xerces_deserializer::deserialize_internal(TranscodingInputSource &is, 
						   const std::string &source_name) {

  create_parser();
  is.setup();

  entity_complements &ecomps = get_entity_complements();
  ecomps.reset_for_new_document();
  scoped_ptr<xerces_handler> 
    handler(new xerces_handler(&ecomps));
  
  handler->set_source_name(source_name);
  parser_->setContentHandler(handler.get());
  parser_->setErrorHandler(handler.get());
  parser_->parse(is);
  return handler->get_document();
}


document xerces_deserializer::deserialize_stdin(){
  const char *stdin_id = "(std::cin)";
  IStreamInputSource xistm(stdin_id, stdin_id, trfact_, std::cin);
  return deserialize_internal(xistm, stdin_id);
}

document xerces_deserializer::deserialize(const char *buff){
  const char *membuf_id = "(membuf)";
  size_t length = strlen(buff);
  BufferInputSource xistm(membuf_id, membuf_id, trfact_, buff, length);
  return deserialize_internal(xistm, membuf_id);
}

document xerces_deserializer::deserialize(const std::string& document) {
  const char *string_id = "(std::string)";
  BufferInputSource xistm(string_id, string_id,
			  trfact_,
			  document.data(), document.size());
  return deserialize_internal(xistm, string_id);
}

document xerces_deserializer::deserialize(std::istream &istm){
  const char *istm_id = "(std::istream)";
  IStreamInputSource xistm(istm_id, istm_id, trfact_, istm);
  return deserialize_internal(xistm, istm_id);
}

document xerces_deserializer::deserialize_file(const std::string& filename) {
  const char *file_id = filename.c_str();
  std::ifstream file(filename.c_str());
  if (!file.is_open()) {
    throw error("Cannot open file, " + quote(filename) + ".", __FILE__, __LINE__);
  }
  IStreamInputSource xistm(file_id, file_id, trfact_, file);
  return deserialize_internal(xistm, file_id);
}

#endif /* AKAXISO2_USE_XERCESC */
