#include "qname.h"
#include <cassert>
#include "namespace_statics.h"
#include "../util/string_funcs.h"
#include "../unicode/utf8_traits.h"


using namespace aka2;

void qname::validate_name() const {
  if (name_.empty())
    throw error("localpart of qualified name is empty.", __FILE__, __LINE__);
  if (name_[0] == ':') {
    std::string message = "Qualified name " + quote(prefix() + ":" + local()) 
      + " has '::' in it's name field.";
    throw error(message, __FILE__, __LINE__);
  }
}


void qname::set(const pivot_tag &trc, const pstring &rawname, const prefix_map &pfs) {
  int pos = pchar_traits::find_first(rawname.data(), rawname.size(), ':');

  if (pos == -1) { // local, not qualified.
    namespace_id_ = empty_token;
    name_ = rawname;
  }
  else {
    pstring prefix = rawname.substr(0, pos);;
    namespace_id_ = pfs.get_prefix_id(prefix);
    if (namespace_id_ == unregistered_token) {
      throw error("namespace prefix(\"" 
		  + quote(to_lcp(prefix)) 
		  + "\") is not registered.",
		  __FILE__, __LINE__);
    }
    name_ = rawname.substr(pos + 1, rawname.length() - pos);
  }
  validate_name();
}

void qname::set(const pivot_tag &trc, 
		const pstring &uri, const pstring &localname) {
  if (uri.empty()) {
    namespace_id_ = empty_token;
  }
  else {
    namespace_id_ = g_namespaces_.get_namespace_id(uri);
    if (namespace_id_ == unregistered_token)
      namespace_id_ = g_any_namespaces_.get_namespace_id(uri);
  }
  name_ = localname;
  validate_name();
}

void qname::set(const pivot_tag &trc, const id_type namespace_id, const pstring &name ) {
  namespace_id_ = namespace_id;
  name_ = name;
  validate_name();
}

qname::qname(const pivot_tag &trc, const pstring &uri, const pstring &name) {
  set(pivot, uri, name);
}

qname::qname(const pivot_tag &trc, const pstring &rawname, const prefix_map &pfs) {
  set(pivot, rawname, pfs);
}

qname::qname(const pivot_tag &trc, const pstring &rawname) {
  set(pivot, rawname, *g_pfs_);
}

qname::qname(const pivot_tag &trc, const id_type namespace_id, const pstring &name) {
  set(pivot, namespace_id, name);
}


pstring qname::qualified(const pivot_tag &trc, const prefix_map &pfs) const {
  assert(namespace_id_ != unregistered_token);

  if (is_qualified()) {
    const pstring &prefix = pfs.get_prefix(namespace_id_);
    prefix_cache_ = prefix.data();
    if (prefix.empty()) // Default namespace.  Prefix is "".
      return name_;
    return prefix + pchar_t(':') + name_;
  }
  else {
    return name_;
  }
}

pstring qname::qualified(const pivot_tag &trc) const{
  return qualified(trc, *g_pfs_);
}

const pstring &qname::local(const pivot_tag &trc) const {
  return name_;
}

const pstring &qname::prefix(const pivot_tag &trc, const prefix_map &pfs) const {
  return pfs.get_prefix(namespace_id_);
}

const pstring &qname::prefix(const pivot_tag &trc) const {
  return g_pfs_->get_prefix(namespace_id_);
}

bool qname::empty() const {
  return name_.empty();
}

/** LCP-specific versions */

qname::qname(const std::string &uri, const std::string &name) {
  set(pivot, to_pivot(uri), to_pivot(name));
}

qname::qname(const std::string &rawname, const prefix_map &pfs) {
  set(pivot, to_pivot(rawname), pfs);
}

qname::qname(const std::string &rawname) {
  set(pivot, to_pivot(rawname), *g_pfs_);
}

qname::qname(const id_type namespace_id, const std::string &name) {
  set(pivot, namespace_id, to_pivot(name));
}


std::string qname::qualified(const prefix_map &pfs) const{
  return to_lcp(qualified(pivot, pfs));
}

std::string qname::qualified() const{
  return to_lcp(qualified(pivot, *g_pfs_));
}

const std::string qname::local() const {
  return to_lcp(name_);
}

std::string qname::prefix(const prefix_map &pfs) const {
  return to_lcp(pfs.get_prefix(namespace_id_));
}

std::string qname::prefix() const {
  return to_lcp(g_pfs_->get_prefix(namespace_id_));
}


/** helper funcs */

std::string aka2::quote(const qname &str) {
  return quote(str.qualified());
}

std::string aka2::quote(const qname &str, const prefix_map &pfs) {
  return quote(str.qualified(pfs));
}

std::string aka2::tag(const qname &str) {
  return tag(str.qualified());
}

std::string aka2::tag(const qname &str, const prefix_map &pfs) {
  return tag(str.qualified(pfs));
}
