//CUPPA:include=+
//CUPPA:include=-
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestAssert.h>

#include <akaxiso2/akaxiso2.h>
#include <akaxiso2/content_model.h>

#include <iostream>

//CUPPA:namespace=+
//CUPPA:namespace=-


namespace {

  typedef std::list<std::string> fixed_array;
  typedef aka2::fixed_sequential_array<fixed_array, xiso::leaf<std::string> > fixed_array_leaf;
  

  struct all_sample {
    all_sample() : ptrlong_(0){}
    double dbl_;
    float float_;
    long long_;
    short short_;
    std::string string_;
    aka2::deep_ptr<long> ptrlong_;
    fixed_array fixed_array_;
  };


  class all_sample_leaf : public aka::all<all_sample, all_sample_leaf> {
  public:
    void model() {
      xmltype("all_sample");
      member("double", &all_sample::dbl_);
      member("float", &all_sample::float_);
      member("long", &all_sample::long_);
      member("short", &all_sample::short_);
      member("string", &all_sample::string_);
      ptrmember("ptrlong", &all_sample::ptrlong_); // minOccurs = 0, maxOccurs = 1.
      fixed_member<std::string>("fixed_string", "fixed_value");
      fixed_member<std::string>("fixed_string2", "fixed_value2", xiso::leaf<std::string>());
      fixed_array("fixed_array", "array_value", 
		  &all_sample::fixed_array_, fixed_array_leaf(),
		  0, 1);
    }
  };

  struct all_sample1 {  };

  typedef std::list<long> dummy_list;
  struct dummy_list_leaf : public aka::sequential_array<dummy_list, xiso::leaf<long> > {  };

  struct empty_all {
    dummy_list list_;
  };

  struct empty_all_leaf : public aka::sequence<empty_all, empty_all_leaf> {
    void model() {
      member("dummy", &empty_all::list_, dummy_list_leaf(), 0, 1);
    }
  };
}


class allTest : public CppUnit::TestFixture {
  CPPUNIT_TEST_SUITE(allTest);
//CUPPA:suite=+
  CPPUNIT_TEST(test_equality0);
  CPPUNIT_TEST(test_equality1);
  CPPUNIT_TEST(test_serialize);
  CPPUNIT_TEST(test_serialize_empty_member_array);
  CPPUNIT_TEST(test_parse);
  CPPUNIT_TEST(test_copy);
  CPPUNIT_TEST(test_replicate);
  CPPUNIT_TEST(test_parse_empty_member_array);
  CPPUNIT_TEST(test_wrongAll0);
  CPPUNIT_TEST(test_wrongAll1);
//CUPPA:suite=-
  CPPUNIT_TEST_SUITE_END();
private:

//    // your stuff...

public:

  virtual void setUp() { 
    aka::initialize();
    aka::doctype("all_test", all_sample_leaf());
    aka::doctype("empty_array", empty_all_leaf());
  }

  virtual void tearDown(){
    aka::uninitialize();
  }


  static void setValues(all_sample &seq){
    seq.dbl_ = 0.2253;
    seq.float_ = 0.225f;
    seq.long_ = 1638453;
    seq.short_ = 32600;
    seq.string_ = "Test String for a member in all.";
  }

//CUPPA:decl=+
  void test_equality0(){
    all_sample seq;
    all_sample seq1;

    setValues(seq);
    setValues(seq1);

    bool equal = aka::equals(seq, seq1, all_sample_leaf());
    CPPUNIT_ASSERT_MESSAGE("Equality test failed.", equal);
  }

  void test_equality1(){
    all_sample seq;
    all_sample seq1;

    setValues(seq);
    setValues(seq1);
    seq.dbl_ = 0;

    bool equal = aka::equals(seq, seq1, all_sample_leaf());
    CPPUNIT_ASSERT_MESSAGE("Equality test failed.", !equal);
  }


  void test_copy() {
    all_sample seq;
    all_sample seq1;

    setValues(seq);
    aka::copy_element(seq1, seq, all_sample_leaf());
    bool is_equal = aka::equals(seq, seq1, all_sample_leaf());
    CPPUNIT_ASSERT_MESSAGE("Copy failed.", is_equal);
  }

  void test_replicate(){
    all_sample seq;

    setValues(seq);
    all_sample *seq1 = aka::replicate_element(seq, all_sample_leaf());
    bool is_equal = aka::equals(seq, *seq1, all_sample_leaf());
    delete seq1;

    CPPUNIT_ASSERT_MESSAGE("Replicate failed.", is_equal);
  }

//   void test_compare_different_type(){
//     all_sample seq;
//     all_sample1 seq1;
    
//     bool exception_caught = false;
    
//     try {
//       element_equals(&seq, &seq1, all_sample_leaf::dispatcher_);
//     }
//     catch (... ) {
//       exception_caught = true;
//     }
//     CPPUNIT_ASSERT_MESSAGE("Exception 'invalid_argument' not thrown", exception_caught);
//   }


  void test_serialize() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    all_sample root;
    setValues(root);
    ser.serialize(root, "all_test", ostm);
//      std::cout << ostm.rdbuf()->str() << std::endl;
  }


  void test_serialize_empty_member_array() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    empty_all root;
    ser.serialize(root, "empty_array", ostm);
//      std::cout << ostm.rdbuf()->str() << std::endl;
  }

  void test_parse() {
    std::ostringstream ostm;
    aka::xml_serializer ser;
    
    all_sample root;
    setValues(root);
    root.fixed_array_.push_back(std::string());

    ser.serialize(root, "all_test", ostm);
    //    std::cout << ostm.rdbuf()->str() << std::endl;

    aka::expat_deserializer deserializer;

    std::string errMsg;
    aka::document deserialized;
    try {
      deserialized = deserializer.deserialize(ostm.rdbuf()->str());
    }
    catch (const std::exception &e) {
      errMsg = e.what();
    }

    CPPUNIT_ASSERT_MESSAGE(errMsg.c_str(), errMsg.empty());

    CPPUNIT_ASSERT_MESSAGE("wrong document name. ", 
			   aka::document_of(deserialized, "all_test"));


    all_sample *seq = aka::root_cast<all_sample>(deserialized);
    bool is_equal = aka::equals(*seq, root, all_sample_leaf());
    if (!is_equal)
      ser.serialize(*seq, "all_test", std::cout);
    CPPUNIT_ASSERT_MESSAGE("Deserialized document is not the same as original.", is_equal);

  }


  void test_parse_empty_member_array() {
    std::ostringstream ostm;
    aka::xml_serializer ser;

    empty_all root;
    ser.serialize(root, "empty_array", ostm);

    aka::document doc = aka::deserialize(ostm.rdbuf()->str());
    CPPUNIT_ASSERT_MESSAGE("parsed document has wrong name.", 
			   aka::document_of(doc, "empty_array"));
    
    empty_all *parsed = aka::root_cast<empty_all>(doc);
    CPPUNIT_ASSERT_MESSAGE("parsed document has wrong-sized array", parsed->list_.empty());
  }


   void test_wrongAll0() {
    
     std::string doc = 
       "<?xml version=\"1.0\"?>\n"
       "<all_test>\n"
       "  <double>0.2</double>\n"
       "  <fixed_string>fixed_value</fixed_string>\n"
       "  <float>3.0</float>\n"
       "  <long>2</long>\n"
       "  <fixed_array>array_value</fixed_array>\n"
       "  <string>Test</string>\n"
       "  <short>2</short>\n"
       "  <ptrlong>11</ptrlong>\n"
       "  <fixed_string2>fixed_value2</fixed_string2>\n"
       "</all_test>";
     
     bool err = false;
     try {
       aka::deserialize(doc);
     }
     catch (const std::exception &) {
       err = true;
     }
     CPPUNIT_ASSERT_MESSAGE("cannot handle inversed child elements.", !err);
   }

   void test_wrongAll1() {
    
     std::string doc = 
       "<?xml version=\"1.0\"?>\n"
       "<all_test>\n"
       "  <double>0.2</double>\n"
       "  <float>3.0</float>\n"
       "  <long>2</long>\n"
       "  <short>2</short>\n"
       "  <string>Test</string>\n"
       "  <ptrlong>11</ptrlong>\n"
       "  <fixed_string>fixed_value</fixed_string>\n"
       // Lack of fixed_string.
//        "  <fixed_string2>fixed_value2</fixed_string2>\n"
       "  <fixed_array>array_value</fixed_array>\n"
       "</all_test>";

     bool err = false;
     try {
       aka::deserialize(doc);
     }
     catch (const std::exception &) {
       err = true;
     }
     CPPUNIT_ASSERT_MESSAGE("parse error not reported.", err);
   }


//CUPPA:decl=-
};

//CUPPA:impl=+
//CUPPA:impl=-

CPPUNIT_TEST_SUITE_REGISTRATION(allTest);
