#include "osixaka2.h"
#include "preprocessor.h"
#include "processor.h"
#include "dtd_processor.h"
#include "generator.h"
#include "akaxisonizer.h"
#include "dot_generator.h"

#include "serializer.h"
#include <fstream>

using namespace osx;

void osixaka2::initialize() {
  aka::xmlns("std", "http://akaxiso.sourceforge.jp/akaxiso2/c++-std");
  osx_instantiate_xiso();
}

void osixaka2::set_basedir(const std::string &basedir) {
  basedir_ = basedir;
}

void osixaka2::set_outputdir(const std::string &outputdir) {
  outputdir_ = outputdir;
}

int osixaka2::preprocess(const std::string &filename) {
  try {
    preprocessor pre(registry_, ostm_, verbose_);
    pre.preprocess(filename);
    return 0;
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    ostm_ << "Failed to generate files. Aborting..." << std::endl;
    return 1;
  }
  catch ( ... ) {
    ostm_ << "Unknown exception caught. Aborting..." << std::endl;
    return 1;
  }
}  

int osixaka2::process() {
  try {
    processor prc(registry_, ostm_, verbose_);
    units uts = registry_.get_units();

    units::iterator it;
    for (it = uts.begin(); it != uts.end(); ++it) {
      if (verbose_)
	ostm_ << "scanning " << (*it)->filename_ << "..." << std::flush;
      prc.scan(**it);
      if (verbose_)
	ostm_ << "done." << std::endl;
    }
    for (it = uts.begin(); it != uts.end(); ++it) {
      if (verbose_)
	ostm_ << "process " << (*it)->filename_ << "." << std::endl;
      prc.process(**it);
    }
    return 0;
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    ostm_ << "Failed to process Schema. Aborting..." << std::endl;
    return 1;
  }
  catch ( ... ) {
    ostm_ << "Unknown exception caught. Aborting..." << std::endl;
    return 1;
  }
}  


#ifdef AKAXISO2_USE_EXPAT
int osixaka2::process_dtd(const std::string &filepath) {
  dtd_processor processor(registry_, ostm_, verbose_, true);
  return processor.process(filepath);
}
#else

int osixaka2::process_dtd(const std::string &filepath) {
  ostm_ << "To use DTD processor, enable expat." << std::endl;
  return 1;
}
#endif


void osixaka2::generate_all_in_one() {

  output_file elm(outputdir_, basedir_);
  output_file xiso(outputdir_, basedir_);
  output_file ximpl(outputdir_, basedir_);

  units uts = registry_.get_units();
  const registry_unit &unit = *uts.back();
  
  generator gen(registry_, elm, xiso, ximpl, use_soft_array_, ostm_, verbose_);
  gen.open(unit.basename_);

  gen.prepare();
  
  gen.write_custom_includes(registry_.get_imported_unit());

  for (units::iterator it = uts.begin(); it != uts.end(); ++it) {
    gen.write_custom_includes(**it);
    gen.resolve(**it, true);
  }

  func_generator fgen(registry_, elm, ximpl, 
		      instantiate_xiso_name_,
		      ostm_, verbose_);
  fgen.generate(); 

  gen.close();
}


void osixaka2::generate_target_ns() {

  output_file elm(outputdir_, basedir_);
  output_file xiso(outputdir_, basedir_);
  output_file ximpl(outputdir_, basedir_);

  units uts = registry_.get_units();
  const registry_unit &unit = *uts.back();
  
  generator gen(registry_, elm, xiso, ximpl, use_soft_array_, ostm_, verbose_);
  gen.open(unit.basename_);

  gen.prepare();
  
  for (units::iterator it = uts.begin(); unit.target_ns_id_ != (*it)->target_ns_id_; ++it) {
    gen.write_include((*it)->basename_);
    gen.resolve(**it, false);
  }

  gen.write_custom_includes(registry_.get_imported_unit());
  gen.write_custom_includes(unit);
  gen.resolve(unit, true);

  gen.close();
}



int osixaka2::generate(bool one_ns) {

  try {
    if (one_ns) {
      generate_target_ns();
    }
    else {
      generate_all_in_one();
    }
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Code generation failed." << std::endl;
    return 1;
  }
  return 0;
}


int osixaka2::generate_functions(const std::string &filename) {
  
  try {
    output_file header(outputdir_, basedir_), impl(outputdir_, basedir_);
    func_generator fgen(registry_, header, impl, 
			instantiate_xiso_name_,
			ostm_, verbose_);
    fgen.open(filename);
    fgen.write_includes();
    fgen.generate();
    fgen.close();
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Code generation failed." << std::endl;
    return 1;
  }
  return 0;
}

int osixaka2::generate_dot() {

  units uts = registry_.get_units();
  const registry_unit &unit = *uts.back();
  std::string filepath = outputdir_ + unit.basename_ + ".dot";
  std::ofstream ostm(filepath.c_str());
  if (!ostm.is_open()) {
    ostm_ << "Failed to open file, " + aka::quote(filepath) + "." << std::endl;
    return 1;
  }

  try {
    dot_generator gen(registry_, ostm);
    gen.generate();
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Failed to generate dot file." << std::endl;
    return 1;
  }
  ostm.close();
  return 0;
}


int osixaka2::akaxisonize(int level) {
  try {
    units uts = registry_.get_units();
    for (units::const_iterator uit = uts.begin(); uit != uts.end(); ++uit) {
      akaxisonizer akx(registry_, registry_[(*uit)->target_ns_id_], ostm_, verbose_);
      akx.check_names();
      akx.dereference();
      akx.add_missings();
      if (level == 1)
	akx.optimize();
      akx.akaxisonize();
    }
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl
	  << "Failed during akaxisonization." << std::endl;
    return 1;
  }
  return 0;
}

int osixaka2::dump_schemas(std::ostream &ostm) const {
  try {
    const_units uts = registry_.get_units();
    if (uts.size() == 1) {
      serialize(*uts.front()->schema_, ostm);
    }
    else {
      for (const_units::const_iterator it = uts.begin(); it != uts.end(); ++it) {
	ostm << "----------------------------------------------" << std::endl
	     << "File name: " << (*it)->filename_ << std::endl
	     << "----------------------------------------------" << std::endl
	     << std::endl;
	serialize(*(*it)->schema_, ostm);
      }
    }
  }
  catch (const std::exception &e) {
    ostm_ << e.what() << std::endl;
    return 1;
  }
  return 0;
}

void osixaka2::set_instantiate_xiso_name(const std::string &instantiate_xiso_name) {
  instantiate_xiso_name_ = instantiate_xiso_name;
}
