/* -*- c++ -*- */
#ifndef OSIXAKA2_PROCESSOR_H__
#define OSIXAKA2_PROCESSOR_H__

#include <osixaka2/serializer.h>
#include <osixaka2/exception.h>
#include <osixaka2/registry.h>
#include <osixaka2/platform.h>

namespace osx {

  typedef std::list<aka::item> xschoice;

  template<class T>
  class declarations {
    struct value {
      value() : t_(0){}
      value(T *t, const aka::qname &name) : name_(name), t_(t){}
      aka::qname name_;
      const T* t_;
    };
    typedef std::map<aka::qname, value, aka::qname_less> cache;
    cache cache_;
  public:
    void add(const aka::qname &name, T* t);
    T* get(const aka::qname &name);
    bool exists(const aka::qname &name) const;
    typedef TYPENAME cache::iterator iterator;

    iterator begin() { return cache_.begin(); }
    iterator end() { return cache_.end(); }
  };

  template<class T>
  void declarations<T>::add(const aka::qname &name, T* t) {
    std::pair<TYPENAME cache::iterator, bool> res =
      cache_.insert(OSX_TYPENAME cache::value_type(name, value(t, name)));
    if (!res.second)
      raise_name_error("type", name, "declared more than once.",
		       __FILE__, __LINE__);
  }

  template<class T>
  T* declarations<T>::get(const aka::qname &name) {
    TYPENAME cache::iterator it = cache_.find(name);
    if (it != cache_.end())
      return it->second.t_;
    return 0;
  }

  template<class T>
  bool declarations<T>::exists(const aka::qname &name) const {
    TYPENAME cache::const_iterator it = cache_.find(name);
    return it != cache_.end();
  }



  template<class T>
  inline bool is_array(const T& t) {
    return (t.minOccurs_ != 1) || (t.maxOccurs_ != 1);
  }

  template<class T>
  aka::occurrence occurrence(const T& t) {
    return aka::occurrence(t.minOccurs_, t.maxOccurs_);
  }

  class processor {
  public:
    processor(registry &reg, std::ostream &ostm, bool verbose)
      : registry_(reg), target_ns_id_(aka::unregistered_token), ostm_(ostm), verbose_(verbose) {}
    void scan(registry_unit &unit);
    void process(registry_unit &unit);

  private:

    // For schemaTop.
    type_ref process_topLevelElement(const aka::qname &name, 
				     const xs::topLevelElement *el = 0);
    void check_reference(const xs::topLevelElement &el);

    const gattribute_def *process_topLevelAttribute(const aka::qname &name, 
						    const xs::topLevelAttribute &attr);
    const attributeGroup_def *process_topLevelAttributeGroup(const aka::qname &name, 
							     const xs::namedAttributeGroup *ag = 0);
    // For redefinables.
    void process_redefinable(const xs::redefinable &redefinable);
    type_ref process_topLevelSimpleType(const aka::qname &name, 
					const xs::topLevelSimpleType *st = 0);
    type_ref process_topLevelComplexType(const aka::qname &name, 
					 const xs::topLevelComplexType *ct = 0);
    type_ref declare_topLevelComplexType(const aka::qname &name);

    type_ref process_namedGroup(const aka::qname &name, 
				const xs::namedGroup *gp = 0);
    
    void process_attrDecls(const aka::qname &hint, 
			   attribute_defs &attrdefs, const xs::attrDecls &decls);
    static bool is_attrDeclsEmpty(const xs::attrDecls &decls);
    
    void process_attribute(const aka::qname &hint,
			   attribute_defs &attrdefs, const xs::attribute &attr);
    type_ref process_localSimpleType(const aka::qname &hint, const xs::localSimpleType &lst,
				     user_classes &uss);
    type_ref process_localComplexType(const aka::qname &name, const xs::localComplexType &lct,
				      user_classes &uss);
    type_ref process_typeChoice(const aka::qname &hint, const xschoice &choice, bool is_toplevel);
    
    type_ref process_complexTypeModel(const aka::qname &name, 
				      const aka::qname &classname,
				      const xs::complexTypeModel &model,
				      user_classes &uss);
    element_type process_complexTypeParticle(const aka::qname &name,
					     const xs::typeDefParticle &particle);
    type_ref process_restrictionClassDeclaration(const aka::qname &name,
						 const user_class &super_def,
						 const xs::typeDefParticle &particle, 
						 const xs::attrDecls &attrDecls);
    type_ref process_complexContent(const aka::qname &name, const aka::qname &classname,
				    const xs::complexContent &cc,
				    user_classes &uss);
    type_ref process_complexRestrictionType(const aka::qname &name, 
					    const aka::qname &classname,
					    const xs::complexRestrictionType &cr,
					    user_classes &uss);
    type_ref process_extensionType(const aka::qname &name, 
				   const aka::qname &classname,
				   const xs::extensionType &et,
				   user_classes &uss);
    
    type_ref process_simpleContent(const aka::qname &name, const aka::qname &classname,
				   const xs::simpleContent &sc, user_classes &uss);
    type_ref process_simpleRestrictionType(const aka::qname &name, 
					   const aka::qname &classname,
					   const xs::simpleRestrictionType &sr,
					   user_classes &uss);
    type_ref process_simpleExtensionType(const aka::qname &name, 
					 const aka::qname &classname,
					 const xs::simpleExtensionType &se,
					 user_classes &uss);
    
    type_ref process_sequence(const aka::qname &name, const aka::qname &classname,
			      const aka::occurrence &occ,
			      const xs::nestedParticle &np,
			      user_classes &uss);
    type_ref process_choice(const aka::qname &name, const aka::qname &classname,
				     const aka::occurrence &occ,
				     const xs::nestedParticle &np,
				     user_classes &uss);
    type_ref process_all(const aka::qname &name, const aka::qname &classname,
			 const aka::occurrence &occ,
			 const xs::narrowMaxMin_array &ams,
			 user_classes &uss);
    
    void process_all_members(element_types &etypes, 
			     const aka::qname &hint, const xs::narrowMaxMin_array &am);

    type_ref process_simpleDerivation(const aka::qname &name,
				      const xs::simpleDerivation &der,
				      user_classes &uss);
    type_ref process_simpleRestrictionModel(const aka::qname &name,
					    const xs::simpleRestrictionModel &srm,
					    user_classes &uss);

    void process_nestedParticle(element_types &etypes, const aka::qname &hint, 
				const xs::nestedParticle &np);

    element_type process_localElement(const xs::localElement &le, const aka::qname &hint);

    element_type process_localSequence(const aka::qname &name, const xs::explicitGroup &seq);
    element_type process_localChoice(const aka::qname &name, const xs::explicitGroup &cho);
    element_type process_localAll(const aka::qname &name, const xs::all &all);

    element_type process_allElementMember(const xs::narrowMaxMin &ame, const aka::qname &hint);
    element_type process_any(const xs::any &any);
    element_type process_member_groupRef(const xs::groupRef &ref);

    void restrict_attrDecls(const aka::qname &hint, 
			    attribute_defs &attrdefs, const xs::attrDecls &attrDecls);
    void restrict_attribute(attribute_defs &attrdefs, const attribute_type &atype);
    void prohibit_attribute(attribute_defs &attrdefs, const aka::qname &tagname);
    attribute_type get_local_attribute_type(const aka::qname &hint, const xs::attribute &attr);
    void extend_attrDecls(const aka::qname &name, 
			  attribute_defs &attrdefs, const xs::attrDecls &attrDecls);
    void extend_attribute(const aka::qname &name,
			  attribute_defs &attrdefs, const xs::attribute &attr);

    void process_requested_type(const aka::qname &name);
    void process_requested_attributeGroup(const aka::qname &name);



    osx::imported_type *process_osx_resolve(const aka::qname &name, const aka::qname &osx_resolve);
    osx::imported_type *process_osx_leaf(const aka::qname &name, 
				    const xs::token &osx_leaf, 
				    const xs::token &osx_classname, 
				    const xs::token &osx_typedef_of);
  
    static aka::qname create_anonymous_name(const aka::qname &name, const std::string &postfix,
					    size_t index);
    aka::qname create_attribute_name(const aka::qname &name, const xs::formChoice *form);
    aka::qname create_element_name(const aka::qname &name, const xs::formChoice *form);
    aka::qname create_array_name(const aka::qname &name);
    aka::qname create_classname(const aka::qname &name, const std::string &classname) const;
    aka::qname create_local_name(const std::string &classname) const;


    registry &registry_;

    aka::id_type target_ns_id_;
    registry_unit *current_unit_;
    std::string basepath_;
    std::ostream &ostm_;
    bool verbose_;

    bool element_qualified_;
    bool attribute_qualified_;

    typedef declarations<const xs::topLevelComplexType> topLevelComplexTypes;
    typedef declarations<const xs::topLevelElement> topLevelElements;
    typedef declarations<const xs::topLevelAttribute> topLevelAttributes;
    typedef declarations<const xs::namedAttributeGroup> topLevelAttributeGroups;
    typedef declarations<const xs::topLevelSimpleType> topLevelSimpleTypes;
    typedef declarations<const xs::namedGroup> namedGroups;

    topLevelComplexTypes topLevelComplexTypes_;
    topLevelElements topLevelElements_;
    topLevelAttributes topLevelAttributes_;
    topLevelAttributeGroups topLevelAttributeGroups_;
    topLevelSimpleTypes topLevelSimpleTypes_;
    namedGroups namedGroups_;

    qname_set simpleType_processing_;
    qname_set attrG_processing_;
    qname_set complexType_processing_;
  };

}
#endif

