#include "test_executor.h"

#include <akaxiso2/akaxiso2.h>
#include <osixaka2/osixaka2.h>
#include <akaxiso2/util/filepath.h>
#include <akaxiso2/util/tokenizer.h>

#include <fstream>
#include <sstream>

#include <sys/stat.h>
#include <unistd.h>
#include <libgen.h>
#include <errno.h>


void test_executor::mkdir_easy(const std::string &dirpath) { 
  std::string simplified = aka::simplify_path(dirpath);

  typedef std::vector<std::string> strings;
  strings path_tokens;

  cuppa::tokenizer<char> tokenizer(dirpath);
  while (tokenizer.has_more_token("/\\")) {
    path_tokens.push_back(tokenizer.next_token("/\\"));
  }

  std::string path(".");
  for (strings::const_iterator it = path_tokens.begin();
       it != path_tokens.end(); ++it) {
    path = path + std::string("/") + *it;
    int res = mkdir(path.c_str(), 0777);
    if (res != 0) {
      if (errno != EEXIST) {
	perror("mkdir");
      }
    }
  }
}

std::string test_executor::dirname(const std::string &path) {
  char *buffer = strdup(path.c_str());
  std::string dirpath = ::dirname(buffer);
  free(buffer);
  return dirpath;
}

void test_executor::do_check_schema_wellformedness(test_result *result) { 

  std::string schema_path = result->get_schema_path();
  std::ifstream ifile(schema_path.c_str());
  aka::expat_parser parser(0);

  try {
    parser.parse_istream(ifile, schema_path);
  }
  catch (const std::exception &e) {
    result->add_error(schema_not_wellformed, e.what());
    return;
  }
  catch ( ... ) {
    result->add_error(schema_not_wellformed, "Unhandled error.");
    return;
  }
}

void test_executor::do_validate_schema_deserialization(test_result *result) { 

  std::string schema_path = result->get_schema_path();

  try {
    aka::deserialize_file(schema_path);
  }
  catch (const std::exception &e) {
    result->add_error(schema_not_deserialized, e.what());
    return;
  }
  catch ( ... ) {
    result->add_error(schema_not_deserialized, "Unhandled error.");
    return;
  }
  
}

void test_executor::do_validate_source_generation(test_result *result, 
						  const std::string &testdir) { 
  std::ostringstream ostm;
  osx::osixaka2 osi(ostm, false, false);

  std::string output_path = testdir + '/' + result->get_test_dir_path();
  mkdir_easy(output_path);

  osi.load_defaults();
  osi.commit_preference();
  osi.set_outputdir(output_path);
  
  bool ok = false;
  if (osi.preprocess(result->get_schema_path()) != 0) {
    result->add_error(source_generation_failed, ostm.str());
  }
  else if (osi.process() != 0) {
    result->add_error(source_generation_failed, ostm.str());
  }
  else if (osi.akaxisonize(1) != 0) {
    result->add_error(source_generation_failed, ostm.str());
  }
  else if (osi.generate(false) != 0) {
    result->add_error(source_generation_failed, ostm.str());
  }
  else {
    ok = true;
  }

}

void test_executor::prepare() { 
  results_.clear();
}

void test_executor::set_testSuite(const ts::testSuite &testSuite, 
				  const std::string &testSuite_path) { 

  std::string dirpath = dirname(testSuite_path);

  for (ts::ref_array::const_iterator refit = testSuite.testSetRef_.begin();
       refit != testSuite.testSetRef_.end(); ++refit) {
    bool error_found = false;
    aka::document doc;
    std::string testSet_path = dirpath + '/' + refit->xlink_href_;
    try {
      doc = aka::deserialize_file(testSet_path);
    }
    catch (const std::exception &e) {
      std::cerr << e.what() << std::endl
		<< "Unhandled exception." << std::endl;
      error_found = true;
    }
    catch ( ... ) {
      std::cerr << "Unhandled exception." << std::endl;
      error_found = true;
    }

    if (!error_found) {
      if (aka::document_of(doc, "ts:testSet")) {
	set_testSet(*aka::root_cast<ts::testSet>(doc), testSet_path);
      }
      else {
	std::cerr << testSet_path << ": Unknown tagname, " 
		  << doc.get_name().qualified() << "." << std::endl;
      }
    }
  }
}

void test_executor::set_testSet(const ts::testSet &testSet, const std::string &testSet_path) { 

  std::string dirpath = dirname(testSet_path);
  ts::testSet *ts = new ts::testSet(testSet);
  testSets_.push_back(ts);

  for (ts::testGroup_array::const_iterator setit = ts->testGroup_.begin(); 
       setit != ts->testGroup_.end(); ++setit) {
    const ts::testGroup &testGroup = *setit;
    if (testGroup.schemaTest_.present()) {
      const ts::schemaTest &schemaTest = *testGroup.schemaTest_;
      for (ts::ref_array::const_iterator refit = schemaTest.schemaDocument_.begin();
	   refit != schemaTest.schemaDocument_.end(); ++refit) {
	test_result result(*ts, dirpath, 
			   testGroup, schemaTest,
			   *refit);
	results_.push_back(result);
      }
    }
    else {
      std::cerr << testSet_path << ':' << ts->name_ << ':' << testGroup.name_ 
		<< ": No ts:schemaTest found." << std::endl;
    }
  }
}

void test_executor::check_files() { 
  for (test_results::iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    std::string schema_path = resit->get_schema_path();

    struct stat statb;
    int res = stat(schema_path.c_str(), &statb);

    if (res != 0) {
      resit->add_error(file_not_found, strerror(errno));
    }
    else if (!S_ISREG(statb.st_mode)) {
      resit->add_error(file_not_found, "File is not a regular file.");
    }
  }
}

void test_executor::check_schema_wellformedness() { 
  for (test_results::iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    if (resit->passed()) {
      do_check_schema_wellformedness(&*resit);
    }
  }
}


void test_executor::validate_schema_deserialization() { 
  for (test_results::iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    if (resit->passed()) {
      do_validate_schema_deserialization(&*resit);
    }
  }
}

void test_executor::validate_source_generation() { 
  const char *testdir = "./generated";

  mkdir(testdir, 0777);

  for (test_results::iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    if (resit->passed()) {
      do_validate_source_generation(&*resit, testdir);
    }
  }
}

void test_executor::write_invalid_passes(std::ostream &ostm) { 
  for (test_results::iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    if (resit->succeeded_but_invalid()) {
      resit->add_error(succeeded_but_invalid, "Succeeded but invalid.");
      resit->write_error(succeeded_but_invalid, ostm);
    }
  }
}

void test_executor::write_error(outcome oc, std::ostream &ostm) const {
  for (test_results::const_iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    if (resit->failed_but_valid())
      resit->write_error(oc, ostm);
  }
}

int test_executor::get_num_tests() const {
  return results_.size();
}

int test_executor::get_num_tests(outcome oc) const {
  int num = 0;
  for (test_results::const_iterator resit = results_.begin();
       resit != results_.end(); ++resit) {
    if (resit->get_outcome() == oc)
      ++num;
  }
  return num;
}

