#include <akaxiso2/akaxiso2.h>
#include <iostream>
#include <akaxiso2/util/string_funcs.h>
#include <akaxiso2/transcoders/win32_transcoder.h>

#if !defined(_WIN32)
#  include "macros.h"
#  include <unistd.h>
#else
  extern "C" int getopt (int argc, char *const *argv, const char *shortopts);
  extern "C" int optind;
#endif

static void usage() {
  std::cout << "aka2-config" << std::endl
	    << " Show akaxiso2 library configuration." << std::endl
#if defined(HAVE_CONFIG_H)
	    << "  -i  : show include path." << std::endl
	    << "  -l  : show LDFLAGS." << std::endl
      << "  -C  : show CFLAGS." << std::endl
      << "  -X  : show CXXFLAGS." << std::endl
#endif
	    << "  -e  : show default encoding name." << std::endl
	    << "  -E  : show internal encoding name." << std::endl
	    << "  -t  : show transcoder configuration." << std::endl
	    << "  -x  : show XML parser configuration." << std::endl
	    << "  -h  : show this message." << std::endl
	    << "  -v  : show version." << std::endl;
}



std::string internal_encoding() {
#ifdef AKAXISO2_INTERNAL_UCS2
  return "UCS-2";
#else
  return "UTF-8";
#endif
}



#if defined(HAVE_CONFIG_H)

std::string cflags() {
  return AKAXISO2_CFLAGS;
}

std::string cxxflags() {
  return AKAXISO2_CXXFLAGS;
}

std::string includes() {
  return AKAXISO2_INST_INCLUDES;
}

std::string ldflags() {
  return AKAXISO2_INST_LDFLAGS;
}

#endif

std::string xercesc_enabled() {
#ifdef AKAXISO2_USE_XERCESC
	    return "enabled";
#else
	    return "disabled";
#endif
}

std::string expat_enabled() {
#ifdef AKAXISO2_USE_EXPAT
	    return "enabled";
#else
	    return "disabled";
#endif
}

std::string iconv_enabled() {
#ifdef AKAXISO2_USE_ICONV
	    return "enabled";
#else
	    return "disabled";
#endif
}

std::string icu_enabled() {
#ifdef AKAXISO2_USE_ICU
	    return "enabled";
#else
	    return "disabled";
#endif
}

static void show_xml_parser_libraries() {
  std::cout << "XML parser configuration." << std::endl
	    << " Xerces-C++ : " << xercesc_enabled() << std::endl
	    << " expat      : " << expat_enabled() << std::endl;
}

static void show_transcoder_libraries() {
  std::cout << "transcoder configuration." << std::endl
	    << " iconv  : " << iconv_enabled() << std::endl
	    << " ICU    : " << icu_enabled() << std::endl;

#ifdef _WIN32
  if (aka::win32_encodings::is_mlang_available()) {
    std::cout << " MLang  : enabled" << std::endl;
    std::cout << " Win32  : disabled" << std::endl;
  }
  else {
    std::cout << " MLang  : disabled" << std::endl;
    std::cout << " Win32  : enabled" << std::endl;
  }
#endif
}

void show_version() {
  std::cout << "akaxiso2 version : " << aka::version() << std::endl
	    << "compiler         : " << aka::compiler() << std::endl;
}



void show_all() {
  std::cout 

#ifdef HAVE_CONFIG_H
    << "variables:" << std::endl
    << " INCLUDES=" << includes() << std::endl
    << " LDFLAGS=" << ldflags() << std::endl
    << " CFLAGS=" << cflags() << std::endl
    << " CXXFLAGS=" << cxxflags() << std::endl
#endif
    << "default XML parser:" << std::endl
    << " " << aka::default_parser_name() << std::endl
    << "default transcoder:" << std::endl
    << " " << aka::default_transcoder_name() << std::endl
    << "encodings:" << std::endl
    << " LCP      " << aka::get_default_encoding() << std::endl
    << " internal " << internal_encoding() << std::endl;
  show_xml_parser_libraries();
  show_transcoder_libraries();
}



int main(const int argc, char* argv[]) {

  aka::initialize();

  const char optstr[] = "ilCXeExtvh";
  char ch;
  while ((ch = getopt(argc, argv, optstr)) != -1) {
    switch (ch) {
    case 'e' : {
      std::cout << aka::get_default_encoding() << std::endl;
      break;
    }
    case 'E' : {
      std::cout << internal_encoding() << std::endl;
      break;
    }
    case 'x' : {
      show_xml_parser_libraries();
      break;
    }
    case 't' : {
      show_transcoder_libraries();
      break;
    }
    case 'h': {
      usage();
      break;
    }
    case 'H': {
      std::cout << internal_encoding() << std::endl;
      break;
    }
#if defined(HAVE_CONFIG_H)
    case 'C' : {
      std::cout << cflags() << std::endl;
      break;
    }
    case 'X' : {
      std::cout << cxxflags() << std::endl;
      break;
    }
    case 'i' : {
      std::cout << includes() << std::endl;
      break;
    }
    case 'l' : {
      std::cout << ldflags() << std::endl;
      break;
    }
#endif
    case 'v': {
      show_version();
      break;
    }
    case '?':
    default:
      std::cerr << "Unknown option, " << ch << "." << std::endl;
      usage();
      break;
    }
  }

  if (argc == 1) {
    show_all();
  }
  else if (optind != argc) {
    usage();
  }

  aka::uninitialize();
  return 0;
} 
