// Lexer.h
// (c) 2004-2006 exeal

#ifndef LEXER_H_
#define LEXER_H_
#include "UnicodeUtils.h"
#include <list>
#include <set>
#include <map>
#include <algorithm>	// std::lower_bound


namespace Ascension {

	/// ľ`
	enum NumberFormat {
		NF_NUMERAL_FOLLOWED_BY_ALPHANUMERAL,	///< ̌ɃAt@xbgAA܂͏_ ()
		NF_CPLUSPLUS,							///< C++ le
		NF_PERL,								///< Perl 5 le
		NF_RUBY,								///< Ruby 1.8 le
		NF_VBSCRIPT,							///< VBScript 5.6 le
		NF_JAVASCRIPT_15,						///< JavaScript 1.5 le
		NF_JAVASCRIPT_20,						///< JavaScript 2.0 le
		NF_COUNT,
	};

	/// ߂̋K
	typedef uchar AnnotationConstraint;
	const AnnotationConstraint AC_NONE				= 0x00;	///< 
	const AnnotationConstraint AC_ONLYSTARTOFLINE	= 0x01;	///< ŝ
	const AnnotationConstraint AC_ONLYFIRSTCHAR		= 0x02;	///< 󔒗ޕȊO̍ŏ̃g[N̂


	/**
	 *	@brief URI ̌A
	 *
	 *	{ÃNX RFC 3986 y RFC 3987 Œ`\ɏ]ׂA
	 *	݂͊ȈՓIȉ͂sȂ
	 */
	class URIDetector : public Manah::SelfAssertable {
	public:
		static const char_t*	eatMailAddress(const char_t* first, const char_t* last, bool asIRI);
		static const char_t*	eatURL(const char_t* first, const char_t* last, bool asIRI);
		static void				setSchemes(const std::set<string_t>& schemes);
	};


	/// g[N
	/// @see Tokens
	class Token : public Manah::SelfAssertable, public Manah::UseMemoryPool<Token> {
		// ^Ȃ
	public:
		typedef short Cookie;	///< L[[hARgȂǕ݂ނ̃g[Nʂl
		static const Cookie NULL_COOKIE;	///< ʂȃNbL[l
		/// g[N̎
		/// @see EmphaticTextType
		enum Type {
			FIRST,
			WHITESPACE = FIRST,	///< 󔒗ޕ
			TAB,				///< ^u
			KEYWORD,			///< L[[h
			ANNOTATION,			///<  (Rg)
			OPERATOR,			///< Zq
			IDENTIFIER,			///< ʎq
			NUMERAL,			///< 
			NUMBER,				///< l
			SINGLE_QUOTATION,	///< dp
			DOUBLE_QUOTATION,	///< dp
			OTHER_QUOTATION,	///< ̑ Unicode p
			ASCII_CONTROL,		///< ASCII 䕶
			UNICODE_CONTROL,	///< Unicode 䕶
			UNSPECIFIED,		///< 
			COUNT
		};

		// RXgN^
	public:
		/// ftHgRXgN^
		Token() throw() {}
		/// RXgN^
		Token(length_t index, Type type, Cookie cookie) throw() : index_(index), type_((type << 12) | cookie) {}

		// \bh
	public:
		/// g[NɊ֘AtꂽNbL[lԂ (RgAL[[ĥݗL)
		Cookie getCookie() const throw() {assertValid(); return type_ & 0x0FFF;}
		/// ͕̒ł̐擪̈ʒuԂ
		length_t getIndex() const throw() {assertValid(); return index_;}
		/// g[N̎ނԂ
		Type getType() const throw() {assertValid(); return static_cast<Type>(type_ >> 12);}
	private:
		void setCookie(Cookie cookie) {assertValid(); type_ &= 0xF000; type_ |= 0x0FFF & cookie;}
		void setType(Type type) {assertValid(); type_ &= 0x0FFF; type_ |= (type & 0x000F) << 12;}

		// f[^o
	private:
		length_t	index_;	///< sł̈ʒu
		ushort		type_;	///< 4rbg (Token::Type)A12rbgʒl (Token::Cookie)
		friend class Lexer;
	};

	/// g[N̔z
	struct Tokens {
		std::size_t	count;	///< g[Ň
		Token*		array;	///< z

		/// RXgN^
		Tokens() : count(0), array(0) {}
		/// fXgN^
		~Tokens() {delete[] array;}
	};


	/**
	 *	@brief ͊
	 *
	 *	L[[h̑啶ʂȂꍇ̃P[XtHfBO Unicode 4.1 ɂ͏]ĂȂB
	 *
	 *	@c addXxxx œo^g[N̗D揇ʂ͕񒷂̒̂ȂB
	 */
	class Lexer : public Manah::SelfAssertable, public Manah::Noncopyable {
	public:
		/// ͊̃Cxgnh
		class IEventListener {
		public:
			/// fXgN^
			virtual ~IEventListener() {}
			/**
			*	@brief L[[hRgǉꂽ
			*
			*	̌ onLexerChanged Ăяo
			*	@param type		g[N̎
			*	@param cookie	g[ÑNbL[
			*/
			virtual	void onLexerAddedIdentifiedToken(Token::Type type, Token::Cookie cookie) = 0;
			/// ͂̋Kς
			virtual void onLexerChanged() = 0;
			/// ͂̋KSč폜ꂽ
			virtual void onLexerCleared() = 0;
			/**
			*	@brief	L[[hRg폜ꂽ
			*
			*	̌ onLexerChanged Ăяo
			*	@param type		g[N̎
			*	@param cookie	g[ÑNbL[
			*/
			virtual	void onLexerRemovedIdentifiedToken(Token::Type type, Token::Cookie cookie) = 0;
		};

		// RXgN^
	public:
		Lexer(IEventListener* eventListener);
		virtual ~Lexer();

		// \bh
	public:
		/*  */
		Token::Cookie	addMultilineAnnotation(const string_t& startDelimiter,
							const string_t& endDelimiter, AnnotationConstraint constraint = AC_NONE);
		Token::Cookie	addSinglelineAnnotation(const string_t& startDelimiter, AnnotationConstraint constraint = AC_NONE);
		Token::Cookie	addSinglelineAnnotation(const string_t& startDelimiter,
							const string_t& endDelimiter, AnnotationConstraint constraint = AC_NONE);
		Token::Cookie	addKeywords(const std::set<string_t>& keywords);
		void			enableBackSolidusEscape(bool enable);
		void			enableToken(Token::Type type, bool enable);
		void			enableUnicodeAlphabets(bool enable);
		void			enableUnicodeWhiteSpaces(bool enable);
		void			freeze() throw();
		bool			getBracketTraits(char_t bracketChar, char_t& pairChar, bool& opener) const;
		void			getKeywords(string_t* keywords) const;
		NumberFormat	getNumberFormat() const;
		void			ignoreCase(bool ignore);
		bool			isBackSolidusEscapeEnabled() const throw();
		bool			isCaseSensitive() const throw();
		bool			isFreezed() const throw();
		bool			isTokenEnabled(Token::Type type) const throw();
		bool			isUnicodeAlphabetsEnabled() const throw();
		bool			isUnicodeWhiteSpacesEnabled() const throw();
		void			removeAll();
		void			removeIdentifiedToken(Token::Cookie nCookie);
		void			reset();
		void			setAdditionalAlphabets(const char_t* first, const char_t* last);
		void			setAdditionalAlphabets(const CodePoint* first, const CodePoint* last);
		void			setBrackets(const char_t* first);
		void			setNumberFormat(NumberFormat format);
		void			setOperators(const std::set<string_t>& operators);
		void			unfreeze();

		/*  */
		void			parse(const char_t* first, const char_t* last, Token::Cookie& cookie, std::list<Token>& tokens) const;
		void			parse(const string_t& text, Token::Cookie& cookie, std::list<Token>& tokens) const;
		Token::Cookie	parseMultilineAnnotation(const char_t* first, const char_t* last, Token::Cookie cookie) const;
		Token::Cookie	parseMultilineAnnotation(const string_t& text, Token::Cookie cookie) const;

		/*  */
		static CodePoint	getQuotationCloser(CodePoint opener) throw();
		static bool			isASCIIControl(CodePoint cp) throw();
		static bool			isDigit(CodePoint cp) throw();
		bool				isIdentifierContinueCodePoint(CodePoint cp) const;
		bool				isIdentifierStartCodePoint(CodePoint cp) const;
		static bool			isUnicodeControl(CodePoint cp) throw();
		bool				isWhiteSpace(CodePoint cp, bool includeTab) const throw();

		/* g[N̐؂o */
		static const char_t*	eatASCIIControls(const char_t* first, const char_t* last);
		const char_t*			eatIdentifier(const char_t* first, const char_t* last) const;
		bool					eatKeyword(const char_t* first, const char_t* last, Token::Cookie& cookie) const;
		const char_t*			eatOperators(const char_t* first, const char_t* last) const;
		static const char_t*	eatQuotation(const char_t* first, const char_t* last, bool escapeByBackSolidus);
		const char_t*			eatMultilineAnnotation(const char_t* first, const char_t* last,
									AnnotationConstraint constraint, Token::Cookie& cookie, bool& continued) const;
		static const char_t*	eatNumerals(const char_t* first, const char_t* last);
		const char_t*			eatNumbers(const char_t* first, const char_t* last) const;
		const char_t*			eatSinglelineAnnotation(const char_t* first, const char_t* last,
									AnnotationConstraint constraint, Token::Cookie& cookie) const;
		static bool				eatUnicodeControls(const char_t* first, const char_t* last);
		const char_t*			eatWhiteSpaces(const char_t* first, const char_t* last, bool includeTab) const;

		/* [eBeB */
		static void	getAsciiControlSubstitutionGlyph(uchar ch, char_t* glyphs) throw();

	private:
		void			clearKeywords();
		template<NumberFormat format>
		const char_t*	doEatNumbers(const char_t* first, const char_t* last) const;
		void			notifyChange();

		// Jo萔
	public:
		static const char_t	ASCII_OPENERS[];
		static const char_t	UNICODE_OPENERS[];

		// Jf[^^
	private:
		/// L[[h𔻒肷邽߂̃nbVe[u
		class HashTable {
		public:
			HashTable(const std::set<string_t>& data, bool caseSensitive);
			~HashTable();
			bool find(const char_t* first, const char_t* last) const;
			static ulong getHashCode(const char_t* first, const char_t* last);
		private:
			struct Entry {
				string_t	str_;
				Entry*		next_;
				Entry(const string_t& str) : str_(str) {}
				~Entry() {delete next_;}
			};
			Entry**				entries_;
			const std::size_t	size_;
			std::size_t			maxLength_;	// ŒL[[h
			const bool			caseSensitive_;
		};
		/// sŏIPs
		struct SinglelineAnnotationEndedByBreak {
			string_t	startDelimiter;			///< Jn
			AnnotationConstraint	constraint;	///< 
		};

		/// wf~^ŏIPs
		struct SinglelineAnnotationEndedByDelimiter {
			string_t	startDelimiter;			///< Jn
			string_t	endDelimiter;			///< I
			AnnotationConstraint	constraint;	///<  (Jnf~^ɂ̂݉e)
		};

		/// s
		struct MultilineAnnotation {
			string_t	startDelimiter;			///< Jn
			string_t	endDelimiter;			///< I
			AnnotationConstraint	constraint;	///<  (Jnf~^ɂ̂݉e)
		};

		typedef std::map<Token::Cookie, HashTable*>								KeywordsMap;
		typedef std::map<Token::Cookie, SinglelineAnnotationEndedByBreak>		SAnnotationBMap;
		typedef std::map<Token::Cookie, SinglelineAnnotationEndedByDelimiter>	SAnnotationDMap;
		typedef std::map<Token::Cookie, MultilineAnnotation>					MAnnotationMap;
		typedef std::set<string_t, std::greater<string_t> >						OperatorSet;
		typedef std::map<char_t, OperatorSet>									OperatorMap;

		// f[^o
	private:
		bool					freezed_;
		bool					caseSensitive_;
		bool					escapeByBackSolidus_;
		bool					enableUnicodeAlphabets_;
		bool					enableUnicodeWhiteSpaces_;
		bool					enabledTokenTypes_[Token::COUNT];
		NumberFormat			numberFormat_;
		char_t*					brackets_;
		static Token::Cookie	nextCookie_;
		IEventListener*			eventListener_;
		std::set<CodePoint>		additionalAlphabets_;	// At@xbgƂ݂ȂR[h|Cg

		KeywordsMap		keywords_;					// L[[hQ
		SAnnotationBMap	singlelineAnnotationBs_;	// Ps
		SAnnotationDMap	singlelineAnnotationDs_;	// Ps
		MAnnotationMap	multilineAnnotations_;		// s
		OperatorMap		operators_;					// Zq
	};


#define ENABLE_SWITCH(a, b)	\
	do {					\
		if(a != b) {		\
			a = b;			\
			notifyChange();	\
		}					\
	} while(false)

	/**
	 *	ASCII 䕶𒲂ׂB^u䕶Ƃ݂Ȃ
	 *	@param first, last	ׂ镶
	 *	@return				ASCII 䕶̏I_
	 */
	inline const char_t* Lexer::eatASCIIControls(const char_t* first, const char_t* last) {
		while(first < last && isASCIIControl(*first))
			++first;
		return first;
	}

	/**
	 *	ʎq𒲂ׂ
	 *	@param first, last	ׂ镶
	 *	@return				ʎq̏I_
	 *	@see				Lexer::isIdentifierContinueChar, Lexer::isIdentifierStartChar
	 */
	inline const char_t* Lexer::eatIdentifier(const char_t* first, const char_t* last) const {
		assertValid();

		const char_t* const originalFirst = first;
		CodePoint cp;
		while(first < last) {
			if(UTF16Surrogates::isHighSurrogate(*first)
					&& first < last - 1
					&& UTF16Surrogates::isLowSurrogate(first[1]))
				cp = UTF16Surrogates::decode(first, last - first);
			else
				cp = *first;
			if((first == originalFirst && isIdentifierStartCodePoint(cp))
					|| isIdentifierContinueCodePoint(cp)) {
				if(cp >= 0x010000)
					++first;
				++first;
				continue;
			}
			return first;
		}
		return last;
	}

	/**
	 *	𒲂ׂ
	 *	@param first, last	ׂ镶
	 *	@return				̏I[
	 *	@see				Lexer::isDigitCodePoint
	 */
	inline const char_t* Lexer::eatNumerals(const char_t* first, const char_t* last) {
		CodePoint cp;
		while(first < last) {
			if(UTF16Surrogates::isHighSurrogate(*first)
					&& first < last - 1
					&& UTF16Surrogates::isLowSurrogate(first[1]))
				cp = UTF16Surrogates::decode(first, last - first);
			else
				cp = *first;
			if(Lexer::isDigit(cp)) {
				if(cp >= 0x010000)
					++first;
				++first;
				continue;
			}
			return first;
		}
		return last;
	}

	/**
	 *	p𒲂ׂ (p̂܂܂)
	 *	@param first, last			ׂ镶
	 *	@param escapeByBackSolidus	'\' ŕpGXP[v邩
	 *	@return						p̏I[BĂȂ @a last 𓯂lԂ
	 */
	inline const char_t* Lexer::eatQuotation(const char_t* first, const char_t* last, bool escapeByBackSolidus) {
		const char_t closer = static_cast<char_t>(getQuotationCloser(*first));	//  BMP ̂...

		if(closer == 0xFFFF)
			return first;
		while(++first < last) {
			if(*first == L'\\' && escapeByBackSolidus)	// ͖̕
				++first; 
			else if(*first == closer)
				return first + 1;
		}
		return last;
	}

	/**
	 *	Unicode 䕶𒲂ׂ
	 *	@param first, last	ׂ镶
	 *	@return				Unicode 䕶̏ꍇ true Ԃ
	 */
	inline bool Lexer::eatUnicodeControls(const char_t* first, const char_t* last) {
		return isUnicodeControl(UTF16Surrogates::decode(first, last - first));}

	/**
	 *	󔒗ޕ𒲂ׂ
	 *	@param first, last	ׂ镶
	 *	@param includeTab	^u󔒕Ƃ݂Ȃꍇ true
	 *	@return				󔒗ޕ̏I[
	 */
	inline const char_t* Lexer::eatWhiteSpaces(const char_t* first, const char_t* last, bool includeTab) const {
		assertValid();
		while(first < last && isWhiteSpace(*first, includeTab))	// BMP ̂...
			++first;
		return first;
	}

	/// IpobNXbVŃGXP[v\ݒ肷
	inline void Lexer::enableBackSolidusEscape(bool enable) {assertValid(); ENABLE_SWITCH(escapeByBackSolidus_, enable);}

	/**
	 *	g[NL/ɂ
	 *	@param type		g[N̎
	 *	@param enable	Lɂꍇ true
	 */
	inline void Lexer::enableToken(Token::Type type, bool enable) {
		assertValid();
		if(type >= Token::COUNT)
			return;
		ENABLE_SWITCH(enabledTokenTypes_[type], enable);
	}

	/// Unicode At@xbgAt@xbgƂĎgp邩ݒ肷
	inline void Lexer::enableUnicodeAlphabets(bool enable) {assertValid(); ENABLE_SWITCH(enableUnicodeAlphabets_, enable);}

	/// Unicode 󔒗ޕ󔒗ޕƂĎgp邩ݒ肷
	inline void Lexer::enableUnicodeWhiteSpaces(bool enable) {assertValid(); ENABLE_SWITCH(enableUnicodeWhiteSpaces_, enable);}

	/// ݒ肪ύXĂCxgnhɒʒmȂ悤ɂ
	inline void Lexer::freeze() throw() {assertValid(); freezed_ = true;}

	/**
	 *	ASCII 䕶̑փeLXgԂ
	 *	@param ch		䕶
	 *	@param glyphs	[out] փeLXgBK2̃KvBI[ null ͕tȂ
	 */
	inline void Lexer::getAsciiControlSubstitutionGlyph(uchar ch, char_t* glyphs) throw() {
		glyphs[0] = L'^'; glyphs[1] = ch + ((ch != 0x7F) ? 0x40 : -0x40);}

	/// ͂Ŏgl`Ԃ
	inline NumberFormat Lexer::getNumberFormat() const throw() {assertValid(); return numberFormat_;}

	/// Jnp @a opener ɑΉIpԂB@a opener płȂꍇ U+FFFF Ԃ
	inline CodePoint Lexer::getQuotationCloser(CodePoint opener) throw() {
#if ASCENSION_UNICODE_VERSION != 0x0410
#error This code is based on old version of Unicode.
#endif
		switch(opener) {
		case 0x0022:	return 0x0022;	// Quotation Mark
		case 0x0027:	return 0x0027;	// Apostrophe
		case 0x00AB:	return 0x00BB;	// Left-Pointing Double Angle Quotation Mark, Right-...
		case 0x2018:	return 0x2019;	// Left Single Quotation Mark, Right ...
//		case 0x201A:	return 0x????;	// Single Low-9 Quotation Mark
//		case 0x201B:	return 0x????;	// Single High-Reserved-9 Quotation Mark
		case 0x201C:	return 0x201D;	// Left Double Quotation Mark, Right ...
//		case 0x201E:	return 0x????;	// Double Low-9 Quotation Mark
//		case 0x201F:	return 0x????;	// Double High-Reversed-9 Quotation Mark
		case 0x2039:	return 0x203A;	// Single Left-Pointing Angle Quotation Mark, Single Right-...
		case 0x300C:	return 0x300D;	// Left Corner Bracket, Right ...
		case 0x300E:	return 0x300F;	// Left White Corner Bracket, Right ...
//		case 0x301D:	return 0x301E or 0x301F;	break;
		case 0xFE41:	return 0xFE42;	// Presentation Form For Vertical Left Corner Bracket, ... Right ...
		case 0xFE43:	return 0xFE44;	// Presentation Form For Vertical Left White Corner Bracket, ... Right ...
		case 0xFF02:	return 0xFF02;	// Fullwidth Quotation Mark
		case 0xFF07:	return 0xFF07;	// Fullwidth Apostrophe
		case 0xFF62:	return 0xFF63;	// Halfwidth Left Corner Bracket, ... Right ...
		default:		return 0xFFFF;	// 
		}
	}

	/// ASCII 䕶𔻒肷B^u䕶Ƃ݂Ȃ
	inline bool Lexer::isASCIIControl(CodePoint cp) throw() {return cp < 0x20 || cp == 0x7F;}

	/// IpobNXbVŃGXP[v\Ԃ
	inline bool Lexer::isBackSolidusEscapeEnabled() const throw() {assertValid(); return escapeByBackSolidus_;}

	/// 𔻒肷
	inline bool Lexer::isDigit(CodePoint cp) throw() {return CharProperty::getGeneralCategory(cp) == CharProperty::GC_NUMBER_DECIMAL_DIGIT;}

	/// L[[hő啶ʂ邩Ԃ
	inline bool Lexer::isCaseSensitive() const throw() {assertValid(); return caseSensitive_;}

	/// ԂԂ
	inline bool Lexer::isFreezed() const throw() {assertValid(); return freezed_;}

	/// g[NLԂ
	inline bool Lexer::isTokenEnabled(Token::Type type) const throw() {assertValid(); return enabledTokenTypes_[type];}

	/// Unicode At@xbgAt@xbgƂĎgp邩Ԃ
	inline bool Lexer::isUnicodeAlphabetsEnabled() const throw() {assertValid(); return enableUnicodeAlphabets_;}

	/// Unicode 䕶̔
	inline bool Lexer::isUnicodeControl(CodePoint cp) throw() {
		const CharProperty::GeneralCategory	gc = CharProperty::getGeneralCategory(cp);
		return cp == CharProperty::GC_OTHER_CONTROL || cp == CharProperty::GC_OTHER_FORMAT;
	}

	/// Unicode 󔒗ޕ󔒗ޕƂĎgp邩Ԃ
	inline bool Lexer::isUnicodeWhiteSpacesEnabled() const throw() {assertValid(); return enableUnicodeWhiteSpaces_;}

	/// 󔒗ޕ̔
	inline bool Lexer::isWhiteSpace(CodePoint cp, bool includeTab) const throw() {
		if(!enableUnicodeWhiteSpaces_)	// Unicode 󔒗ޕFȂꍇ
			return cp == L' ' || (includeTab && cp == L'\t');
		else	// Unicode 󔒗ޕFꍇ
			return CharProperty::getGeneralCategory(cp) == CharProperty::GC_SEPARATOR_SPACE || includeTab && cp == L'\t';
	}

	/// ύXCxgXiɒʒm
	inline void Lexer::notifyChange() {
		assertValid();
		if(!freezed_ && eventListener_ != 0)
			eventListener_->onLexerChanged();
	}

	/// @see Lexer::parse
	inline void Lexer::parse(const string_t& text, Token::Cookie& cookie, std::list<Token>& tokens) const {
		parse(text.data(), text.data() + text.length(), cookie, tokens);}

	/// @see Lexer::parseMultilineAnnotation
	inline Token::Cookie Lexer::parseMultilineAnnotation(const string_t& text, Token::Cookie cookie) const {
		return parseMultilineAnnotation(text.data(), text.data() + text.length(), cookie);}

	/// ͂Ŏgl`ݒ肷
	/// @throw std::invalid_argument	l`sȂƂX[
	inline void Lexer::setNumberFormat(NumberFormat format) {
		assertValid();
		if(format >= NF_COUNT)
			throw std::invalid_argument("");
		ENABLE_SWITCH(numberFormat_, format);
	}

	/// Lexer::freeze ɂ铀ACxgnhɒʒm
	inline void Lexer::unfreeze() {assertValid(); ENABLE_SWITCH(freezed_, false);}

	/// RXgN^
	inline Lexer::HashTable::HashTable(const std::set<string_t>& data, bool caseSensitive)
			: size_(data.size()), maxLength_(0), caseSensitive_(caseSensitive) {
		entries_ = new Entry*[size_];
		std::fill<Entry**, Entry*>(entries_, entries_ + size_, 0);

		for(std::set<string_t>::const_iterator it = data.begin(); it != data.end(); ++it) {
			Entry* newEntry;

			if(caseSensitive_)
				newEntry = new Entry(*it);
			else {
				char_t* folded = CharacterFolder::foldCase(it->data(), it->data() + it->length());
				newEntry = new Entry(string_t(folded, it->length()));
				delete[] folded;
			}

			const std::size_t h = getHashCode(newEntry->str_.data(), newEntry->str_.data() + newEntry->str_.length());
			if(it->length() > maxLength_)
				maxLength_ = it->length();
			newEntry->next_ = (entries_[h % size_] != 0) ? entries_[h % size_] : 0;
			entries_[h % size_] = newEntry;
		}
	}

	/// fXgN^
	inline Lexer::HashTable::~HashTable() {
		for(std::size_t i = 0; i < size_; ++i)
			delete entries_[i];
		delete[] entries_;
	}

	/// 
	inline bool Lexer::HashTable::find(const char_t* first, const char_t* last) const {
		if(static_cast<std::size_t>(last - first) > maxLength_)
			return false;

		const char_t* const	folded = caseSensitive_ ? first : CharacterFolder::foldCase(first, last);
		const std::size_t	h = getHashCode(folded, folded + (last - first));
		Entry*				entry = entries_[h % size_];
		bool				found = false;

		while(entry != 0) {
			if(entry->str_.length() == last - first
#ifdef _DEBUG
					&& std::wcsncmp(entry->str_.data(), folded, last - first) == 0) {
#else
					&& std::memcmp(entry->str_.data(), folded, sizeof(char_t) * (last - first)) == 0) {
#endif /* _DEBUG */
				found = true;
				break;
			}
			entry = entry->next_;
		}
		if(!caseSensitive_)
			delete[] folded;
		return found;
	}

	/// nbVl𓾂
	inline ulong Lexer::HashTable::getHashCode(const char_t* first, const char_t* last) {
		ulong h = 0;
		while(first < last) {
			h *= 2;
			h += *first++;
		}
		return h;
	}

#undef ENABLE_SWITCH

} // namespace Ascension

#endif /* LEXER_H_ */

/* [EOF] */