// TextSearcher.h
// (c) 2004-2006 exeal

#ifndef TEXT_SEARCHER_H_
#define TEXT_SEARCHER_H_
#include "UnicodeUtils.h"
#include "../../Manah/Object.hpp"
#include <stdexcept>
#include <stack>
#include <bitset>


namespace boost {
	template<class charT> class cpp_regex_traits;
	template<class charT, class traits> class basic_regex;
	template<class BidirectionalIterator> struct sub_match;
	template<class BidirectionalIterator, class Allocator> class match_results;
}

namespace Ascension {

	namespace Private {
		// 2̃NX TextSearcher ANZX邽߂ɖOt̖OԂɓĂ
#ifndef ASCENSION_NO_REGEX
		class AscensionRegexTraits;
		typedef boost::basic_regex<char_t, AscensionRegexTraits> AscensionRegex;
#endif /* !ASCENSION_NO_REGEX */
		class BMSearcher;
	} // namespace Private

	/// tH[fBOIvṼrbg
	/// @see UTR #30: Character Foldings (http://www.unicode.org/reports/tr30/)
	namespace FoldingOptions {
		enum {
			// tH[fBO
			FOLDING_START,
			ACCENT_REMOVAL = FOLDING_START,	///< ANZgL̏
			CANONICAL_DUPLICATES,			///< do^R[h|Cg
			DASHES,							///< _bV (Pd)  U+002D: Hyphen-Minus 
			GREEK_LETTERFORMS,				///< MVLMV
			HEBREW_ALTERNATES,				///< wuC֕ (Sp) wuC
			JAMO,							///< nO݊`
			MATH_SYMBOL,					///< wL <tHg݊> }bsO
			NATIVE_DIGIT,					///<  (Nd)  ASCII ̐
			NOBREAK,						///< <s֎~݊> }bsO
			OVERLINE,						///<  U+203E: Overline 
			POSITIONAL_FORMS,				///< <ꓪ`Aꒆ`Aꖖ`A`> ݊}bsO
			SMALL_FORMS,					///< <`> ݊}bsO
			SPACE,							///< 󔒗ޕ (Zs)  U+0020: Space 
			SPACING_ACCENTS,				///< 肠ANZgL𑗂薳ANZgL (񐄏)
			SUBSCRIPT,						///< <t> ݊}bsO
			SYMBOL,							///< L𕶎 (񐄏)
			UNDERLINE,						///<  U+005F: Underline 
			VERTICAL_FORMS,					///< <c> ݊}bsO
			FOLDING_END,

			// WJIvV
			MULTIGRAPH_EXPANSION_START = FOLDING_END,
			EXPAND_CIRCLED_SYMBOLS = MULTIGRAPH_EXPANSION_START,	///< <ەt> ݊}bsO
			EXPAND_DOTTED,											///< hbgt (U+2488..249B)
			EXPAND_ELLIPSIS,										///< ȗ (U+2024..2026)
			EXPAND_FRACTION,										///< <> ݊}bsO
			EXPAND_INTEGRAL,										///< ϕL (U+222C..222D, U+222F..2230)
			EXPAND_LIGATURE,										///< ̑̍
			EXPAND_PARENTHESIZED,									///< ʕt̕
			EXPAND_PRIMES,											///< vC (U+2033..2034, U+2036..2037)
			EXPAND_ROMAN_NUMERALS,									///< [} (U+2160..2183)
			EXPAND_SQUARED,											///< <gݕ> ݊}bsO
			EXPAND_SQUARED_UNMARKED,								///< <gݕ> tĂȂ <gݕ> ݊}bsOs
			EXPAND_DIGRAPH,											///< d
			EXPAND_OTHER_MULTIGRAPHS,								///< d
			MULTIGRAPH_EXPANSION_END,

			// ătH[fBO
			PROVISIONAL_FOLDING_START = MULTIGRAPH_EXPANSION_END,
			DIACRITIC_REMOVAL = PROVISIONAL_FOLDING_START,	///< 敪L̏
			HAN_RADICAL,									///< 񂩂犿
			KANA,											///< ƕЉ
			LETTER_FORMS,									///< ώ핶ƌ^
			SIMPLIFIED_HAN,									///< ȑ̎
			SUPERSCRIPT,									///< <t> ݊}bsO
			SUZHOU_NUMERAL,									///< hB () Ɗ
			WIDTH,											///< Sp`Ɣp`
			PROVISIONAL_FOLDING_END,

			// 镶
			CHARACTER_SKIP_START = PROVISIONAL_FOLDING_END,
			SKIP_PUNCTUATIONS = CHARACTER_SKIP_START,	///< ؂蕶
			SKIP_SYMBOLS,								///< L
			SKIP_WHITESPACES,							///< 󔒗ޕ
			SKIP_DIACRITICS,							///< 敪L
			SKIP_VOWELS,								///< ꉹ
			SKIP_KASHIDA,								///< JV_
			SKIP_CONTROLS,								///< 䕶
			CHARACTER_SKIP_END,
			FO_COUNT = CHARACTER_SKIP_END
		};
	} // namespace FoldingOptions

#ifndef ASCENSION_NO_REGEX
	/// G[
	class SearchError {
	public:
		/// G[R[h
		enum Code {
			SUCCEEDED,	///< 
			NO_MATCH,	///< }b`Ȃ
			// ȉAsȐK\p^[̃G[ (boost::regex_constants::error_type 蕡)
			BADPATTERN_BACK_REFERENCE_TO_NON_EXSISTANT,	///< ݂Ȃւ̌Q
			BADPATTERN_INVALID_BRACE_CONTENTS,			///< ʂ̒gs
			BADPATTERN_INVALID_CHARACTER_CLASS_NAME,	///< sȕNX
			BADPATTERN_INVALID_CHARACTER_RANGE,			///< sȕ͈
			BADPATTERN_INVALID_CHARACTER_SET,			///< sȕZbg
			BADPATTERN_INVALID_COLLATING_ELEMENT,		///< sȏƍvf
			BADPATTERN_INVALID_OR_TRAILING_ESCAPE,		///< sȃGXP[v
			BADPATTERN_INVALID_REPEAT,					///< sȌJԂ
			BADPATTERN_MISMATCHED_PAREN,				///< Ήۊʂ
			BADPATTERN_MISMATCHED_BRACE,				///< Ή钆ʂ
			BADPATTERN_TOO_COMPLEX,						///< p^[GȂ߃RpCłȂ
			BADPATTERN_OUT_OF_MEMORY,					///< s
			BADPATTERN_OUT_OF_STACK_SPACE,				///< X^bNԂ̌͊
			BADPATTERN_UNKNOWN_ERROR,					///< p^[Ȃڍוs
			// ȉAK\sG[
			RUNTIME_ERROR_FOR_COMPLEXITY,	///< p^[GȂ߃p^[pb`Ɏs
#ifndef ASCENSION_NO_MIGEMO
			// ȉAMigemo G[
			MIGEMO_ERROR,	///< Migemo K\p^[̍쐬Ɏs
#endif /* !ASCENSION_NO_MIGEMO */
		};

		/// RXgN^
		SearchError() throw() {}
		/// RXgN^
		explicit SearchError(Code code, length_t position = -1) throw() : code_(code), position_(position) {}
		/// G[R[h SUCCEEDED ł true Ԃ
		operator bool() const throw() {return code_ == SUCCEEDED;}
		/// G[R[hԂ
		Code getCode() const throw() {return code_;}
		/// p^[擪̈ʒuԂBʒusȏꍇ -1 Ԃ
		length_t getPosition() const throw() {return position_;}
		/// K\p^[ԈĂ邱ƂɂG[
		bool isBadRegexError() const throw() {return code_ >= BADPATTERN_BACK_REFERENCE_TO_NON_EXSISTANT && code_ <= BADPATTERN_UNKNOWN_ERROR;}
		/// vIȃG[
		bool isFatalError() const throw() {return code_ != SUCCEEDED && code_ != NO_MATCH;}

	private:
		Code code_;
		length_t position_;
	};
	typedef SearchError	SearchResult;
#else
	typedef bool		SearchResult;
#endif /* !ASCENSION_NO_REGEX */

/*	/// {BtO (ꎋ镶A\LȂǁBMS Word ̃pN)B
	/// ̃tO͐K\ɂ͎gpłȂ
	typedef ushort	JapaneseFuzzySearchFlag;
	const JapaneseFuzzySearchFlag
		JFSF_KANATYPE					= 0x0001,	///< /Љ
		JFSF_YOUON_SOKUON				= 0x0002,	///< X/
		JFSF_MINUS_PROLONGEDMARK_DASH	= 0x0004,	///< }CiX//_bV
		JFSF_ITERATIONMARK				= 0x0008,	///< JԂL
		JFSF_UNUNIFIEDKANJI				= 0x0010,	///< \L̂
		JFSF_LEGACY_MODERN_KANAFIGURE	= 0x0020,	///< ̐V/
		JFSF_PROLONGEDMARK_VOWEL		= 0x0040,	///< ƕꉹ
		JFSF_DI_JI_DU_ZU				= 0x0080,	///< a/WAd/Y
		JFSF_BA_VA_HA_FA				= 0x0100,	///< o/@An/t@
		JFSF_TSI_THI_TI_DHI_JI			= 0x0200,	///< cB/eB/`AfB/W
		JFSF_HYU_FYU_BYU_VYU			= 0x0400,	///< q/tAr/
		JFSF_SE_SYE_ZE_JE				= 0x0800,	///< Z/VFA[/WF
		JFSF_A_YA_FOLLOWING_I_E			= 0x1000,	///< CiAGiɑA/
		JFSF_KI_KU_FOLLOWEDBY_S			= 0x2000;	///< Ts̑ÕL/N
*/

	class Lexer;
	class BoundaryDetector;

	/**
	 *	@brief eLXgs
	 *
	 *	GfB^͂̃NX̃CX^XێĂA錟ɂ̃IuWFNggpB
	 *	NCAg̓GfB^ێĂIuWFNg EditView::getTextSearcher
	 *	\bhœ邱Ƃł鑼AŃCX^X쐬邱Ƃł
	 */
	class TextSearcher : public Manah::SelfAssertable, public Manah::Noncopyable {
		// ^
	public:
		/// @
		enum Type {
			LITERAL,			///< ʏ̌
#ifndef ASCENSION_NO_REGEX
			REGULAR_EXPRESSION,	///< K\
#endif /* !ASCENSION_NO_REGEX */
			WILD_CARD,			///< ChJ[h ()
#ifndef ASCENSION_NO_MIGEMO
			MIGEMO				///< Migemo
#endif /* !ASCENSION_NO_MIGEMO */
		};

		/// 啶̓ꎋ
		enum CaseSensitivity {
			CASE_SENSITIVE,		///< ʂ
			CASE_FOLD_ASCII,	///< ASCII At@xbĝ݋ʂȂ
			CASE_FOLD_SIMPLE,	///< Unicode P[XtHfBO (P)
			CASE_FOLD_FULL		///< Unicode P[XtHfBO (S)()
		};

		/// IvV
		struct Options {
			bool			wholeWord;			///< PPʂŌ
			bool			performAsUCS4;		///< UTF-32 PʂŌ
			Type			type;				///< @
			CaseSensitivity	caseSensitivity;	///< 啶̋
			std::bitset<FoldingOptions::FO_COUNT>	foldings;	///< tH[fBOIvV
#ifndef ASCENSION_NO_REGEX
			bool	unicodePropertyEnabled;	///< Unicode vpeBLɂ
#endif /* !ASCENSION_NO_REGEX */

			/// RXgN^
			Options() throw() : wholeWord(false), performAsUCS4(false), type(LITERAL), caseSensitivity(CASE_SENSITIVE)
#ifndef ASCENSION_NO_REGEX
				, unicodePropertyEnabled(true)
#endif /* !ASCENSION_NO_REGEX */
			{}
			/// Zq
			bool operator ==(const Options& rhs) const throw() {
				return wholeWord == rhs.wholeWord
					&& performAsUCS4 == rhs.performAsUCS4
					&& type == rhs.type
					&& caseSensitivity == rhs.caseSensitivity
					&& foldings == rhs.foldings
#ifndef ASCENSION_NO_REGEX
					&& unicodePropertyEnabled == rhs.unicodePropertyEnabled
#endif /* !ASCENSION_NO_REGEX */
				;
			}
			/// 񓙉Zq
			bool operator !=(const Options& rhs) const throw() {return !(*this == rhs);}
			/// tH[fBOŕ񒷂̕ςIvVݒ肳Ă邩
			bool isComplex() const throw() {
				if(foldings[FoldingOptions::ACCENT_REMOVAL] || foldings[FoldingOptions::JAMO])
					return true;
				for(int f = FoldingOptions::EXPAND_CIRCLED_SYMBOLS; f <= FoldingOptions::EXPAND_OTHER_MULTIGRAPHS; ++f) {
					if(foldings[f])
						return true;
				}
				for(int f = FoldingOptions::SKIP_PUNCTUATIONS; f <= FoldingOptions::SKIP_CONTROLS; ++f) {
					if(foldings[f])
						return true;
				}
				return false;
			}
			/// ̓ꎋsIvV1ݒ肳ĂȂ
			bool isEmpty() const throw() {return caseSensitivity == CASE_SENSITIVE && foldings.none();}
		};

		/// }b`Ώۂ̃eLXg
		/// @see TextSearcher::match, TextSearcher::replace
		struct MatchTarget {
			const char_t* first;		///< }b`̐擪
			const char_t* last;			///< }b`̖
			const char_t* entireFirst;	///< Ŝ̐擪
			const char_t* entireLast;	///< Ŝ̖
			/// ̃RXgN^
			MatchTarget() throw() {}
			/// RXgN^
			MatchTarget(const char_t* ef, const char_t* el, const char_t* f, const char_t* l) throw()
				: first(f), last(l), entireFirst(ef), entireLast(el) {assert(isNormalized());}
			/// RXgN^
			MatchTarget(const char_t* f, const char_t* l) throw()
				: first(f), last(l), entireFirst(f), entireLast(l) {assert(isNormalized());}
			/// o̒l̐𒲂ׂ
			bool isNormalized() const throw() {
				return first <= last && entireFirst <= entireLast && first >= entireFirst && last <= entireLast;
			}
		};

		/// Ώۂ̃eLXg
		/// @see TextSearcher::search
		typedef MatchTarget SearchTarget;

		/// 
		struct MatchedRange {
			const char_t* first;	///< 擪
			const char_t* last;		///< I[
		};

		// RXgN^
	public:
		TextSearcher();
		virtual ~TextSearcher();

		// \bh
	public:
		/*  */
		const string_t&	getReplaceText() const throw();
		const string_t&	getSearchText() const throw();
		const Options&	getOptions() const throw();
		bool			isChangedSinceLastSearch() const throw();
		static bool		isMigemoAvailable();
		bool			isMultilinePattern() const throw();
		static bool		isRegexAvailable() throw();
		void			setOptions(const Options& options);
		void			setReplaceText(const string_t& text);
		void			setSearchText(const string_t& text);
		/*  */
		SearchResult	match(const MatchTarget& target, const BoundaryDetector& boundary) const;
		SearchResult	replace(const MatchTarget& target, string_t& replaced, const BoundaryDetector& boundary) const;
		SearchResult	search(const SearchTarget& target, bool forward, MatchedRange& result, const BoundaryDetector& boundary) const;
	private:
		void			clearPatternCache();
#ifndef ASCENSION_NO_REGEX
		SearchError		compileRegexPattern() const throw();
#endif /* !ASCENSION_NO_REGEX */
		CodePoint		foldCharacter(CodePoint cp) const;
		template<Type type>
		SearchResult	doMatch(const MatchTarget& target, const BoundaryDetector& boundary) const;
		template<Type type>
		SearchResult	doReplace(const MatchTarget& target, string_t& replaced, const BoundaryDetector& boundary) const;
		template<Type type>
		SearchResult	doSearch(const SearchTarget& target, bool forward, MatchedRange& result, const BoundaryDetector& boundary) const;

		// f[^o
	protected:
		string_t	findWhat_;		// 
		string_t	replaceWith_;	// u
		Options		options_;		// IvV
		bool		changedFromLast_;
		bool		multilinePattern_;

		friend class Private::BMSearcher;
		Private::BMSearcher* lastLiteralPattern_;
#ifndef ASCENSION_NO_REGEX
		friend class Private::AscensionRegexTraits;
		Ascension::Private::AscensionRegex* lastRegexPattern_;
		boost::match_results<
			const char_t*, std::allocator<
				boost::sub_match<const char_t*>
			>
		>* lastRegexMatchResults_;
#endif /* !ASCENSION_NO_REGEX */
#ifndef ASCENSION_NO_MIGEMO
		boost::basic_regex<
			char_t, boost::cpp_regex_traits<char_t>
		>* lastMigemoPattern_;
#endif /* !ASCENSION_NO_MIGEMO */
	};


	/// hLǧs
	class DocumentSearcher : public Manah::SelfAssertable {
	public:
		// RXgN^
		DocumentSearcher(const EditDoc& document, const TextSearcher& searcher, const BoundaryDetector& boundary) throw();
		// \bh
		SearchResult	replace(const TextRange& target, string_t& result) const;
		SearchResult	search(const TextRange& target, bool forward, TextRange& result) const;
	private:
		/// s}b`̂߂ɌΏۃeLXg𕡐
		class TargetDuplicate : public Manah::Noncopyable {
		public:
			TargetDuplicate(const EditDoc& document, length_t startLine, length_t lineCount);
			~TargetDuplicate() throw();
			const char_t*	getBuffer() const throw();
			length_t		getLength() const throw();
			length_t		getLineCount() const throw();
			void			getResult(const TextSearcher::MatchedRange& range, TextRange& result) const;
		private:
			char_t* buffer_;
			char_t** lineHeads_;
			const length_t startLine_;
			const length_t lineCount_;
			length_t length_;
		};
		const EditDoc& document_;
		const TextSearcher& searcher_;
		const BoundaryDetector& boundary_;
	};


	/// CN^
	class IncrementalSearcher : public Manah::SelfAssertable {
	public:
		/// CxgXi
		class IEventListener {
		public:
			/// OnPatternChanged œn錋
			enum Result {
				FOUND,		///<  (̏ꍇ̌ʂɂȂ)
				NOT_FOUND,	///< Ȃ
				BAD_REGEX,	///< sȐK͂ꂽ
				REGEX_ERROR	///< K\ɃG[
			};
			/// fXgN^
			virtual ~IEventListener() {}
			/// ~B̌ onISearchCompleted Ăяo
			virtual void onISearchAborted() = 0;
			/// I
			virtual void onISearchCompleted() = 0;
			/**
			 *	ύXꂽ
			 *	@param result Vł̌
			 */
			virtual void onISearchPatternChanged(const SearchResult& result) = 0;
			/// JnB̌ onISearchPatternChanged Ăяo
			virtual void onISearchStarted() = 0;
		};

		/// CxgXi̋
		class EventAdapter : virtual public IEventListener {
			virtual ~EventAdapter() {}
			virtual void onISearchAborted() {}
			virtual void onISearchCompleted() {}
			virtual void onISearchPatternChanged(const SearchResult&) {}
			virtual void onISearchStarted() {}
		};

		/// CN^JnĂȂƂ\O
		/// @see IncrementalSearcher
		class NotRunningException : public std::logic_error {
		public:
			/// RXgN^
			NotRunningException() : std::logic_error("Incremental search is not running") {}
		};

		/// CN^AhDłȂԂŃAhD悤ƂƂ\O
		/// @see IncrementalSearcher::undo
		class EmptyUndoBufferException : public std::logic_error {
		public:
			/// RXgN^
			EmptyUndoBufferException() : std::logic_error("Undo buffer of incremental search is empty and not undoable.") {}
		};

		// \bh
	public:
		/*  */
		bool				canUndo() const throw();
		bool				getSearchDirection() const;
		const string_t&		getSearchText() const throw();
		TextSearcher::Type	getSearchType() const;
		bool				isRunning() const throw();
		/*  */
		void			abort();
		SearchResult	addCharacter(char_t ch);
		SearchResult	addCharacter(CodePoint cp);
		SearchResult	addString(const char_t* first, const char_t* last);
		SearchResult	addString(const string_t& text);
		void			end();
		SearchResult	jumpToNextMatch(bool forward);
		void			reset();
		void			start(EditView& view, TextSearcher::Type type, bool forward, IEventListener* eventListener = 0);
		SearchResult	undo();
	private:
		SearchResult	update();
		
		// f[^o
	private:
		enum Operation {TYPE, JUMP};
		struct Status {
			TextRange	range;		// }b`ʒu
			bool		forward;	// ̂Ƃ̌
		};
		EditView*				view_;				// Ώۃr[
		TextSearcher::Type		type_;				// ^Cv
		IEventListener*			eventListener_;		// CxgXi
		std::stack<Operation>	operationHistory_;	// 엚
		std::stack<Status>		statusHistory_;		// ԗ
		Status*					firstStatus_;		// statusHistory_ ̈ԉւ̃|C^
		string_t				pattern_;			// 
		string_t				lastPattern_;		// O񌟍Ǐ
		bool					matched_;			// Ō update ĂяoňveLXg
	};


	/// NX^EAPꋫEAE郆[eBeB
	class BoundaryDetector : public Manah::SelfAssertable {
	public:
		/// ̌\bhŎgtO
		enum SearchPart {
			START					= 0x01,						///< 擪
			END						= 0x02,						///< I[
			AROUND					= START | END,				///< [
			ALPHA_NUMERIC			= 0x04,						///< P\̂
			WORD_START				= START | ALPHA_NUMERIC,	///< P擪
			WORD_END				= END | ALPHA_NUMERIC,		///< PI[
			WORD_AROUND				= AROUND | ALPHA_NUMERIC,	///< Pꗼ[
			ONLY_CURRENT_LINE		= 0x08,						///< Jnsł̂݌
			RESTRICTION_EFFECTIVE	= 0x10						///< i[CỎe󂯂
		};

		// RXgN^
		BoundaryDetector(const Lexer& lexer) throw();
		// 
		const Lexer&	getLexer() const throw();
		// 
		static bool	areSameScriptType(CodePoint cp1, CodePoint cp2);
		bool		hasParagraphBoundaryAt(const char_t* first, const char_t* last, const char_t* at) const;
		bool		hasParagraphBoundaryAt(const string_t& text, length_t index) const;
		bool		hasSentenceBoundaryAt(const char_t* first, const char_t* last, const char_t* at) const;
		bool		hasSentenceBoundaryAt(const string_t& text, length_t index) const;
		bool		hasWordBoundaryAt(const char_t* first, const char_t* last, const char_t* at) const;
		bool		hasWordBoundaryAt(const string_t& text, length_t index) const;
		static bool	isGraphemeExtend(CodePoint cp);
		const char_t*	searchGraphemeBase(const char_t* first, const char_t* last, bool forward) const;
		const char_t*	searchParagraphBoundary(const char_t* first, const char_t* last,
							const char_t* start, bool forward, SearchPart part) const;
		const char_t*	searchSentenceBoundary(const char_t* first, const char_t* last,
							const char_t* start, bool forward, SearchPart part) const;
		const char_t*	searchWordBoundary(const char_t* first, const char_t* last,
							const char_t* start, bool forward, SearchPart part) const;
	private:
		enum WBClass {	// PꋫENX (UAX #29)
			FORMAT, A_LETTER, MID_LETTER, MID_NUM_LET, MID_NUM,
			NUMERIC, SPACE, OTHER, UNCALCULATED
		} getWordBoundaryClass(CodePoint cp) const;

		// f[^o
	private:
		const Lexer& lexer_;
	};


	/// hLg̃NX^EAPꋫEAE
	class DocumentBoundaryDetector : public BoundaryDetector {
	public:
		// RXgN^
		DocumentBoundaryDetector(const Lexer& lexer, const EditDoc& document) throw();
		// 
		const EditDoc&	getDocument() const throw();
		// 
		CharPos	searchGraphemeBase(const CharPos& pos, bool forward) const;
		CharPos	searchParagraphBoundary(const CharPos& pos, bool forward, SearchPart part) const;
		CharPos	searchSentenceBoundary(const CharPos& pos, bool forward, SearchPart part) const;
		CharPos	searchWordBoundary(const CharPos& pos, bool forward, SearchPart part) const;
	private:
		const EditDoc& document_;
	};


// inline implementation
////////////////////////////////////////////////////////////////////////////////
	
	/// obt@ւ̃|C^Ԃ
	inline const char_t* DocumentSearcher::TargetDuplicate::getBuffer() const throw() {return buffer_;}
	
	/// Ԃ
	inline length_t DocumentSearcher::TargetDuplicate::getLength() const throw() {return length_;}

	/// sԂ
	inline length_t DocumentSearcher::TargetDuplicate::getLineCount() const throw() {return lineCount_;}

	/// AhD\Ԃ
	inline bool IncrementalSearcher::canUndo() const throw() {assertValid(); return !operationHistory_.empty();}

	/**
	 *	Ԃ
	 *	@retval true	O
	 *	@retval false	
	 *	@throw IncrementalSearcher::NotRunningException	łȂꍇX[
	 */
	inline bool IncrementalSearcher::getSearchDirection() const {
		assertValid();
		if(!isRunning())
			throw NotRunningException();
		return statusHistory_.top().forward;
	}

	/// ݂̌Ԃ
	inline const string_t& IncrementalSearcher::getSearchText() const throw() {assertValid(); return pattern_;}

	/// ^CvԂ
	/// @throw IncrementalSearcher::NotRunningException łȂꍇX[
	inline TextSearcher::Type IncrementalSearcher::getSearchType() const {
		assertValid();
		if(!isRunning())
			throw NotRunningException();
		return type_;
	}

	/// ݌sԂ
	inline bool IncrementalSearcher::isRunning() const throw() {assertValid(); return !statusHistory_.empty();}

	/// ݐݒ肳Ă錟Ԃ
	inline const TextSearcher::Options& TextSearcher::getOptions() const throw() {assertValid(); return options_;}

	/// uԂ
	inline const string_t& TextSearcher::getReplaceText() const throw() {assertValid(); return replaceWith_;}

	/// Ԃ
	inline const string_t& TextSearcher::getSearchText() const throw() {assertValid(); return findWhat_;}

	/// ŌɌĂ猟IvVύXĂ邩Ԃ
	inline bool TextSearcher::isChangedSinceLastSearch() const throw() {assertValid(); return changedFromLast_;}

	/// 񂪕sɃ}b`p^[Ԃ
	inline bool TextSearcher::isMultilinePattern() const throw() {assertValid(); return multilinePattern_;}

	/// K\\ǂԂ
	inline bool TextSearcher::isRegexAvailable() throw() {
#ifdef ASCENSION_NO_REGEX
		return false;
#else
		return true;
#endif /* ASCENSION_NO_REGEX */
	}

	/**
	 *	݂̖̌ɕǉ
	 *	@param text	ǉ镶
	 *	@return		Č̐
	 *	@throw IncrementalSearcher::NotRunningException	łȂꍇX[
	 *	@throw std::invalid_argument					@a text 󕶎̂ƂX[
	 */
	inline SearchResult IncrementalSearcher::addString(const string_t& text) {return addString(text.data(), text.data() + text.length());}

	/// ͊Ԃ
	inline const Lexer& BoundaryDetector::getLexer() const throw() {assertValid(); return lexer_;}

	/**
	 *	񂪎wʒuɒiEǂׂ
	 *	@param text		ׂ镶
	 *	@param index	ׂʒu
	 *	@return			@a index iȄꍇ true
	 */
	inline bool BoundaryDetector::hasParagraphBoundaryAt(const string_t& text, length_t index) const {
		assertValid(); return hasParagraphBoundaryAt(text.data(), text.data() + text.length(), text.data() + index);}

	/**
	 *	񂪎wʒuɕߋEǂׂ
	 *	@param text		ׂ镶
	 *	@param index	ׂʒu
	 *	@return			@a index ߋȄꍇ true
	 */
	inline bool BoundaryDetector::hasSentenceBoundaryAt(const string_t& text, length_t index) const {
		assertValid(); return hasSentenceBoundaryAt(text.data(), text.data() + text.length(), text.data() + index);}

	/**
	 *	񂪎wʒuɒPꋫEǂׂ
	 *	@param text		ׂ镶
	 *	@param index	ׂʒu
	 *	@return			@a index PꋫȄꍇ true
	 */
	inline bool BoundaryDetector::hasWordBoundaryAt(const string_t& text, length_t index) const {
		assertValid(); return hasWordBoundaryAt(text.data(), text.data() + text.length(), text.data() + index);}

	/// hLgԂ
	inline const EditDoc& DocumentBoundaryDetector::getDocument() const throw() {assertValid(); return document_;}

} // namespace Ascension

#endif /* TEXT_SEARCHER_H_ */

/* [EOF] */