// DocumentView.hpp
// (c) 2003-2006 exeal

#ifndef DOCUMENT_VIEW_HPP_
#define DOCUMENT_VIEW_HPP_
#include <vector>
#include <algorithm>	// std::find


namespace Manah {

template<class Update> class Document;

template<class DocumentUpdate> class AbstractView {
protected:
	virtual ~AbstractView() {}
protected:
	virtual void onAddedToDocument() = 0;	// called by Document::addView
	virtual void onResetDocument() = 0;		// called by Document::resetContent
	virtual void onUpdate(const DocumentUpdate& update) = 0;	// called by Document::updateAllViews
	friend class Manah::Document<DocumentUpdate>;
};

template<class ConcreteDocument /* implements Document */, class DocumentUpdate>
class View : public AbstractView<DocumentUpdate> {
public:
	typedef ConcreteDocument ConcreteDocument;
public:
	View(ConcreteDocument& document) throw() : document_(document) {}
	View(const View<ConcreteDocument, DocumentUpdate>& rhs) throw() : document_(rhs.document_) {}
	ConcreteDocument& getDocument() const throw() {return document_;}
protected:
	virtual ~View() throw() {}
private:
	ConcreteDocument& document_;
};

template<class Update> class Document;

template<class Update> class Document : public SelfAssertable, public Noncopyable {
public:
	typedef AbstractView<Update> View;
	class IListener {
	public:
		virtual ~IListener() {}
		virtual void onChangedViewList(Document<Update>& document) = 0;
		virtual void onChangedModification(Document<Update>& document) = 0;
	};
public:
	Document() throw() : modified_(false), listener_(0) {}
	virtual ~Document();
public:
	bool		addView(View& view);
	std::size_t	getCount() const throw() {assertValid(); return views_.size();}
	View&		getView(std::size_t index) const {assertValid(); return *views_.at(index);}
	bool		isModified() const throw() {assertValid(); return modified_;}
	void		removeAndReleaseView(View& view);
	void		removeAndReleaseView(std::size_t index) {removeAndReleaseView(*views_.at(index));}
	void		resetContent();
	void		setBasicListener(IListener* listener) throw() {listener_ = listener;}
	void		setModified(bool modified = true) throw();
	void		updateAllViews(const Update& update);
protected:
	virtual void	doResetContent() = 0;
private:
	bool modified_;
	std::vector<View*> views_;
	IListener* listener_;
};

template<class Update> class FileBoundDocument;

template<class Update> class FileBoundDocument : public Document<Update> {
public:
	class IListener {
	public:
		virtual ~IListener() {}
		virtual void onChangedFileName(FileBoundDocument<Update>& document) = 0;
	};
public:
	FileBoundDocument() throw() : pathName_(0), listener_(0) {}
	virtual ~FileBoundDocument() throw() {delete[] pathName_;}
public:
	const TCHAR*	getExtensionName() const throw();
	const TCHAR*	getFileName() const throw();
	const TCHAR*	getPathName() const throw() {assertValid(); return pathName_;}
	bool			hasInstance() const throw() {assertValid(); return pathName_ != 0;}
	void			setPathName(const TCHAR* newName);
	void			setFileNameListener(IListener* listener) throw() {listener_ = listener;}
protected:
	virtual void	doResetContent() {setPathName(0);}
private:
	TCHAR* pathName_;
	IListener* listener_;
};


// Document class implementation
/////////////////////////////////////////////////////////////////////////////

template<class Update> inline Document<Update>::~Document() throw() {
	for(std::vector<View*>::iterator it = views_.begin(); it != views_.end(); ++it)
		delete *it;
	views_.erase(views_.begin(), views_.end());
}

template<class Update> inline bool Document<Update>::addView(View& view) {
	assertValid();
	if(std::find(views_.begin(), views_.end(), &view) != views_.end())
		throw std::invalid_argument("This view is already added to the document.");
	views_.push_back(&view);
	view.onAddedToDocument();
	if(listener_ != 0)
		listener_->onChangedViewList(*this);
	return true;
}

template<class Update> inline void Document<Update>::removeAndReleaseView(View& view) {
	assertValid();
	std::vector<View*>::iterator it = std::find(views_.begin(), views_.end(), &view);
	if(it == views_.end())
		throw std::invalid_argument("Specified view does not belong this document.");
	views_.erase(it);
	delete &view;
	if(listener_ != 0)
		listener_->onChangedViewList(*this);
}

template<class Update> inline void Document<Update>::resetContent() {
	assertValid();
	doResetContent();
	for(std::vector<View*>::iterator it = views_.begin(); it != views_.end(); ++it)
		(*it)->onResetDocument();
}

template<class Update> inline void Document<Update>::setModified(bool modified /* = true */) throw() {
	assertValid();
	if(modified != modified_) {
		modified_ = modified;
		if(listener_ != 0)
			listener_->onChangedModification(*this);
	}
}

template<class Update> inline void Document<Update>::updateAllViews(const Update& update) {
	assertValid();
	for(std::vector<View*>::iterator it = views_.begin(); it != views_.end(); ++it)
		(*it)->onUpdate(update);
}

template<class Update> inline const TCHAR* FileBoundDocument<Update>::getExtensionName() const throw() {
	assertValid();
	if(pathName_ == 0)											return 0;
	else if(const TCHAR* p = std::_tcsrchr(pathName_, _T('.')))	return p + 1;
	else														return 0;
}

template<class Update> inline const TCHAR* FileBoundDocument<Update>::getFileName() const throw() {
	assertValid();
	if(pathName_ == 0)												return 0;
	else if(const TCHAR* p = std::_tcsrchr(pathName_, _T('\\')))	return p + 1;
	else															return pathName_;
}

template<class Update> inline void FileBoundDocument<Update>::setPathName(const TCHAR* newName) {
	assertValid();
	if(newName != 0) {
		if(pathName_ != 0 && std::wcscmp(newName, pathName_) == 0)
			return;
		delete[] pathName_;
		pathName_ = new TCHAR[std::_tcslen(newName) + 1];
		std::_tcscpy(pathName_, newName);
	} else if(pathName_ == 0)
		return;
	else {
		delete[] pathName_;
		pathName_ = 0;
	}
	if(listener_ != 0)
		listener_->onChangedFileName(*this);
}

} // namespace Manah

#endif /* DOCUMENT_VIEW_HPP_ */

/* [EOF] */