// File.hpp
// (c) 2003-2006 exeal

#ifndef FILE_HPP_
#define FILE_HPP_
#include "Object.hpp"


namespace Manah {
namespace Windows {
namespace IO {

class FileException : std::runtime_error {
public:
	explicit FileException(const std::string& what_arg) : std::runtime_error(what_arg) {}
};

struct FileBase {
	enum OpenMode {READ_ONLY, WRITE_ONLY, READ_WRITE};
	enum ShareMode {EXCLUSIVE, DENY_WRITE, DENY_READ, DENY_NONE};
	enum {
		NO_INHERIT	= 0x01,	// qvZXŃt@CpȂ
		CREATE		= 0x02,	// (t@CɂĂ) t@C쐬
		NO_TRUNCATE	= 0x04	// CREATE Ƃ̑gݍ킹Ŋ̃t@CjȂ
	};
	enum PointerMovementMode {FROM_BEGIN = FILE_BEGIN, FROM_CURRENT = FILE_CURRENT, FROM_END = FILE_END};
};

template<bool noThrow> class File : public SelfAssertable, public Noncopyable, public FileBase {
	// RXgN^
public:
	explicit File(HANDLE handle = 0);
	File(const TCHAR* fileName, OpenMode openMode, ShareMode shareMode = DENY_NONE, int options = 0);
	virtual ~File();

	// Zq (P^ȊOł͎gĂ͂ȂȂ)
public:
	template<class T> File&	operator <<(const T& buffer) {write<T>(buffer); return *this;}
	template<class T> File&	operator >>(T& buffer) {read<T>(buffer); return *this;}

	// \bh
public:
	/* \z */
	virtual void				abort();
	virtual std::auto_ptr<File>	duplicate() const;
	virtual bool				open(const TCHAR* fileName, OpenMode openMode, ShareMode shareMode = DENY_NONE, int options = 0);
	virtual bool				close();

	/* o (ev[gł͒P^ȊOł͎gĂ͂ȂȂ) */
	virtual bool	read(void* buffer, DWORD bytes, DWORD* readBytes = 0);
	template<typename T>
	bool			read(T& buffer) {
		char buf[sizeof(T)];
		if(read(buf, sizeof(T))) {
			std::uninitialized_copy(buf, buf + 1, buffer);
			return true;
		}
		return false;
	}
	virtual bool	write(const void* lpBuffer, DWORD cb, DWORD* pcbWritten = 0);
	template<typename T>
	bool			write(const T& buffer) {return Write(&buffer, sizeof(T));}
	virtual bool	flush();

	/* ʒu */
	virtual ULONGLONG	getLength() const;
	virtual ULONGLONG	seek(LONGLONG offset, PointerMovementMode mode);
	void				seekToBegin();
	ULONGLONG			seekToEnd();
	virtual void		setLength(ULONGLONG newLength);

	/* bN */
	virtual bool	lockRange(DWORD pos, DWORD count);
	virtual bool	unlockRange(DWORD pos, DWORD count);

	/*  */
	virtual DWORD	getPosition() const;
	virtual bool	getFileTime(LPFILETIME creationTime,
						LPFILETIME lastAccessTime, LPFILETIME lastWriteTime) const;
	virtual DWORD	getCompressedFileSize(DWORD* fileSizeHigh) const;
	virtual bool	isOpened() const;

protected:
#ifdef _DEBUG
	virtual void	assertValidAsFile() const {assertValid(); assert(isOpened());}
#else
	virtual void	assertValidAsFile() const {}
#endif /* _DEBUG */
private:
	static std::string	getLastErrorMessage();

	// f[^o
private:
	HANDLE	handle_;	// file handle
protected:
	TCHAR*	fileName_;	// full path
	bool	autoClose_;	// whether close file when deleted object
};


// File class implementation
/////////////////////////////////////////////////////////////////////////////

template<bool noThrow> inline File<noThrow>::File(HANDLE handle /* = 0*/) : handle_(handle), fileName_(0), autoClose_(false) {}

template<bool noThrow> inline File<noThrow>::File(const TCHAR* fileName, OpenMode openMode,
		ShareMode shareMode /* = DENY_NONE */, int options /* = 0 */) : handle_(0), fileName_(0) {
	assert(fileName != 0);
	open(fileName, openMode, shareMode, options);
}

template<bool noThrow> inline File<noThrow>::~File() {if(handle_ != 0 && autoClose_) close();}

template<bool noThrow> inline void File<noThrow>::abort() {
	assertValid();
	if(handle_ != 0) {
		::CloseHandle(handle_);
		handle_ = 0;
	}
	delete[] fileName_;
	fileName_ = 0;
}

template<bool noThrow> inline bool File<noThrow>::close() {
	assertValid();
	if(handle_ != 0) {
		if(!toBoolean(::CloseHandle(handle_))) {
			if(noThrow)
				return false;
			throw FileException(getLastErrorMessage());
		}
		handle_ = 0;
	}
	autoClose_ = false;
	delete[] fileName_;
	fileName_ = 0;
	return true;
}

template<bool noThrow> inline std::auto_ptr<File<noThrow> > File<noThrow>::duplicate() const {
	assertValid();

	if(handle_ == 0)
		return std::auto_ptr<File<noThrow> >(0);

	HANDLE handle;
	if(!toBoolean(::DuplicateHandle(::GetCurrentProcess(), handle_,
			::GetCurrentProcess(), &handle, 0, false, DUPLICATE_SAME_ACCESS))) {
		if(noThrow)
			return std::auto_ptr<File<noThrow> >(0);
		throw FileException(getLastErrorMessage());
	}
	std::auto_ptr<File> newFile(new File());
	newFile->handle_ = handle;
	assert(newFile->handle_ != 0);
	newFile->autoClose_ = autoClose_;

	return newFile;
}

template<bool noThrow> inline bool File<noThrow>::flush() {
	assertValidAsFile();
	if(!toBoolean(::FlushFileBuffers(handle_))) {
		if(noThrow)
			return false;
		throw FileException(getLastErrorMessage());
	}
	return true;
}

template<bool noThrow> inline std::string File<noThrow>::getLastErrorMessage() {
	std::string	message;
	void*		buffer = 0;
	DWORD		n = ::GetLastError();

	::FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER
		| FORMAT_MESSAGE_FROM_SYSTEM
		| FORMAT_MESSAGE_IGNORE_INSERTS,
		0, n, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		reinterpret_cast<char*>(&buffer), 0, 0);
	message = static_cast<char*>(buffer);
	::LocalFree(buffer);
	return message;
}

template<bool noThrow> inline ULONGLONG File<noThrow>::getLength() const {
	assertValidAsFile();

	::ULARGE_INTEGER len;
	len.LowPart = ::GetFileSize(handle_, &len.HighPart);
	if(len.LowPart == -1 && ::GetLastError() != NO_ERROR && !noThrow)
		throw FileException(getLastErrorMessage());
	return len.QuadPart;
}

template<bool noThrow> inline DWORD File<noThrow>::getCompressedFileSize(DWORD* fileSizeHigh) const {
	assertValidAsFile();
	const DWORD	size = ::GetCompressedFileSize(fileName_, fileSizeHigh);
	if(size == static_cast<DWORD>(-1) && ::GetLastError() != NO_ERROR && !noThrow)
		throw FileException(getLastErrorMessage());
	return size;
}

template<bool noThrow> inline bool File<noThrow>::getFileTime(
		LPFILETIME creationTime, LPFILETIME lastAccessTime, LPFILETIME lastWriteTime) const {
	assertValidAsFile();
	const bool succeeded = toBoolean(::GetFileTime(handle_, creationTime, lastAccessTime, lastWriteTime));
	if(!succeeded && ::GetLastError() != NO_ERROR && !noThrow)
		throw FileException(getLastErrorMessage());
	return succeeded;
}

template<bool noThrow> inline DWORD File<noThrow>::getPosition() const {
	assertValidAsFile();
	const DWORD pos = ::SetFilePointer(handle_, 0, 0, FILE_CURRENT);
	if(pos == static_cast<DWORD>(-1) && !noThrow)
		throw FileException(getLastErrorMessage());
	return pos;
}

template<bool noThrow> inline bool File<noThrow>::isOpened() const {assertValid(); return handle_ != 0;}

template<bool noThrow> inline bool File<noThrow>::lockRange(DWORD pos, DWORD count) {
	assertValidAsFile();
	if(!toBoolean(::LockFile(handle_, pos, 0, count, 0))) {
		if(noThrow)
			return false;
		throw FileException(getLastErrorMessage());
	}
	return true;
}

template<bool noThrow> inline bool File<noThrow>::open(const TCHAR* fileName,
		OpenMode openMode, ShareMode shareMode /* = DENY_NONE */, int options /* = 0 */) {
	assertValid();
	assert(fileName != 0);

	if(isOpened()) {
		if(noThrow)
			return false;
		else
			throw FileException("File is already opened.");
	}

	DWORD accessMode, shareFlag, createFlag;

	switch(openMode) {
	case READ_ONLY:		accessMode = GENERIC_READ;					break;
	case WRITE_ONLY:	accessMode = GENERIC_WRITE;					break;
	case READ_WRITE:	accessMode = GENERIC_READ | GENERIC_WRITE;	break;
	default:			assert(false);
	}

	switch(shareMode) {
	case EXCLUSIVE:		shareFlag = 0;									break;
	case DENY_WRITE:	shareFlag = FILE_SHARE_READ;					break;
	case DENY_READ:		shareFlag = FILE_SHARE_WRITE;					break;
	case DENY_NONE:		shareFlag = FILE_SHARE_READ | FILE_SHARE_WRITE;	break;
	}

	if(toBoolean(options & CREATE))
		createFlag = toBoolean(options & NO_TRUNCATE) ? OPEN_ALWAYS : CREATE_ALWAYS;
	else
		createFlag = OPEN_EXISTING;

	HANDLE handle = ::CreateFile(fileName, accessMode, shareFlag, 0, createFlag, FILE_ATTRIBUTE_NORMAL, 0);
	if(handle == INVALID_HANDLE_VALUE) {
		if(noThrow)
			return false;
		else
			throw FileException(getLastErrorMessage());
	}

	handle_ = handle;
	fileName_ = new TCHAR[STD_::_tcslen(fileName) + 1];
	STD_::_tcscpy(fileName_, fileName);
	autoClose_ = true;

	return true;
}

template<bool noThrow> inline bool File<noThrow>::read(void* buffer, DWORD bytes, DWORD* readBytes /* = 0 */) {
	assertValidAsFile();
	assert(buffer != 0);

	if(bytes == 0) {
		if(readBytes != 0)
			*readBytes = 0;
		return true;
	}

	DWORD readBytesBuffer;
	if(!toBoolean(::ReadFile(handle_, buffer, bytes, &readBytesBuffer, 0))) {
		if(noThrow)
			return false;
		throw FileException(getLastErrorMessage());
	}
	if(readBytes != 0)
		*readBytes = readBytesBuffer;
	return true;
}

template<bool noThrow> inline ULONGLONG File<noThrow>::seek(LONGLONG offset, PointerMovementMode mode) {
	assertValidAsFile();

	::LARGE_INTEGER	offsets;
	offsets.QuadPart = offset;
	offsets.LowPart = ::SetFilePointer(handle_, offsets.LowPart, &offsets.HighPart, mode);
	if(offsets.LowPart == -1 && ::GetLastError() != NO_ERROR) {
		if(noThrow)
			return false;
		throw FileException(getLastErrorMessage());
	}
	return true;
}

template<bool noThrow> inline void File<noThrow>::setLength(ULONGLONG newLength) {
	assertValidAsFile();
	seek(newLength, FROM_BEGIN);
	if(!toBoolean(::SetEndOfFile(handle_)) && !noThrow)
		throw FileException(getLastErrorMessage());
}

template<bool noThrow> inline void File<noThrow>::seekToBegin() {seek(0, File<noThrow>::FROM_BEGIN);}

template<bool noThrow> inline ULONGLONG File<noThrow>::seekToEnd() {return seek(0, File<noThrow>::FROM_END);}

template<bool noThrow> inline bool File<noThrow>::unlockRange(DWORD pos, DWORD count) {
	assertValidAsFile();
	if(!toBoolean(::UnlockFile(handle_, pos, 0, count, 0))) {
		if(noThrow)
			return false;
		throw FileException(getLastErrorMessage());
	}
	return true;
}

template<bool noThrow> inline bool File<noThrow>::write(const void* buffer, DWORD bytes, DWORD* writtenBytes /* = 0 */) {
	assertValidAsFile();
	assert(buffer != 0);

	if(bytes == 0)
		return true;

	DWORD writtenBytesBuffer;
	if(!toBoolean(::WriteFile(handle_, buffer, bytes, (writtenBytes != 0) ? writtenBytes : &writtenBytesBuffer, 0))) {
		if(noThrow)
			return false;
		throw FileException(getLastErrorMessage());
	}
	return true;
}

} /* namespace IO */
} /* namespace Windows */
} /* namespace Manah */

#endif /* FILE_HPP_ */

/* [EOF] */