// Memory.hpp
// (c) 2005-2006 exeal

#ifndef MEMORY_HPP_
#define MEMORY_HPP_

// This file contains following classes
//	GapBuffer
//	UseMemoryPool
//	SmallObject

#include "Object.hpp"
#include <memory>
#include <iterator>
#include <vector>
#include <limits>	// std::numeric_limits
#include <stdexcept>


namespace Manah {

// GapBuffer class definition
/////////////////////////////////////////////////////////////////////////////

// for GapBuffer ElementsDeleter template parameter
template<class T> struct GapBuffer_DoNothing {void operator()(T*, T*) {}};
template<class T> struct GapBuffer_DeletePointer {void operator()(T* first, T* last) {while(first != last) delete *(first++);}};

template<class T, class ElementsDeleter = GapBuffer_DoNothing<T>, class Allocator = std::allocator<T> >	// T must be primitive...
/* final */ class GapBuffer : public Noncopyable {
public:
	// types
#if(_MSC_VER < 1300)
	typedef Allocator::size_type SizeType;
	typedef Allocator::difference_type DifferenceType;
	typedef Allocator::value_type ValueType;
	typedef Allocator::pointer Pointer;
	typedef Allocator::reference Reference;
	typedef Allocator::const_pointer ConstPointer;
	typedef Allocator::const_reference ConstReference;
#else
	typedef typename Allocator::size_type SizeType;
	typedef typename Allocator::difference_type DifferenceType;
	typedef typename Allocator::value_type ValueType;
	typedef typename Allocator::pointer Pointer;
	typedef typename Allocator::reference Reference;
	typedef typename Allocator::const_pointer ConstPointer;
	typedef typename Allocator::const_reference ConstReference;
#endif /* _MSC_VER < 1300 */
	class Iterator;
	class ConstIterator;
#if(_MSC_VER < 1300)
    typedef std::reverse_iterator<Iterator, ValueType, Reference, DifferenceType> ReverseIterator;
    typedef std::reverse_iterator<ConstIterator, ValueType, ConstReference, DifferenceType> ConstReverseIterator;
#else
    typedef std::reverse_iterator<Iterator> ReverseIterator;
    typedef std::reverse_iterator<ConstIterator> ConstReverseIterator;
#endif /* _MSC_VER < 1300 */
public:
	// constructors
	explicit GapBuffer(SizeType initialSize = 10, const Allocator& allocator = Allocator()) :
		allocator_(allocator),
		bufferFirst_(allocator_.allocate(std::max<SizeType>(initialSize, 10), 0)),
		bufferLast_(bufferFirst_ + std::max<SizeType>(initialSize, 10)),
		gapFirst_(bufferFirst_), gapLast_(bufferLast_) {}
	GapBuffer(SizeType count, ConstReference value, const Allocator& allocator = Allocator()) : 
		allocator_(allocator),
		bufferFirst_(allocator_.allocate(count, 0)), bufferLast_(bufferFirst_ + count),
		gapFirst_(bufferFirst_), gapLast_(bufferLast_) {insert(0, count, value);}
	template<class InputIterator> GapBuffer(InputIterator first, InputIterator last, const Allocator& allocator = Allocator()) :
		allocator_(allocator), bufferFirst_(0), bufferLast_(0), gapFirst_(0), gapLast_(0) {insert(0, first, last);}
	~GapBuffer() {clear(); allocator_.deallocate(bufferFirst_, getCapacity());}
	// iterations
	Iterator begin() throw() {return Iterator(*this, bufferFirst_);}
	ConstIterator begin() const throw() {return ConstIterator(*this, bufferFirst_);}
	Iterator end() throw() {return Iterator(*this, bufferLast_);}
	ConstIterator end() const throw() {return ConstIterator(*this, bufferLast_);}
	ReverseIterator rbegin() throw() {return ReverseIterator(end());}
	ConstReverseIterator rbegin() const throw() {return ConstReverseIterator(end());}
	ReverseIterator rend() throw() {return ReverseIterator(begin());}
	ConstReverseIterator rend() const throw() {return ConstReverseIterator(begin());}
	// attributes
	Reference getAt(DifferenceType index) throw() {if(index >= getSize()) throw std::out_of_range(""); return operator[](index);}
	ConstReference getAt(DifferenceType index) const throw() {if(index >= size()) throw std::out_of_range(""); return operator[](index);}
	Reference operator [](DifferenceType index) throw() {return bufferFirst_[(bufferFirst_ + index < gapFirst_) ? index : index + getGap()];}
	ConstReference operator [](DifferenceType index) const throw() {return bufferFirst_[(bufferFirst_ + index < gapFirst_) ? index : index + getGap()];}
	bool isEmpty() const throw() {return getSize() == 0;}
	SizeType getSize() const throw() {return getCapacity() - getGap();}	// returns not byte-count but element count
	SizeType getCapacity() const throw() {return bufferLast_ - bufferFirst_;}	// returns not byte-count but element count
	SizeType getMaxSize() const {return allocator_.max_size();}	// returns not byte-count but element count
	Reference front() throw() {return *begin();}
	ConstReference front() const throw() {return *begin();}
	Reference back() throw() {return *--end();}
	ConstReference back() const throw() {return *--end();}
	// operations (versions take iterator are slow)
	template<class InputIterator> void assign(InputIterator first, InputIterator last) {clear(); insert(0, first, last);}
	void assign(SizeType count, ConstReference value = ValueType()) {clear(); insert(0, count, value);}
	void insert(DifferenceType index, ConstReference value) {
		makeGapAt(bufferFirst_ + index);
		*(gapFirst_++) = value;
		if(gapFirst_ == gapLast_)
			reallocate(getCapacity() * 2);
	}
	template<class InputIterator> void insert(DifferenceType index, InputIterator first, InputIterator last) {
		makeGapAt(bufferFirst_ + getSize());
		makeGapAt(bufferFirst_ + index);
		if(getGap() <= last - first)
			reallocate(std::max(getCapacity() + (last - first) + 1, getCapacity() * 2));
		memcpy(gapFirst_, first, (last - first) * sizeof(ValueType));
		gapFirst_ += last - first;
	}
	Iterator insert(Iterator position, ConstReference value) {
		const DifferenceType offset = position.getOffset(); insert(offset, value); return begin() + offset;}
	template<class InputIterator>
	void insert(ConstIterator position, InputIterator first, InputIterator last) {insert(position.getOffset(), first, last);}
	void clear() {erase(begin(), end());}
	void erase(DifferenceType index, SizeType length = 1) {
		if(bufferFirst_ + index <= gapFirst_ && gapFirst_ <= bufferFirst_ + index + length) {
			ElementsDeleter()(bufferFirst_ + index, gapFirst_);
			length -= (gapFirst_ - bufferFirst_) - index;
			gapFirst_ = bufferFirst_ + index;
		} else
			makeGapAt(bufferFirst_ + index);
		ElementsDeleter()(gapLast_, gapLast_ + length);
		gapLast_ += length;
	}
	Iterator erase(Iterator position) {const DifferenceType offset = position.getOffset(); erase(offset); return begin() + offset;}
	void erase(ConstIterator first, ConstIterator last) {erase(first.getOffset(), last - first);}
	void swap(GapBuffer<ValueType, Allocator>& rhs) {
		std::swap(bufferFirst_, rhs.bufferFirst_);
		std::swap(bufferLast_, rhs.bufferLast_);
		std::swap(gapFirst_, rhs.gapFirst_);
		std::swap(gapLast_, rhs.gapLast_);
	}

	class ConstIterator : public std::iterator<std::random_access_iterator_tag, ValueType, DifferenceType, ConstPointer, ConstReference> {
	public:
		ConstIterator() : target_(0), current_(0) {}
	protected:
		ConstIterator(const GapBuffer<ValueType, ElementsDeleter, Allocator>& target, Pointer position)
			: target_(&target), current_(position) {assert(current_ != 0);}
	public:
		ConstReference operator *() const throw() {return *current_;}
		ConstReference operator ->() const throw() {return **this;}
		ConstIterator& operator ++() throw() {if(++current_ == target_->gapFirst_) current_ = target_->gapLast_; return *this;}
		ConstIterator operator ++(int) throw() {ConstIterator temp(*this); ++*this; return temp;}
		ConstIterator& operator --() throw() {if(--current_ == target_->gapLast_ - 1) current_ = target_->gapFirst_ - 1; return *this;}
		ConstIterator operator --(int) throw() {ConstIterator temp(*this); --*this; return temp;}
		ConstIterator& operator +=(DifferenceType n) throw() {
            if(current_ + n >= target_->gapFirst_ && current_ + n < target_->gapLast_)
                n += target_->getGap();
            current_ += n;
            return *this;
		}
		ConstIterator& operator -=(DifferenceType n) throw() {return *this += -n;}
		ConstIterator operator +(DifferenceType n) const throw() {ConstIterator temp(*this); return temp += n;}
		ConstIterator operator -(DifferenceType n) const throw() {ConstIterator temp(*this); return temp -= n;}
		DifferenceType operator -(const ConstIterator& rhs) const {return getOffset() - rhs.getOffset();}
		bool operator ==(const ConstIterator& rhs) const throw() {return getOffset() == rhs.getOffset();}
		bool operator !=(const ConstIterator& rhs) const throw() {return !(*this == rhs);}
		bool operator <(const ConstIterator& rhs) const throw() {return getOffset() < rhs.getOffset();}
		bool operator <=(const ConstIterator& rhs) const throw() {return *this == rhs || *this < rhs;}
		bool operator >(const ConstIterator& rhs) const throw() {return !(*this <= rhs);}
		bool operator >=(const ConstIterator& rhs) const throw() {return !(*this < rhs);}
		friend ConstIterator operator +(DifferenceType lhs, const ConstIterator& rhs) throw() {return rhs + lhs;}
	protected:
		DifferenceType getOffset() const throw() {
			return (current_ <= target_->gapFirst_) ?
				current_ - target_->bufferFirst_ : current_ - target_->gapLast_ + target_->gapFirst_ - target_->bufferFirst_;}
		const GapBuffer<ValueType, ElementsDeleter, Allocator>* target_;
		Pointer current_;
		friend class GapBuffer<ValueType, ElementsDeleter, Allocator>;
	};
	class Iterator : public ConstIterator {
	public:
		Iterator() {}
	private:
		Iterator(const GapBuffer<ValueType, ElementsDeleter, Allocator>& target, Pointer position) : ConstIterator(target, position) {}
	public:
		typedef Pointer pointer;
		typedef Reference reference;
		Reference operator *() throw() {return *current_;}
		Reference operator ->() throw() {return **this;}
		Iterator& operator ++() throw() {if(++current_ == target_->gapFirst_) current_ = target_->gapLast_; return *this;}
		Iterator operator ++(int) throw() {Iterator temp(*this); ++*this; return temp;}
		Iterator& operator --() throw() {if(--current_ == target_->gapLast_ - 1) current_ = target_->gapFirst_ - 1; return *this;}
		Iterator operator --(int) throw() {Iterator temp(*this); --*this; return temp;}
		Iterator& operator +=(DifferenceType n) throw() {
            if(current_ + n >= target_->gapFirst_ && current_ + n < target_->gapLast_)
                n += target_->getGap();
            current_ += n;
            return *this;
		}
		Iterator& operator -=(DifferenceType n) throw() {return *this += -n;}
		Iterator operator +(DifferenceType n) const throw() {Iterator temp(*this); return temp += n;}
		Iterator operator -(DifferenceType n) const throw() {Iterator temp(*this); return temp -= n;}
		DifferenceType operator -(const Iterator& rhs) const throw() {return getOffset() - rhs.getOffset();}
		bool operator ==(const Iterator& rhs) const throw() {return getOffset() == rhs.getOffset();}
		bool operator !=(const Iterator& rhs) const throw() {return !(*this == rhs);}
		bool operator <(const Iterator& rhs) const throw() {return getOffset() < rhs.getOffset();}
		bool operator <=(const Iterator& rhs) const throw() {return *this == rhs || *this < rhs;}
		bool operator >(const Iterator& rhs) const throw() {return !(*this <= rhs);}
		bool operator >=(const Iterator& rhs) const throw() {return !(*this < rhs);}
		friend Iterator operator +(DifferenceType lhs, const Iterator& rhs) throw() {return rhs + lhs;}
		friend class GapBuffer<ValueType, ElementsDeleter, Allocator>;
	};
private:
	// helpers
	DifferenceType getGap() const throw() {return gapLast_ - gapFirst_;}
	void makeGapAt(Pointer position) {
		using namespace std;
		if(position < gapFirst_) {
			gapLast_ -= gapFirst_ - position;
			memmove(gapLast_, position, (gapFirst_ - position) * sizeof(T));
		} else if(position > gapFirst_) {
			const Pointer p = position + getGap();
			memmove(gapFirst_, gapLast_, (p - gapLast_) * sizeof(T));
			gapLast_ = p;
		}
		gapFirst_ = position;
	}
	void reallocate(SizeType newSize) {	// size is not byte-count but element-count
		Pointer newBuffer = allocator_.allocate(newSize, 0);
		Pointer old = bufferFirst_;
		const DifferenceType tailOffset = gapLast_ - bufferFirst_;
		const SizeType tailLength = getCapacity() - tailOffset;

		std::uninitialized_copy(old, old + (gapFirst_ - bufferFirst_), newBuffer);
		std::uninitialized_copy(old + tailOffset, old + tailOffset + tailLength, newBuffer + newSize - tailLength);
		allocator_.deallocate(old, getCapacity());
		gapFirst_ = newBuffer + (gapFirst_ - bufferFirst_);
		bufferFirst_ = newBuffer;
		gapLast_ = (bufferFirst_ + newSize) - tailLength;
		bufferLast_ = bufferFirst_ + newSize;
	}

private:
	Allocator allocator_;
	Pointer bufferFirst_, bufferLast_;
	Pointer gapFirst_, gapLast_;

	friend class Iterator;
	friend class ConstIterator;
};


// UseMemoryPool class definition
/////////////////////////////////////////////////////////////////////////////

// std::auto_ptr alternative for VC6's broken one
template<class T> class PoolHolder {
public:
	PoolHolder() : p_(0) {}
	~PoolHolder() {delete p_;}
	void operator =(T* p) {delete p_; p_ = p;}
	T* operator ->() {return p_;}
private:
	T* p_;
};

// efficient memory pool implementation (from MemoryPool of Efficient C++).
// template paramater of Allocator always bound to char
#if(_MSC_VER < 1300)
template<class T, std::size_t expansionChunkCount = 32, class Allocator = std::allocator<char> >
#else
template<class T, std::size_t expansionChunkCount = 32, template<class> class Allocator = std::allocator>
#endif /* _MSC_VER < 1300 */
/* final */ class UseMemoryPool {
public:
	static void* operator new(std::size_t cb) {
		if(pool_.operator->() == 0)
			pool_ = new MemoryPool;
		return pool_->allocate(cb);
	}
	static void operator delete(void* p, std::size_t) {pool_->deallocate(p);}
private:
	class MemoryPool {
	public:
		MemoryPool() {expandList();}
		~MemoryPool() {
#if(_MSC_VER < 1300)
			const std::size_t CHUNK_SIZE = std::max(sizeof(T), sizeof(MemoryPool*));
#endif /* _MSC_VER < 1300 */
			for(MemoryPool* next = next_; next != 0; next = next_) {
				next_ = next_->next_;
				allocator_.deallocate(reinterpret_cast<char*>(next), CHUNK_SIZE);
			}
		}
		void* allocate(std::size_t) {
			if(next_ == 0)
				expandList();
			MemoryPool* head = next_;
			next_ = head->next_;
			return head;
		}
		void deallocate(void* p) {
			MemoryPool* head = static_cast<MemoryPool*>(p);
			head->next_ = next_;
			next_ = head;
		}
	private:
		void expandList() {
#if(_MSC_VER < 1300)
			const std::size_t CHUNK_SIZE = std::max(sizeof(T), sizeof(MemoryPool*));
#endif /* _MSC_VER < 1300 */
			MemoryPool* p = reinterpret_cast<MemoryPool*>(allocator_.allocate(CHUNK_SIZE));

			next_ = p;
			for(int i = 0; i < expansionChunkCount; ++i) {
				p->next_ = reinterpret_cast<MemoryPool*>(allocator_.allocate(CHUNK_SIZE));
				p = p->next_;
			}
			p->next_ = 0;
		}

	private:
#if((!defined _MSC_VER) || (_MSC_VER >= 1300))
		enum {CHUNK_SIZE = (sizeof(T) > sizeof(MemoryPool*)) ? sizeof(T) : sizeof(MemoryPool*)};
#endif /* (!defined _MSC_VER) || (_MSC_VER >= 1300) */
		MemoryPool*	next_;
	};
	static PoolHolder<MemoryPool> pool_;
#if(_MSC_VER < 1300)
	static Allocator allocator_;
#else
	static Allocator<char> allocator_;
#endif /* _MSC_VER < 1300 */
};

#if(_MSC_VER < 1300)
template<class T, std::size_t expansionChunkCount, class Allocator>
PoolHolder<UseMemoryPool<T, expansionChunkCount, Allocator>::MemoryPool>
UseMemoryPool<T, expansionChunkCount, Allocator>::pool_;
template<class T, std::size_t size, class Allocator>
Allocator UseMemoryPool<T, size, Allocator>::allocator_;
#else
template<class T, std::size_t expansionChunkCount, template<class> class Allocator>
PoolHolder<typename UseMemoryPool<T, expansionChunkCount, Allocator>::MemoryPool>
UseMemoryPool<T, expansionChunkCount, Allocator>::pool_;
template<class T, std::size_t size, template<class> class Allocator>
Allocator<char> UseMemoryPool<T, size, Allocator>::allocator_;
#endif /* _MSC_VER < 1300 */


// SmallObject class definition
/////////////////////////////////////////////////////////////////////////////

#ifndef SMALL_OBJECT_DEFAULT_CHUNK_SIZE
#define SMALL_OBJECT_DEFAULT_CHUNK_SIZE	4096
#endif
#ifndef MAX_SMALL_OBJECT_SIZE
#define MAX_SMALL_OBJECT_SIZE	64
#endif

template<std::size_t chunkSize, std::size_t maxSmallObjectSize>
class SmallObjectAllocator;

struct NoLock {};

// small object (from Loki (http://sourceforge.net/projects/loki-lib/))
template<
	class Locker = NoLock,
	std::size_t chunkSize = SMALL_OBJECT_DEFAULT_CHUNK_SIZE,
	std::size_t maxSmallObjectSize = MAX_SMALL_OBJECT_SIZE>
class SmallObject : public SelfAssertable {
public:
	virtual	~SmallObject() {}
	static void* operator new(std::size_t size) {
#if(SMALL_OBJECT_DEFAULT_CHUNK_SIZE != 0) && (MAX_SMALL_OBJECT_SIZE != 0)
		Locker locker;
		return getAllocator().allocate(size);
#else
		return ::operator new(size);
#endif
}
	static void operator delete(void* p, std::size_t size) {
#if(DEFAULT_CHUNK_SIZE != 0) && (MAX_SMALL_OBJECT_SIZE != 0)
		Locker locker;
		getAllocator().deallocate(p, size);
#else
		::operator delete(p, size);
#endif
}
private:
	static SmallObjectAllocator<chunkSize, maxSmallObjectSize>&	getAllocator() {
		static SmallObjectAllocator<chunkSize, maxSmallObjectSize> allocator;
		return allocator;
	}
};

class FixedAllocator {
public:
	typedef unsigned char SizeType;
	explicit FixedAllocator(std::size_t blockSize = 0) : blockSize_(blockSize), lastAllocatedChunk_(0), lastDeallocatedChunk_(0) {
		assert(blockSize_ > 0);
		next_ = prev_ = this;
		std::size_t blockCount = SMALL_OBJECT_DEFAULT_CHUNK_SIZE / blockSize;
		if(blockCount > std::numeric_limits<SizeType>::max())
			blockCount = std::numeric_limits<SizeType>::max();
		else if(blockCount == 0)
			blockCount = 8 * blockSize;
		blockCount_ = static_cast<SizeType>(blockCount);
		assert(blockCount_ == blockCount);
	}
	FixedAllocator(const FixedAllocator& rhs) : blockSize_(rhs.blockSize_), blockCount_(rhs.blockCount_), chunks_(rhs.chunks_) {
		prev_ = &rhs;
		next_ = rhs.next_;
		rhs.next_->prev_ = this;
		rhs.next_ = this;
		lastAllocatedChunk_ = (rhs.lastAllocatedChunk_ != 0) ?
			&chunks_.front() + (rhs.lastAllocatedChunk_ - &rhs.chunks_.front()) : 0;
		lastDeallocatedChunk_ = (rhs.lastDeallocatedChunk_ != 0) ?
			&chunks_.front() + (rhs.lastDeallocatedChunk_ - &rhs.chunks_.front()) : 0;
	}
	~FixedAllocator() {
		if(prev_ != this) {
			prev_->next_ = next_;
			next_->prev_ = prev_;
			return;
		}
		assert(prev_ == next_);
		for(ChunkVector::iterator it = chunks_.begin(); it != chunks_.end(); ++it) {
//			assert(it->getAvailableBlockCount() == blockCount_);
			it->release();
		}
	}

	// \bh
public:
	void* allocate() {
		if(lastAllocatedChunk_ != 0 && lastAllocatedChunk_->isAvailable())
			return lastAllocatedChunk_->allocate(blockSize_);
		for(ChunkVector::iterator it = chunks_.begin(); ; ++it) {
			if(it == chunks_.end()) {
				chunks_.reserve(chunks_.size() + 1);
				chunks_.push_back(Chunk(blockSize_, blockCount_));
				lastAllocatedChunk_ = &chunks_.back();
				lastDeallocatedChunk_ = &chunks_.front();
				break;
			} else if(it->isAvailable()) {
				lastAllocatedChunk_ = &*it;
				break;
			}
		}
		assert(lastAllocatedChunk_ != 0 && lastAllocatedChunk_->isAvailable());
		return lastAllocatedChunk_->allocate(blockSize_);
	}
	void deallocate(void* p) {
		assert(p != 0);
		assert(!chunks_.empty());
		assert(&chunks_.front() <= lastDeallocatedChunk_);
		assert(&chunks_.back() >= lastDeallocatedChunk_);
		assert(lastDeallocatedChunk_ != 0);

		const std::size_t	chunkLength = blockSize_ * blockCount_;
		Chunk*				lower = lastDeallocatedChunk_;
		Chunk*				upper = lower + 1;
		const Chunk*		lowerBound = &chunks_.front();
		const Chunk*		upperBound = &chunks_.back() + 1;

		if(upper == upperBound)
			upper = 0;
		while(true) {
			if(lower != 0) {
				if(lower->isContaining(p, chunkLength)) {
					lastDeallocatedChunk_ = lower;
					break;
				} else if(lower == lowerBound)
					lower = 0;
				else
					--lower;
			}
			if(upper != 0) {
				if(upper->isContaining(p, chunkLength)) {
					lastDeallocatedChunk_ = upper;
					break;
				} else if(++upper == upperBound)
					upper = 0;
			}
		}

		assert(lastDeallocatedChunk_ != 0);
		assert(lastDeallocatedChunk_->isContaining(p, chunkLength));
		lastDeallocatedChunk_->deallocate(p, blockSize_);
		if(lastDeallocatedChunk_->getAvailableBlockCount() == blockCount_) {
			Chunk* lastChunk = &chunks_.back();
			if(lastChunk == lastDeallocatedChunk_) {
				if(chunks_.size() > 1 && lastDeallocatedChunk_[-1].getAvailableBlockCount() == blockCount_) {
					lastChunk->release();
					chunks_.pop_back();
					lastAllocatedChunk_ = lastDeallocatedChunk_ = &chunks_.front();
				}
				return;
			}
			if(lastChunk->getAvailableBlockCount() == blockCount_) {
				lastChunk->release();
				chunks_.pop_back();
				lastAllocatedChunk_ = lastDeallocatedChunk_;
			} else {
				std::swap(*lastDeallocatedChunk_, *lastChunk);
				lastAllocatedChunk_ = &chunks_.back();
			}
		}
	}
	std::size_t	GetBlockSize() const throw() {return blockSize_;}

	// Zq
public:
	FixedAllocator& operator =(const FixedAllocator& rhs) {
		FixedAllocator copy(rhs);
		std::swap(blockSize_, copy.blockSize_);
		std::swap(blockCount_, copy.blockCount_);
		chunks_.swap(copy.chunks_);
		std::swap(lastAllocatedChunk_, copy.lastAllocatedChunk_);
		std::swap(lastDeallocatedChunk_, copy.lastDeallocatedChunk_);
		return *this;
	}
	bool operator <(std::size_t rhs) const throw() {return blockSize_ < rhs;}
private:

	// NX
private:
	class Chunk {
	public:
		Chunk(std::size_t blockSize, SizeType blockCount) : firstAvailableBlock_(0), availableBlockCount_(blockCount) {
			assert(blockSize > 0 && blockCount > 0);
			assert((blockSize * blockCount) / blockSize == blockCount);
			data_ = new SizeType[blockSize * blockCount];
			SizeType* p = data_;
			for(SizeType i = 0; i < blockCount; p += blockSize)
				*p = ++i;
		}
		void* allocate(std::size_t size) {
			if(availableBlockCount_ == 0)
				return 0;
			assert((firstAvailableBlock_ * size) / size == firstAvailableBlock_);
			SizeType* p = data_ + firstAvailableBlock_ * size;
			firstAvailableBlock_ = *p;
			--availableBlockCount_;
			return p;
		}
		void deallocate(void* p, std::size_t blockSize) {
			assert(p >= data_);
			SizeType* releasing = static_cast<SizeType*>(p);
			assert((releasing - data_) % blockSize == 0);
			*releasing = firstAvailableBlock_;
			firstAvailableBlock_ = static_cast<SizeType>((releasing - data_) / blockSize);
			assert(firstAvailableBlock_ == (releasing - data_) / blockSize);
			++availableBlockCount_;
		}
		SizeType getAvailableBlockCount() const throw() {return availableBlockCount_;}
		bool isAvailable() const throw() {return availableBlockCount_ > 0;}
		bool isContaining(const void* p, std::size_t size) const throw() {return p >= data_ && p < data_ + size;}
		void release() {delete[] data_;}
	private:
		SizeType*	data_;
		SizeType	firstAvailableBlock_;
		SizeType	availableBlockCount_;
	};

	// f[^o
private:
	typedef std::vector<Chunk> ChunkVector;
	std::size_t	blockSize_;
	SizeType	blockCount_;
	ChunkVector	chunks_;
	Chunk*		lastAllocatedChunk_;
	Chunk*		lastDeallocatedChunk_;
	mutable const FixedAllocator*	next_;
	mutable const FixedAllocator*	prev_;
};

template<std::size_t chunkSize, std::size_t maxSmallObjectSize> class SmallObjectAllocator : public Noncopyable {
	friend class SmallObject<>;
private:
	SmallObjectAllocator() : lastAllocated_(0), lastDeallocated_(0) {}
//	~SmallObjectAllocator() {}
public:
	void* allocate(std::size_t bytes) {
		if(bytes > maxSmallObjectSize)
			return ::operator new(bytes);
		else if(lastAllocated_ != 0 && lastAllocated_->getBlockSize() == bytes)
			return lastAllocated_->allocate();
		std::vector<FixedAllocator>::iterator it = std::lower_bound(pools_.begin(), pools_.end(), bytes);
		if(it == pools_.end() || it->getBlockSize() != bytes) {
			it = pools_.insert(it, FixedAllocator(bytes));
			lastDeallocated_ = &*pools_.begin();
		}
		lastAllocated_ = &*it;
		return lastAllocated_->allocate();
	}
	void deallocate(void* p, std::size_t bytes) {
		if(bytes > maxSmallObjectSize) {
			::operator delete(p);
			return;
		} else if(lastDeallocated_ != 0 && lastDeallocated_->getBlockSize() == bytes) {
			lastDeallocated_->deallocate(p);
			return;
		}		
		std::vector<FixedAllocator>::iterator it = std::lower_bound(pools_.begin(), pools_.end(), bytes);
		assert(it != pools_.end() && it->getBlockSize() == bytes);
		lastDeallocated_ = &*it;
		lastDeallocated_->deallocate(p);
	}
private:
	std::vector<FixedAllocator> pools_;
	FixedAllocator* lastAllocated_;
	FixedAllocator* lastDeallocated_;
};

} // namespace Manah

#endif /* MEMORY_HPP_ */

/* [EOF] */