// Object.hpp
// (c) 2003-2006 exeal

#ifndef OBJECT_HPP_
#define OBJECT_HPP_

#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <cassert>
#include <cstring>	// prevent C header inclusion
#include <cwchar>	// prevent C header inclusion
#include <cstdlib>	// prevent C header inclusion
#include <string>
#include <sstream>
#include <memory>
#undef min
#undef max
#define size_t	std::size_t
//#include <winnt.h>
#include <windows.h>
#include <tchar.h>
#undef size_t

#ifdef _UNICODE
#define tstring	wstring
#else
#define tstring	string
#endif

#if(_MSC_VER < 1300)
#define STD_
#else
#define STD_	std
#endif /* _MSC_VER < 1300 */

namespace Manah {
	namespace Windows {
		class DumpContext;
	}
}


// modern types
/////////////////////////////////////////////////////////////////////////////

typedef unsigned char	uchar;
typedef unsigned short	ushort;
typedef unsigned int	uint;
typedef unsigned long	ulong;


// basic classes
/////////////////////////////////////////////////////////////////////////////

namespace Manah {

	class SelfAssertable {
	protected:
		virtual void assertValid() const {
#ifdef _DEBUG
			assert(this != 0);
#endif /* _DEBUG */
		}
	};

	// unassignable object's base class
	class Unassignable {
	protected:
		Unassignable() {}
		Unassignable(const Unassignable& rhs) {}
		~Unassignable() {}
	private:
		const Unassignable& operator =(const Unassignable&);
	};

	// noncopyable object's base class (from Boost.Noncopyable)
	class Noncopyable {
	protected:
		Noncopyable() {}
		~Noncopyable() {}
	private:
		Noncopyable(const Noncopyable&);
		const Noncopyable& operator =(const Noncopyable&);
	};

	template<class Enum> class Flags {
	public:
		Flags(Enum e) : e_(e) {}
		Flags(int e = 0) : e_(static_cast<Enum>(e)) {}
		Flags(const Flags<Enum>& rhs) : e_(rhs.e_) {}
		Flags<Enum>& operator =(const Flags<Enum>& rhs) {e_ = rhs.e_; return *this;}
		Flags<Enum>& operator =(Enum rhs) {e_ = rhs; return *this;}
		Flags<Enum> operator &(Enum rhs) const {return Flags<Enum>(*this) &= rhs;}
		Flags<Enum> operator |(Enum rhs) const {return Flags<Enum>(*this) |= rhs;}
		Flags<Enum> operator ^(Enum rhs) const {return Flags<Enum>(*this) ^= rhs;}
		Flags<Enum>& operator &=(Enum rhs) {e_ &= rhs; return *this;}
		Flags<Enum>& operator |=(Enum rhs) {e_ |= rhs; return *this;}
		Flags<Enum>& operator ^=(Enum rhs) {e_ ^= rhs; return *this;}
		bool operator ==(Enum rhs) const {return e_ == rhs;}
		bool operator !=(Enum rhs) const {return !(*this == rhs);}
		operator Enum() {return e_;}
		bool has(Enum e) const {return (e_ & e) != 0;}
	private:
		Enum e_;
	};


	namespace Windows {

		struct ResourceID {
			ResourceID(const TCHAR* name) : name_(name) {}
			ResourceID(UINT_PTR id) : name_(MAKEINTRESOURCE(id)) {}
			const TCHAR* const name_;
		};

		// base class for handle-wrapper classes
		template<class Handle>
		class HandleHolder : public SelfAssertable, public Unassignable {
		public:
			struct InvalidHandleException : public std::invalid_argument {
				InvalidHandleException() : std::invalid_argument("invalid handle value.") {}
			};
		protected:
			HandleHolder(Handle handle = 0) : handle_(0), attached_(false) {if(handle != 0) attach(handle);}
			HandleHolder(const HandleHolder& rhs) : handle_(0), attached_(false) {if(rhs.handle_ != 0) attach(rhs.handle_);}
			virtual ~HandleHolder() {detach();}
		public:
			virtual bool attach(Handle handle) {	// subclass can throw InvalidHandleException
				assertValid();
				if(!attached_) {
					handle_ = handle;
					attached_ = true;
					return true;
				}
				return false;
			}
			virtual Handle detach() throw() {
				assertValid();
				if(attached_) {
					Handle old = handle_;
					handle_ = 0;
					attached_ = false;
					return old;
				}
				return 0;
			}
		protected:
			Handle getHandle() const throw() {assertValid(); return handle_;}
			bool isAttached() const throw() {assertValid(); return attached_;}
			void setHandle(Handle newHandle) throw() {assertValid(); detach(); handle_ = newHandle; attached_ = false;}
		private:
			Handle handle_;
			bool attached_;
		};

		// convenient types from ATL
		struct MenuHandleOrControlID {
			MenuHandleOrControlID(HMENU menu) : menu_(menu) {}
			MenuHandleOrControlID(UINT_PTR id) : menu_(reinterpret_cast<HMENU>(id)) {}
			HMENU menu_;
		};

		// others used by (CCustomControl derevied)::GetWindowClass
		struct BrushHandleOrColor {
			BrushHandleOrColor() : brush_(0) {}
			BrushHandleOrColor(HBRUSH brush) : brush_(brush) {}
			BrushHandleOrColor(COLORREF color) : brush_(reinterpret_cast<HBRUSH>(static_cast<HANDLE_PTR>(color + 1))) {}
			BrushHandleOrColor& operator =(HBRUSH brush) {brush_ = brush; return *this;}
			BrushHandleOrColor& operator =(COLORREF color) {brush_ = reinterpret_cast<HBRUSH>(static_cast<HANDLE_PTR>(color + 1)); return *this;}
			HBRUSH brush_;
		};
		struct CursorHandleOrID {
			CursorHandleOrID() : cursor_(0) {}
			CursorHandleOrID(HCURSOR cursor) : cursor_(cursor) {}
			CursorHandleOrID(const TCHAR* systemCursorID) : cursor_(::LoadCursor(0, systemCursorID)) {}
			CursorHandleOrID& operator =(HCURSOR cursor) {cursor_ = cursor; return *this;}
			CursorHandleOrID& operator =(const TCHAR* systemCursorID) {cursor_ = ::LoadCursor(0, systemCursorID); return *this;}
			HCURSOR cursor_;
		};

		class DumpContext : public SelfAssertable {
		public:
			// constructor
			DumpContext(const TCHAR* fileName = 0);
			// operators
			template<class T>
			DumpContext& operator <<(const T& rhs) throw();
			// methods
			void	flush() throw();
			void	hexDump(const TCHAR* line, uchar* pb, int bytes, int width = 0x10) throw();

			// data member
		private:
			TCHAR* fileName_;	// error log
		};

		namespace {
			DumpContext	dout;
		}

} /* namespace Windows */
} /* namespace Manah */


// macros
/////////////////////////////////////////////////////////////////////////////

#define toBoolean(exp)	((exp) != 0)
#ifndef countof
#define countof(array)	(sizeof(array) / sizeof(array[0]))
#endif /* !countof */
#ifndef endof
#define endof(array)	(array + countof(array))
#endif /* !endof */

// sizeof(MENUITEMINFO)
#if(WINVER >= 0x0500 && !defined(MENUITEMINFO_SIZE_VERSION_400))
#define MENUITEMINFO_SIZE_VERSION_400A	(offsetof(MENUITEMINFOA, cch) + sizeof(static_cast<MENUITEMINFOA*>(0)->cch))
#define MENUITEMINFO_SIZE_VERSION_400W	(offsetof(MENUITEMINFOW, cch) + sizeof(static_cast<MENUITEMINFOW*>(0)->cch))
#ifdef UNICODE
#define MENUITEMINFO_SIZE_VERSION_400	MENUITEMINFO_SIZE_VERSION_400W
#else
#define MENUITEMINFO_SIZE_VERSION_400	MENUITEMINFO_SIZE_VERSION_400A
#endif /* !UNICODE */
#endif /* WINVER >= 0x0500 && !defined(MENUITEMINFO_SIZE_VERSION_400) */

// Win32 structure initializement
template<class Base> struct AutoZero : public Base {
	AutoZero() {std::memset(this, 0, sizeof(Base));}
};
template<class Base, typename sizeMemberType, sizeMemberType(Base::*sizeMember)>
struct AutoZeroS : public AutoZero<Base> {
	AutoZeroS() {this->*sizeMember = sizeof(Base);}
};
#if(_MSC_VER < 1300)
template<class Base> struct AutoZeroCB : public AutoZeroS<Base, UINT, typename Base::cbSize> {};
template<class Base> struct AutoZeroLS : public AutoZeroS<Base, DWORD, typename Base::lStructSize> {};
#else
template<class Base> struct AutoZeroCB : public AutoZeroS<Base, UINT, &Base::cbSize> {};
template<class Base> struct AutoZeroLS : public AutoZeroS<Base, DWORD, &Base::lStructSize> {};
#endif



// DumpContext class implementation
/////////////////////////////////////////////////////////////////////////////

inline Manah::Windows::DumpContext::DumpContext(const TCHAR* fileName /* = 0 */) {
	if(fileName != 0)
		throw std::exception("File log is not supported!");
		// STD_::_tcscpy(fileName_, fileName);
}

inline void Manah::Windows::DumpContext::flush() throw() {
	assertValid();
	/* not implemented */
}

inline void Manah::Windows::DumpContext::hexDump(const TCHAR* line, uchar* pb, int bytes, int width /* = 0x10 */) throw() {
	assertValid();

	TCHAR* const output = new TCHAR[static_cast<std::size_t>(
		(STD_::_tcslen(line) + 3 * width + 2) * static_cast<float>(bytes / width))];
	STD_::_tcscpy(output, line);
	
	TCHAR byte[4];
	for(int i = 0; i < bytes; i++){
		::wsprintf(byte, _T(" %d"), pb);
		STD_::_tcscat(output, byte);
		if(i % width == 0){
			STD_::_tcscat(output, _T("\n"));
			STD_::_tcscat(output, line);
		}
	}
	::OutputDebugString(_T("\n>----Dump is started"));
	::OutputDebugString(output);
	::OutputDebugString(_T("\n>----Dump is done"));
	delete[] output;
}

template <typename T>
inline Manah::Windows::DumpContext& Manah::Windows::DumpContext::operator <<(const T& rhs) throw() {
	assertValid();
	std::basic_ostringstream<TCHAR>	ss;
	ss << rhs;
	::OutputDebugString(ss.str().c_str());
	return *this;
}

#endif /* OBJECT_HPP_ */

/* [EOF] */