// Splitter.hpp
// (c) 2005-2006 exeal

#ifndef SPLITTER_HPP_
#define SPLITTER_HPP_
#include "Window.hpp"

namespace Manah {
namespace Windows {
namespace Controls {

enum ChildrenDestructionPolicy {
	CHILDRENDESTRUCTIONPOLICY_DELETE,						// pane objects' destructors
	CHILDRENDESTRUCTIONPOLICY_DONTDELETE,					// do nothing about panes
	CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL	// call SetParent(pane.GetWindow(), 0) when root window destroyed
};
enum SplitDirection {
	SPLITDIRECTION_NOSPLIT,
	SPLITDIRECTION_NS,
	SPLITDIRECTION_WE
};
enum PanePosition {
	PANEPOSITION_LEFT = 0,
	PANEPOSITION_TOP = PANEPOSITION_LEFT,
	PANEPOSITION_RIGHT = 1,
	PANEPOSITION_BOTTOM = PANEPOSITION_RIGHT,
	PANEPOSITION_COUNT = 2
};


namespace Private {
	template<class Pane, ChildrenDestructionPolicy cdp> class Splitter;
}


class AbstractPane {
public:
	virtual ~AbstractPane() {}
public:
	virtual HWND getWindow() const = 0;	// provides pane's window handle
};


template<class Pane /* implements CAbstractPane*/, ChildrenDestructionPolicy cdp = CHILDRENDESTRUCTIONPOLICY_DELETE>
/* final */ class SplitterRoot : public CustomControl<SplitterRoot>, Noncopyable {
public:
	class Iterator : public Unassignable {	// once the structure is changed, iterator is inavailable
	private:
		typedef Private::Splitter<Pane, cdp> Splitter;
		Iterator(const Splitter& root) : parent_(&root) {reset();}
		Iterator(const Splitter& parent, Pane& pane) : parent_(&parent), pane_(&pane) {}
	public:
		Iterator(const Iterator& rhs) : parent_(rhs.parent_), pane_(rhs.pane_) {}
	public:
		Pane& get() const {assert(pane_ != 0); return *pane_;}
		bool isEnd() const {return pane_ == 0;}
		void next() {
			Pane* next = 0;
			Splitter* parent = 0;
			if(parent_->getNextPane(*pane_, true, next, &parent)) {
				parent_ = parent;
				pane_ = next;
			} else	// at end
				pane_ = 0;
		}
		void reset() {
			while(parent_->parent_ != 0)
				parent_ = parent_->parent_;
			parent_->getFirstPane(true, pane_, const_cast<Splitter**>(&parent_));
		}
	private:
		const Splitter* parent_;
		Pane* pane_;
		friend class SplitterRoot<Pane, cdp>;
	};

	DEFINE_WINDOW_CLASS() {
		name = _T("ManahSplitterRoot");
		bgColor = COLOR_BTNFACE;
		style = CS_DBLCLKS;
	}

public:
	SplitterRoot() : defaultActivePane_(0),
			frameWidth_(::GetSystemMetrics(SM_CXSIZEFRAME)), frameHeight_(::GetSystemMetrics(SM_CYSIZEFRAME)),
			minimumPaneWidth_(::GetSystemMetrics(SM_CXMIN)), minimumPaneHeight_(::GetSystemMetrics(SM_CYMIN)),
			draggingSplitter_(0) {}
public:
	void activateNextPane() {doActivateNextPane(true);}
	void activatePreviousPane() {doActivateNextPane(false);}
	void adjustPanes() {
		RECT rect;
		getWindowRect(rect);
		::OffsetRect(&rect, -rect.left, -rect.top);
		root_.adjustPanes(rect, frameWidth_, frameHeight_);
		invalidateRect(0);
	}
	bool create(HWND parent, const RECT& rect, DWORD style, DWORD exStyle, Pane& initialPane) {
		assert(parent == 0 || toBoolean(::IsWindow(parent)));
		if(!CustomControl<SplitterRoot>::create(parent, rect, 0, style, exStyle))
			return false;
		::SetParent(static_cast<AbstractPane&>(initialPane).getWindow(), getHandle());
		root_.children_[PANEPOSITION_LEFT].type = Splitter::PANETYPE_SINGLE;
		root_.children_[PANEPOSITION_LEFT].body.pane = defaultActivePane_ = &initialPane;
		return true;
	}
	Iterator enumeratePanes() const {return Iterator(root_);}
	Pane& getActivePane() const {
		HWND focused = ::GetFocus();
		for(Iterator it = enumeratePanes(); !it.isEnd(); it.next()) {
			if(static_cast<AbstractPane&>(it.get()).getWindow() == focused)
				return it.get();
		}
		if(defaultActivePane_ != 0)
			return *defaultActivePane_;
		else
			throw std::logic_error("There are no panes.");
	}
	uint getSplitterSize(uint& width, uint& height) const throw() {width = splitterWidth_; height = splitterHeight_;}
	bool isSplit(const Pane& pane) const {
		if(Splitter* parent = findPane(root_, pane))
			return parent->direction_ != SPLITDIRECTION_NOSPLIT;
		else
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
	}
	void removeActivePane() {unsplit(getActivePane());}
	void removeInactivePanes() {
		Pane& activePane = getActivePane();
		Splitter* parent = findPane(root_, activePane);
		const bool hasFocus = isChild(::GetFocus());

		// prevent active pane from deletion
		assert(parent != 0);
		if(parent->children_[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SINGLE
				&& parent->children_[PANEPOSITION_LEFT].body.pane == &activePane)
			parent->children_[PANEPOSITION_LEFT].type = Splitter::PANETYPE_EMPTY;
		else {
			assert(parent->children_[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SINGLE
				&& parent->children_[PANEPOSITION_RIGHT].body.pane == &activePane);
			parent->children_[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
		}

		if(root_.children_[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SINGLE
				&& cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
			delete root_.children_[PANEPOSITION_LEFT].body.pane;
		else if(root_.children_[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SPLITTER)
			delete root_.children_[PANEPOSITION_LEFT].body.splitter;
		if(root_.direction_ != SPLITDIRECTION_NOSPLIT) {
			if(root_.children_[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SINGLE
					&& cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
				delete root_.children_[PANEPOSITION_RIGHT].body.pane;
			else if(root_.children_[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SPLITTER)
				delete root_.children_[PANEPOSITION_RIGHT].body.splitter;
		}

		// same as construction...
		root_.children_[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
		root_.direction_ = SPLITDIRECTION_NOSPLIT;
		root_.firstPaneSizeRatio_ = 0.5F;
		root_.children_[PANEPOSITION_LEFT] = Splitter::Child(activePane);
		defaultActivePane_ = &activePane;
		if(hasFocus && static_cast<AbstractPane&>(activePane).getWindow() != ::GetFocus())
			::SetFocus(static_cast<AbstractPane&>(activePane).getWindow());
		adjustPanes();
	}
	void setDefaultActivePane(Pane& pane) {
		if(findPane(root_, pane) == 0)
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
		defaultActivePane_ = &pane;
	}
	void setPaneMinimumSize(uint width, uint height) throw() {minimumPaneWidth_ = width; minimumPaneHeight_ = height;}
	void setSplitterSize(uint width, uint height) {
		splitterWidth_ = width;
		splitterHeight_ = height;
		adjustPanes();
	}
	void splitNS(Pane& pane, Pane& clone) {split(pane, clone, true);}
	void splitWE(Pane& pane, Pane& clone) {split(pane, clone, false);}
	void unsplit(Pane& pane) {
		Splitter* parent = findPane(root_, pane);

		if(parent == 0)
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
		else if(parent->direction_ == SPLITDIRECTION_NOSPLIT)
			throw std::invalid_argument("Specified pane is not split.");

		Splitter::Child& leftTop = parent->children_[PANEPOSITION_LEFT];
		Splitter::Child& rightBottom = parent->children_[PANEPOSITION_RIGHT];
		const bool removedPaneWasDefaultActive = &pane == defaultActivePane_;
		const bool removedPaneHadFocus = static_cast<AbstractPane&>(pane).getWindow() == ::GetFocus();

		// swap panes if deleting pane is left/top
		if(leftTop.type == Splitter::PANETYPE_SINGLE && leftTop.body.pane == &pane)
			std::swap(leftTop, rightBottom);
		assert(rightBottom.type == Splitter::PANETYPE_SINGLE && rightBottom.body.pane == &pane);

		Pane* nextFirstPane = 0;
		if(removedPaneWasDefaultActive || removedPaneHadFocus) {
			if(leftTop.type == Splitter::PANETYPE_SINGLE)
				nextFirstPane = leftTop.body.pane;
			else
				leftTop.body.splitter->getFirstPane(true, nextFirstPane);
		}

		if(cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
			delete rightBottom.body.pane;

		// there are 4 scenarios...
		if(parent == &root_) {	// parent is root
			if(leftTop.type == Splitter::PANETYPE_SINGLE) {	// live pane is a single pane
				rightBottom = Splitter::Child();
				root_.direction_ = SPLITDIRECTION_NOSPLIT;
				root_.firstPaneSizeRatio_ = 0.5F;
			} else {	// live pane is a splitter
				Splitter& liveSplitter = *leftTop.body.splitter;
				root_.direction_ = liveSplitter.direction_;
				root_.firstPaneSizeRatio_ = liveSplitter.firstPaneSizeRatio_;
				leftTop.type = liveSplitter.children_[PANEPOSITION_LEFT].type;
				leftTop.body = liveSplitter.children_[PANEPOSITION_LEFT].body;
				rightBottom.type = liveSplitter.children_[PANEPOSITION_RIGHT].type;
				rightBottom.body = liveSplitter.children_[PANEPOSITION_RIGHT].body;

				if(liveSplitter.children_[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SPLITTER)
					liveSplitter.children_[PANEPOSITION_LEFT].body.splitter->parent_ = &root_;
				if(liveSplitter.children_[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SPLITTER)
					liveSplitter.children_[PANEPOSITION_RIGHT].body.splitter->parent_ = &root_;
				liveSplitter.children_[PANEPOSITION_LEFT].type
					= liveSplitter.children_[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
				delete &liveSplitter;	// this splitter should delete no children
			}
		} else {	// parent is not root
			Splitter& grandParent = *parent->parent_;
			const bool parentIsLeftChildOfGrandParent =
				parent == grandParent.children_[PANEPOSITION_LEFT].body.splitter;
			Splitter::Child& replacedChild =
				grandParent.children_[parentIsLeftChildOfGrandParent ? PANEPOSITION_LEFT : PANEPOSITION_RIGHT];
			if(leftTop.type == Splitter::PANETYPE_SINGLE) {	// live pane is a single pane
				replacedChild.type = Splitter::PANETYPE_SINGLE;
				replacedChild.body.pane = leftTop.body.pane;
			} else {	// live pane is a splitter
				replacedChild.type = Splitter::PANETYPE_SPLITTER;
				replacedChild.body.splitter = leftTop.body.splitter;
				leftTop.body.splitter->parent_ = &grandParent;
			}
			parent->children_[PANEPOSITION_LEFT].type
				= parent->children_[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
			delete parent;	// this splitter should delete no children
		}

		adjustPanes();

		if(removedPaneWasDefaultActive)	defaultActivePane_ = nextFirstPane;
		if(removedPaneHadFocus)			::SetFocus(static_cast<AbstractPane*>(nextFirstPane)->getWindow());
	}
protected:
	LRESULT	dispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
		if(message == WM_CAPTURECHANGED)
			onCaptureChanged(reinterpret_cast<HWND>(lParam));
		return CustomControl<SplitterRoot>::dispatchEvent(message, wParam, lParam);
	}
//	friend class Manah::Windows::Controls::CustomControl<SplitterRoot>;
	void onCaptureChanged(HWND newWindow) {
		if(draggingSplitter_ != 0) {
			root_.sendMessageToChildren(WM_EXITSIZEMOVE);
			if(sizingFirstPaneSize_ != -1) {
				const RECT&	rect = draggingSplitter_->rect_;
				drawSizingSplitterXorBar();	// erase ghost bar
				draggingSplitter_->firstPaneSizeRatio_ =
					(draggingSplitter_->direction_ == SPLITDIRECTION_NS) ?
					(sizingFirstPaneSize_ + frameHeight_ / 2) / static_cast<double>(rect.bottom - rect.top)
					: (sizingFirstPaneSize_ + frameWidth_ / 2) / static_cast<double>(rect.right - rect.left);
				draggingSplitter_->adjustPanes(rect, frameWidth_, frameHeight_);
			}
			draggingSplitter_ = 0;
		}
	}
	void onDestroy() {
		if(cdp == CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL) {
			for(Iterator it = enumeratePanes(); !it.isEnd(); it.next())
				::SetParent(static_cast<AbstractPane&>(it.get()).getWindow(), 0);
		}
	}
	void onLButtonDown(UINT, const POINT& pt) {
		// begin sizing
		if(root_.direction_ == SPLITDIRECTION_NOSPLIT)
			return;
		else if(draggingSplitter_ = root_.hitTest(pt)) {
			const RECT& rect = draggingSplitter_->rect_;
			BOOL fullDraggingEnabled;

			::SystemParametersInfo(SPI_GETDRAGFULLWINDOWS, 0, &fullDraggingEnabled, 0);
			if(toBoolean(fullDraggingEnabled))
				sizingFirstPaneSize_ = -1;
			else if(draggingSplitter_->direction_ == SPLITDIRECTION_NS) {
				sizingFirstPaneSize_ =
					static_cast<uint>((rect.bottom - rect.top) * draggingSplitter_->firstPaneSizeRatio_)
					- frameHeight_ / 2;
				drawSizingSplitterXorBar();
			} else {
				assert(draggingSplitter_->direction_ == SPLITDIRECTION_WE);
				sizingFirstPaneSize_ =
					static_cast<uint>((rect.right - rect.left) * draggingSplitter_->firstPaneSizeRatio_)
					- frameWidth_ / 2;
				drawSizingSplitterXorBar();
			}
			setCapture();
			root_.sendMessageToChildren(WM_ENTERSIZEMOVE);
		}
	}
	void onLButtonDblClk(UINT, const POINT& pt) {
		// double click splitter bar to unsplit
		if(root_.direction_ == SPLITDIRECTION_NOSPLIT)
			return;
		else if(Splitter* splitter = root_.hitTest(pt)) {
			if(splitter->children_[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SINGLE)
				unsplit(*splitter->children_[PANEPOSITION_LEFT].body.pane);
			else if(splitter->children_[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SINGLE)
				unsplit(*splitter->children_[PANEPOSITION_RIGHT].body.pane);
		}
	}
	void onLButtonUp(UINT, const POINT&) {releaseCapture();}
	void onMouseMove(UINT, const POINT& pt) {
		if(draggingSplitter_ == 0)
			return;
		const RECT& rect = draggingSplitter_->rect_;
		double ratio;

		if(draggingSplitter_->direction_ == SPLITDIRECTION_NS) {
			if(static_cast<uint>(rect.bottom - rect.top) <= minimumPaneHeight_ * 2 + frameHeight_)	// too short
				ratio = 0.5F;
			else {
				long y = std::max(pt.y, rect.top + static_cast<long>(minimumPaneHeight_));
				y = std::min(y, rect.bottom - static_cast<long>(minimumPaneHeight_ + frameHeight_));
				ratio = (rect.top != rect.bottom) ?
					static_cast<double>(y - rect.top) / static_cast<double>(rect.bottom - rect.top) : 0.5F;
			}
		} else {
			assert(draggingSplitter_->direction_ == SPLITDIRECTION_WE);
			if(static_cast<uint>(rect.right - rect.left) <= minimumPaneWidth_ * 2 + frameWidth_)	// too narrow
				ratio = 0.5F;
			else {
				long x = std::max(pt.x, rect.left + static_cast<long>(minimumPaneWidth_));
				x = std::min(x, rect.right - static_cast<long>(minimumPaneWidth_ + frameWidth_));
				ratio = (rect.left != rect.right) ?
					static_cast<double>(x - rect.left) / static_cast<double>(rect.right - rect.left) : 0.5F;
			}
		}

		if(sizingFirstPaneSize_ == -1) {	// dynamic sizing
			draggingSplitter_->firstPaneSizeRatio_ = ratio;
			draggingSplitter_->adjustPanes(rect, frameWidth_, frameHeight_);
			invalidateRect(0);
		} else {	// static sizing
			drawSizingSplitterXorBar();	// erase previous ghost
			sizingFirstPaneSize_ = (draggingSplitter_->direction_ == SPLITDIRECTION_NS) ?
				static_cast<uint>((rect.bottom - rect.top) * ratio) : static_cast<uint>((rect.right - rect.left) * ratio);
			drawSizingSplitterXorBar();	// draw new ghost
		}
	}
	void onPaint(GDI::PaintDC& dc) {root_.draw(dc, frameWidth_, frameHeight_);}
	bool onSetCursor(HWND window, UINT hitTest, UINT) {
		if(window == getHandle() && hitTest == HTCLIENT) {
			POINT pt;
			if(toBoolean(::GetCursorPos(&pt))) {
				screenToClient(pt);
				if(Splitter* splitter = root_.hitTest(pt)) {
					if(splitter->direction_ != SPLITDIRECTION_NOSPLIT) {
						::SetCursor(::LoadCursor(0,
							(splitter->direction_ == SPLITDIRECTION_NS) ? IDC_SIZENS : IDC_SIZEWE));
						return true;
					}
				}
			}
		}
		return false;
	}
	void onSetFocus(HWND) {
		if(defaultActivePane_ != 0)
			::SetFocus(static_cast<AbstractPane*>(defaultActivePane_)->getWindow());
	}
	void onSize(UINT, int cx, int cy) {
		HWND window = getHandle(), parent = ::GetParent(getHandle());
		do {
			if(toBoolean(::IsIconic(window)))	// ignore if the window is iconic
				return;
			window = parent;
			parent = ::GetParent(window);
		} while(parent != 0);

		RECT rect = {0, 0, cx, cy};
		root_.adjustPanes(rect, frameWidth_, frameHeight_);
		invalidateRect(0);
	}
private:
	typedef Private::Splitter<Pane, cdp>	Splitter;
	void doActivateNextPane(bool next) {
		Pane& activePane = getActivePane();

		if(Splitter* parent = findPane(root_, activePane)) {
			Pane* nextPane = 0;
			if(parent->getNextPane(activePane, next, nextPane))
				defaultActivePane_ = nextPane;
			else	// current active pane is end/first
				root_.getFirstPane(next, defaultActivePane_);
			if(HWND window = static_cast<AbstractPane*>(defaultActivePane_)->getWindow())
				::SetFocus(window);
		}
	}
	void drawSizingSplitterXorBar() {
		RECT rect;
		const RECT&	splitterRect = draggingSplitter_->rect_;
		Manah::Windows::GDI::WindowDC dc = getWindowDC();

		// create half tone brush (from MFC)
		ushort grayPattern[8];
		for(int i = 0; i < countof(grayPattern); ++i)
			grayPattern[i] = static_cast<ushort>(0x5555 << (i & 1));
		HBITMAP bitmap = ::CreateBitmap(countof(grayPattern), countof(grayPattern), 1, 1, grayPattern);
		HBRUSH brush = ::CreatePatternBrush(bitmap), oldBrush;

		assert(draggingSplitter_ != 0 && sizingFirstPaneSize_ != -1);
		if(draggingSplitter_->direction_ == SPLITDIRECTION_NS)
			::SetRect(&rect, splitterRect.left, splitterRect.top + sizingFirstPaneSize_,
				splitterRect.right, splitterRect.top + sizingFirstPaneSize_ + frameHeight_);
		else {
			assert(draggingSplitter_->direction_ == SPLITDIRECTION_WE);
			::SetRect(&rect, splitterRect.left + sizingFirstPaneSize_, splitterRect.top,
				splitterRect.left + sizingFirstPaneSize_ + frameWidth_, splitterRect.bottom);
		}
		oldBrush = dc.selectObject(brush);
		dc.patBlt(rect, PATINVERT);
		dc.selectObject(oldBrush);
		::DeleteObject(brush);
		::DeleteObject(grayPattern);
	}
	static Splitter* findPane(const Splitter& splitter, const Pane& pane) {	// recursive method which returns pane's parent
		const Splitter::Child& leftTop = splitter.children_[PANEPOSITION_LEFT];
		const Splitter::Child& rightBottom = splitter.children_[PANEPOSITION_RIGHT];

		if(leftTop.type == Splitter::PANETYPE_EMPTY)
			return 0;
		else if(leftTop.type == Splitter::PANETYPE_SINGLE && leftTop.body.pane == &pane)
			return const_cast<Splitter*>(&splitter);
		else if(leftTop.type == Splitter::PANETYPE_SPLITTER) {
			if(Splitter* p = findPane(*leftTop.body.splitter, pane))
				return p;
		}

		if(rightBottom.type == Splitter::PANETYPE_EMPTY)
			return 0;
		else if(rightBottom.type == Splitter::PANETYPE_SINGLE && rightBottom.body.pane == &pane)
			return const_cast<Splitter*>(&splitter);
		else if(rightBottom.type == Splitter::PANETYPE_SPLITTER) {
			if(Splitter* p = findPane(*rightBottom.body.splitter, pane))
				return p;
		}

		return 0;
	}
	void split(Pane& pane, Pane& clone, bool ns) {
		Splitter* const parent = findPane(root_, pane);
		const pos1 = ns ? PANEPOSITION_TOP : PANEPOSITION_LEFT;	// no matter...
		const pos2 = ns ? PANEPOSITION_BOTTOM : PANEPOSITION_RIGHT;

		if(parent == 0)
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
		if(parent->children_[pos2].type == Splitter::PANETYPE_EMPTY) {
			// there is another empty pane
			assert(parent->direction_ == SPLITDIRECTION_NOSPLIT);
			assert(parent->children_[pos1].type == Splitter::PANETYPE_SINGLE);
			assert(parent->children_[pos1].body.pane == &pane);
			parent->children_[pos2] = Splitter::Child(clone);
			parent->direction_ = ns ? SPLITDIRECTION_NS : SPLITDIRECTION_WE;
		} else {
			// this splitter is already split
			Splitter::Child* child = 0;

			assert(parent->direction_ != SPLITDIRECTION_NOSPLIT);
			if(parent->children_[pos1].body.pane == &pane)
				child = &parent->children_[pos1];
			else if(parent->children_[pos2].body.pane == &pane)
				child = &parent->children_[pos2];

			assert(child != 0);
			assert(child->type == Splitter::PANETYPE_SINGLE);
			child->type = Splitter::PANETYPE_SPLITTER;
			child->body.splitter = new Splitter;
			child->body.splitter->parent_ = parent;
			child->body.splitter->direction_ = ns ? SPLITDIRECTION_NS : SPLITDIRECTION_WE;
			child->body.splitter->children_[PANEPOSITION_LEFT] = Splitter::Child(pane);
			child->body.splitter->children_[PANEPOSITION_RIGHT] = Splitter::Child(clone);
		}
		adjustPanes();
	}
private:
	Splitter root_;
	Pane* defaultActivePane_;
	uint frameWidth_, frameHeight_;
	uint minimumPaneWidth_, minimumPaneHeight_;
	Splitter* draggingSplitter_;	// sizing splitter
	uint sizingFirstPaneSize_;	// first pane size of m_pDraggingSplitter (-1 when full dragging is enabled)
};


namespace Private {
template<class Pane /* implements AbstractPane */, ChildrenDestructionPolicy cdp>
class Splitter : public Noncopyable {
private:
	Splitter() : parent_(0), direction_(SPLITDIRECTION_NOSPLIT), firstPaneSizeRatio_(0.5F) {}
	~Splitter() {
		Child& leftTop = children_[PANEPOSITION_LEFT];
		Child& rightBottom = children_[PANEPOSITION_RIGHT];

		if(leftTop.type == PANETYPE_SINGLE) {
			if(cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
				delete leftTop.body.pane;
			else if(cdp == CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL)
				::SetParent(static_cast<AbstractPane*>(leftTop.body.pane)->getWindow(), 0);
		} else if(leftTop.type == PANETYPE_SPLITTER)
			delete leftTop.body.splitter;
		if(rightBottom.type == PANETYPE_SINGLE) {
			if(cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
				delete rightBottom.body.pane;
			else if(cdp == CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL)
				::SetParent(static_cast<AbstractPane*>(rightBottom.body.pane)->getWindow(), 0);
		} else if(rightBottom.type == PANETYPE_SPLITTER)
			delete rightBottom.body.splitter;
	}
	enum PaneType {PANETYPE_EMPTY, PANETYPE_SINGLE, PANETYPE_SPLITTER};
private:
	void adjustPanes(const RECT& newRect, uint frameWidth, uint frameHeight) {
		rect_ = newRect;
		if(direction_ == SPLITDIRECTION_NOSPLIT) {	// is not split
			if(children_[PANEPOSITION_LEFT].type != PANETYPE_EMPTY)
				::MoveWindow(
					static_cast<AbstractPane&>(*children_[PANEPOSITION_LEFT].body.pane).getWindow(),
					newRect.left, newRect.top,
					newRect.right - newRect.left, newRect.bottom - newRect.top, true);
		} else {
			RECT rect = newRect;

			if(direction_ == SPLITDIRECTION_NS)
				rect.bottom = newRect.top +
					static_cast<long>((newRect.bottom - newRect.top) * firstPaneSizeRatio_) - frameHeight / 2;
			else
				rect.right = newRect.left +
					static_cast<long>((newRect.right - newRect.left) * firstPaneSizeRatio_) - frameWidth / 2;
			if(children_[PANEPOSITION_LEFT].type == PANETYPE_SINGLE)
				::MoveWindow(static_cast<AbstractPane*>(children_[PANEPOSITION_LEFT].body.pane)->getWindow(),
					rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top, true);
			else
				children_[PANEPOSITION_LEFT].body.splitter->adjustPanes(rect, frameWidth, frameHeight);

			if(direction_ == SPLITDIRECTION_NS) {
				rect.top = rect.bottom + frameHeight;
				rect.bottom = newRect.bottom;
			} else {
				rect.left = rect.right + frameWidth;
				rect.right = newRect.right;
			}
			if(children_[PANEPOSITION_RIGHT].type == PANETYPE_SINGLE)
				::MoveWindow(static_cast<AbstractPane*>(children_[PANEPOSITION_RIGHT].body.pane)->getWindow(),
					rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top, true);
			else
				children_[PANEPOSITION_RIGHT].body.splitter->adjustPanes(rect, frameWidth, frameHeight);
		}
	}
	void draw(GDI::PaintDC& dc, uint frameWidth, uint frameHeight) {
		if(direction_ == SPLITDIRECTION_NOSPLIT)
			return;

		RECT rect;

		rect.left = (direction_ == SPLITDIRECTION_NS) ? rect_.left :
			rect_.left + static_cast<long>((rect_.right - rect_.left) * firstPaneSizeRatio_) - frameWidth / 2;
		rect.top = (direction_ == SPLITDIRECTION_WE) ? rect_.top :
			rect_.top + static_cast<long>((rect_.bottom - rect_.top) * firstPaneSizeRatio_) - frameHeight / 2;
		rect.right = (direction_ == SPLITDIRECTION_NS) ? rect_.right : rect.left + frameWidth;
		rect.bottom = (direction_ == SPLITDIRECTION_WE) ? rect_.bottom : rect.top + frameHeight;
		dc.fillSolidRect(rect, ::GetSysColor(COLOR_3DFACE));

		if(children_[PANEPOSITION_LEFT].type == PANETYPE_SPLITTER)
			children_[PANEPOSITION_LEFT].body.splitter->draw(dc, frameWidth, frameHeight);
		if(children_[PANEPOSITION_RIGHT].type == PANETYPE_SPLITTER)
			children_[PANEPOSITION_RIGHT].body.splitter->draw(dc, frameWidth, frameHeight);
	}
	void getFirstPane(bool leftTop, Pane*& pane, Splitter<Pane, cdp>** parent = 0) const {
		Splitter<Pane, cdp>* p = const_cast<Splitter<Pane, cdp>*>(this);
		const PanePosition seekDirection = leftTop ? PANEPOSITION_LEFT : PANEPOSITION_RIGHT;
		pane = 0;
		do {
			if(p->children_[seekDirection].type == PANETYPE_SINGLE) {
				pane = p->children_[seekDirection].body.pane;
				assert(pane != 0);
			} else {
				assert(p->children_[seekDirection].type == PANETYPE_SPLITTER);
				p = p->children_[seekDirection].body.splitter;
			}
		} while(pane == 0);
		if(parent != 0)
			*parent = p;
	}
	bool getNextPane(const Pane& pane, bool next, Pane*& nextPane, Splitter<Pane, cdp>** parent = 0) const {
		// returns false if pane is rightmost
		const PanePosition forwardPosition = next ? PANEPOSITION_RIGHT : PANEPOSITION_LEFT;
		const PanePosition backPosition = next ? PANEPOSITION_LEFT : PANEPOSITION_RIGHT;

		if(children_[backPosition].type == PANETYPE_SINGLE && children_[backPosition].body.pane == &pane) {
			if(children_[forwardPosition].type == PANETYPE_SINGLE) {
				nextPane = children_[forwardPosition].body.pane;
				if(parent != 0)	*parent = const_cast<Splitter<Pane, cdp>*>(this);
				return true;
			} else if(children_[forwardPosition].type == PANETYPE_SPLITTER) {
				children_[forwardPosition].body.splitter->getFirstPane(next, nextPane, parent);
				return true;
			} else {
				assert(children_[forwardPosition].type == PANETYPE_EMPTY);
				return false;
			}
		}

		// up one level
		assert(children_[forwardPosition].type == PANETYPE_SINGLE && children_[forwardPosition].body.pane == &pane);
		Splitter* childSplitter = const_cast<Splitter<Pane, cdp>*>(this);
		Splitter* p = parent_;

		if(p == 0)	// `this' is root
			return false;
		while(p != 0) {
			if(p->children_[backPosition].type == PANETYPE_SPLITTER
					&& p->children_[backPosition].body.splitter == childSplitter) {
				if(p->children_[forwardPosition].type == PANETYPE_SINGLE) {
					nextPane = p->children_[forwardPosition].body.pane;
					if(parent != 0)	*parent = p;
					return true;
				} else if(p->children_[forwardPosition].type == PANETYPE_SPLITTER) {
					p->children_[forwardPosition].body.splitter->getFirstPane(next, nextPane, parent);
					return true;
				} else {
					assert(p->children_[forwardPosition].type == PANETYPE_EMPTY);
					break;
				}
			} else {
				// up one level
				assert(p->children_[forwardPosition].type == PANETYPE_SPLITTER
					&& p->children_[forwardPosition].body.splitter == childSplitter);
				childSplitter = p;
				p = p->parent_;
			}
		}
		return false;
	}
	Splitter<Pane, cdp>* hitTest(const POINT& pt) const {
		if(!toBoolean(::PtInRect(&rect_, pt)))
			return 0;
		if(children_[PANEPOSITION_LEFT].type == PANETYPE_SPLITTER) {
			if(Splitter<Pane, cdp>* p = children_[PANEPOSITION_LEFT].body.splitter->hitTest(pt))
				return p;
		}
		if(children_[PANEPOSITION_RIGHT].type == PANETYPE_SPLITTER) {
			if(Splitter<Pane, cdp>* p = children_[PANEPOSITION_RIGHT].body.splitter->hitTest(pt))
				return p;
		}
		return const_cast<Splitter<Pane, cdp>*>(this);
	}
	void sendMessageToChildren(UINT message) {
		Child& leftTop = children_[PANEPOSITION_LEFT];
		Child& rightBottom = children_[PANEPOSITION_RIGHT];

		if(leftTop.type == PANETYPE_SINGLE)
			::SendMessage(static_cast<AbstractPane*>(leftTop.body.pane)->getWindow(), message, 0, 0L);
		else if(leftTop.type == PANETYPE_SPLITTER)
			leftTop.body.splitter->sendMessageToChildren(message);
		if(rightBottom.type == PANETYPE_SINGLE)
			::SendMessage(static_cast<AbstractPane*>(rightBottom.body.pane)->getWindow(), message, 0, 0L);
		else if(rightBottom.type == PANETYPE_SPLITTER)
			rightBottom.body.splitter->sendMessageToChildren(message);
	}
private:
	struct Child {
		PaneType type;
		union {
			Pane* pane;
			Splitter<Pane, cdp>* splitter;
		} body;
		Child() : type(PANETYPE_EMPTY) {}
		Child(Pane& pane) : type(PANETYPE_SINGLE) {body.pane = &pane;}
	} children_[PANEPOSITION_COUNT];	// children ([RIGHT] is EMPTY if this pane is not split)
	Splitter<Pane, cdp>* parent_;		// parent splitter
	SplitDirection direction_;			// SPLITDIRECTION_NOSPLIT for only root
	RECT rect_;
	double firstPaneSizeRatio_;			// (left or top pane's size) / (whole splitter size)

	friend class SplitterRoot<Pane, cdp>;
	friend class SplitterRoot<Pane, cdp>::Iterator;
};

} /* namespace Private */
} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* SPLITTER_HPP_ */

/* [EOF] */