// DosCommand.h
// (c) 2003-2005 exeal

#ifndef _DOS_COMMAND_H_
#define _DOS_COMMAND_H_

#include "Object.h"
#include <list>
#include <string>
#include <sstream>
#include <stdexcept>


namespace Manah {
namespace Windows {

// CDosCommand class definition
/////////////////////////////////////////////////////////////////////////////

class CDosCommand : public CSelfAssertable {
public:
	// enum
	enum ExecuteStatus {
		succeeded,
		timeout,
		aborted,
		failedToCreatePipes,
		failedToCreateNewProcess,
		failedToCreateNewThread
	};

	// internal structure
	struct TExecuteParam {
		std::tstring	strCmdLine;
		bool			bGetOutput;
		CDosCommand*	pDosCommand;
	};

	// internal interface
	interface ICommandEvent {
		virtual ~ICommandEvent() {}
		virtual void OnExit(ExecuteStatus es) = 0;
		virtual void OnReceiveOutput(const std::string& strOutput) = 0;
	};

	// constructors
public:
	CDosCommand() : m_bIncludeInput(true), m_dwTimeout(0), m_bStopped(false), m_hThread(0), m_pEventSink(0) {}
	virtual ~CDosCommand() {if(m_hThread != 0) OnExit(timeout);}

	// methods
public:
	void	Execute(const std::tstring& strCmdLine, bool bGetOutput, ICommandEvent* pEventSink = 0);
	void	SendLine(const std::string& strLine);
	void	SetIncludingInputToOutput(bool bInclude = true);
	void	SetTimeout(DWORD dwTimeout = 0);
	void	Stop();
protected:
	static unsigned int WINAPI ConsoleThread(void* p) throw(std::runtime_error);
private:
	virtual void OnExit(ExecuteStatus es) {
		if(m_hThread != 0) {
			if(m_pEventSink != 0)
				m_pEventSink->OnExit(es);
			::CloseHandle(m_hThread);
			m_hThread = 0;
		}
	}
	virtual void OnReceiveOutput(const std::string& strOutput) {
		if(m_pEventSink != 0)
			m_pEventSink->OnReceiveOutput(strOutput);
	}

	// data members
private:
	bool					m_bIncludeInput;	// if include input to output
	std::list<std::string>	m_listInputs;		// lines to write to standard input
	DWORD					m_dwTimeout;		// timeout (millisecond). 0 to infinity
	bool					m_bStopped;			// if stopped loop by client
	HANDLE					m_hThread;
	unsigned int			m_nThreadId;
	ICommandEvent*			m_pEventSink;
};


// CDosCommand class implementation
/////////////////////////////////////////////////////////////////////////////

inline unsigned int WINAPI CDosCommand::ConsoleThread(void* p) throw(std::runtime_error) {
	assert(p != 0);

	TExecuteParam		ep = *reinterpret_cast<CDosCommand::TExecuteParam*>(p);
	CDosCommand*		pInstance = ep.pDosCommand;
	TCHAR*				pszCmdLine = 0;
	HANDLE				hReadPipe, hWritePipe;
	HANDLE				hStdInReadPipe, hStdInWritePipe;
	OSVERSIONINFO		ovi;
	STARTUPINFO			si;
	PROCESS_INFORMATION	pi;
	SECURITY_ATTRIBUTES	sa = {
		sizeof(SECURITY_ATTRIBUTES), 0, true
	};

	// delete CDosCommand::Execute local epe
	delete reinterpret_cast<CDosCommand::TExecuteParam*>(p);

	if(!::CreatePipe(&hStdInReadPipe, &hStdInWritePipe, &sa, 0)) {
		pInstance->OnExit(failedToCreatePipes);
		return 0;
	}
	if(!::CreatePipe(&hReadPipe, &hWritePipe, &sa, 0)) {
		::CloseHandle(hStdInReadPipe);
		::CloseHandle(hStdInWritePipe);
		pInstance->OnExit(failedToCreatePipes);
		return 0;
	}

	// make uninheritable
//	::DuplicateHandle(::GetCurrentProcess(),
//		hReadPipe, ::GetCurrentProcess(), 0, 0, false, DUPLICATE_SAME_ACCESS);

	ZeroMemory(&ovi, sizeof(OSVERSIONINFO));
	ovi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));
	ZeroMemory(&si, sizeof(STARTUPINFO));
	si.cb = sizeof(STARTUPINFO);
	si.dwFlags = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
	si.wShowWindow = SW_HIDE;
	si.hStdInput = hStdInReadPipe;
	si.hStdOutput = hWritePipe;
	si.hStdError = hWritePipe;

	pszCmdLine = new TCHAR[ep.strCmdLine.length() + 12];
	std::_tcscpy(pszCmdLine, ep.strCmdLine.c_str());

	if(!::CreateProcess(0, pszCmdLine, 0, 0, true, CREATE_NEW_CONSOLE, 0, 0, &si, &pi)) {
		// command-line based application?
		::GetVersionEx(&ovi);
		wsprintf(pszCmdLine,
			((ovi.dwPlatformId == VER_PLATFORM_WIN32_NT) ? _T("cmd /C %s") : _T("command /C %s")),
			ep.strCmdLine.c_str());
		if(!::CreateProcess(0, pszCmdLine, 0, 0, true, CREATE_NEW_CONSOLE, 0, 0, &si, &pi)) {
			delete[] pszCmdLine;
			::CloseHandle(hStdInReadPipe);
			::CloseHandle(hStdInWritePipe);
			::CloseHandle(hReadPipe);
			::CloseHandle(hWritePipe);
			pInstance->OnExit(failedToCreateNewProcess);
			return 0;
		}
	}
	delete[] pszCmdLine;
	pszCmdLine = 0;

	// return if don't get output.
	if(!ep.bGetOutput) {
		pInstance->OnExit(succeeded);
		return 0;
	}

	DWORD	dwResult = ::WaitForInputIdle(pi.hProcess,
						(pInstance->m_dwTimeout != 0) ? pInstance->m_dwTimeout : INFINITE);
	if(dwResult == WAIT_TIMEOUT) {	// timeouted
		::CloseHandle(hStdInReadPipe);
		::CloseHandle(hStdInWritePipe);
		::CloseHandle(hReadPipe);
		::CloseHandle(hWritePipe);
		::TerminateProcess(pi.hProcess, 1);
		pInstance->OnExit(CDosCommand::timeout);
		return 0;
	}

	// loop
	ExecuteStatus	es = succeeded;
	bool			bExpectExit = false;
	bool			bEmptyRead = false;		// if there is no data in output pipe
	DWORD			dwStartCount = (pInstance->m_dwTimeout != 0) ? ::GetTickCount() : 0;
	DWORD			dwExitCode;
	DWORD			dwAvailableSize = 0;	// total bytes can read from pipe
	DWORD			dwReadSize;				// bytes read from pipe
	char*			pszBuffer = 0;			// data read from pipe
	DWORD			dwWrittenSize;			// bytes written to pipe
	while(!bExpectExit) {
		// check client
		if(pInstance->m_bStopped) {
			es = CDosCommand::aborted;
			::TerminateProcess(pi.hProcess, 1);
			break;
		}

		// check timeout
		if(pInstance->m_dwTimeout != 0
				&& ::GetTickCount() - dwStartCount >= pInstance->m_dwTimeout) {
			es = CDosCommand::timeout;
			::TerminateProcess(pi.hProcess, 1);
			break;
		}

		// check if the process is terminated
		::GetExitCodeProcess(pi.hProcess, &dwExitCode);
		if(dwExitCode != STILL_ACTIVE)
			bExpectExit = true;
//		if(::WaitForSingleObject(pi.hProcess, 0) == WAIT_OBJECT_0)
//			bExpectExit = true;

		// try to read data from output pipe
		if(::PeekNamedPipe(hReadPipe, 0, 0, 0, &dwAvailableSize, 0)) {
			if(dwAvailableSize > 0) {	// there are data in pipe
				dwReadSize = 0;
				pszBuffer = new char[dwAvailableSize + 1];
				if(!::ReadFile(hReadPipe, pszBuffer, dwAvailableSize, &dwReadSize, 0) || dwReadSize == 0)
					bEmptyRead = true;
				else
					pInstance->OnReceiveOutput(std::string(pszBuffer, dwReadSize));
				delete[] pszBuffer;
			} else
				bEmptyRead = true;
		}

		// try to write data to input pipe
		if(bEmptyRead) {
			bEmptyRead = false;
			for(std::list<std::string>::const_iterator it =
					pInstance->m_listInputs.begin(); it != pInstance->m_listInputs.end(); ++it) {
				::WriteFile(hStdInWritePipe, it->c_str(), it->length(), &dwWrittenSize, 0);
				::WriteFile(hStdInWritePipe, "\n", 1, &dwWrittenSize, 0);
			//	::FlushConsoleInputBuffer(hWritePipe);
			}
			pInstance->m_listInputs.clear();
		}

		::Sleep(1);
	}

	::CloseHandle(hStdInReadPipe);
	::CloseHandle(hStdInWritePipe);
	::CloseHandle(hReadPipe);
	::CloseHandle(hWritePipe);
	if(pi.hProcess != 0)
		::CloseHandle(pi.hProcess);
	if(pi.hThread != 0)
		::CloseHandle(pi.hThread);

	pInstance->OnExit(es);
	return 0;
}

/**
 *	Execute specified command.
 *	@param strCmdLine	Commnd line.
 *	@param bGetOutput	True to get output string.
 *	@param pEventSink	Event sink to get CDosCommand event. Can be null.
 */
inline void CDosCommand::Execute(const std::tstring& strCmdLine,
		bool bGetOutput, ICommandEvent* pEventSink /* = 0 */) {
	TExecuteParam*	pep = new TExecuteParam;

	pep->strCmdLine = strCmdLine;
	pep->bGetOutput = bGetOutput;
	pep->pDosCommand = this;
	m_bStopped = false;
	m_pEventSink = pEventSink;
	m_hThread = reinterpret_cast<HANDLE>(::_beginthreadex(0, 0, ConsoleThread, pep, 0, &m_nThreadId));
	if(m_hThread == 0 && m_pEventSink != 0)
		m_pEventSink->OnExit(failedToCreateNewThread);
}

/**
 *	Send lind to input.
 *	@param strLine	Line.
 */
inline void CDosCommand::SendLine(const std::string& strLine) {
	m_listInputs.push_back(strLine);
}

inline void CDosCommand::SetIncludingInputToOutput(bool bInclude /* = true */) {
	m_bIncludeInput = bInclude;
}

/**
 *	Set timeout.
 *	@param dwTimeout	Timeout in millisecond. 0 to be infinite.
 */
inline void CDosCommand::SetTimeout(DWORD dwTimeout /* = 0 */) {
	m_dwTimeout = dwTimeout;
}

///	Stop running process.
inline void CDosCommand::Stop() {
	m_bStopped = true;
}

} /* namespace Windows */
} /* namespace Manah */

#endif /* _DOS_COMMAND_H_ */

/* [EOF] */