// Object.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _OBJECT_H_
#define _OBJECT_H_

#include <cassert>
#include <string>
#include <sstream>
#include <memory>
#define size_t	std::size_t
#include <windows.h>
#include <tchar.h>
#undef size_t

#ifdef _UNICODE
#define tstring	wstring
#else
#define tstring	string
#endif

namespace Manah {
	namespace Windows {
		class CDumpContext;
	}
}


// modern types
/////////////////////////////////////////////////////////////////////////////

typedef unsigned char	uchar;
typedef unsigned short	ushort;
typedef unsigned int	uint;
typedef unsigned long	ulong;


// basic classes
/////////////////////////////////////////////////////////////////////////////

namespace Manah {

	class CSelfAssertable {
	protected:
		virtual void AssertValid() const {
#ifdef _DEBUG
			assert(this != 0);
#endif /* _DEBUG */
		}
	};


	// unassignable object's base class
	class CUnassignable {
	protected:
		CUnassignable() {}
		CUnassignable(const CUnassignable& rhs) {}
		~CUnassignable() {}
	private:
		const CUnassignable& operator =(const CUnassignable&);
	};


	// noncopyable object's base class (from Boost.Noncopyable)
	class CNoncopyable {
	protected:
		CNoncopyable() {}
		~CNoncopyable() {}
	private:
		CNoncopyable(const CNoncopyable&);
		const CNoncopyable& operator =(const CNoncopyable&);
	};


	// std::auto_ptr alternative for VC6's broken one
	template<class T> class _CPoolHolder {
	public:
		_CPoolHolder() : m_p(0) {}
		~_CPoolHolder() {delete m_p;}
		void operator =(T* p) {delete m_p; m_p = p;}
		T* operator ->() {return m_p;}
	private:
		T*	m_p;
	};

	// efficient memory pool implementation (from MemoryPool of Efficient C++).
	// template paramater of Allocator always bound to char
	template<class T, std::size_t nExpansionChunkCount = 32, template<class> class Allocator = std::allocator>
	class CUseMemoryPool {
	public:
		static void* operator new(std::size_t cb) {
			if(m_pPool.operator->() == 0)
				m_pPool = new _CMemoryPool;
			return m_pPool->Allocate(cb);
		}
		static void operator delete(void* p, std::size_t) {m_pPool->Deallocate(p);}
	private:
		class _CMemoryPool {
		public:
			_CMemoryPool() {_ExpandList();}
			~_CMemoryPool() {
#if(_MSC_VER < 1300)
				const std::size_t	CHUNK_SIZE = std::max(sizeof(T), sizeof(_CMemoryPool*));
#endif /* _MSC_VER < 1300 */
				for(_CMemoryPool* pNext = m_pNext; pNext != 0; pNext = m_pNext) {
					m_pNext = m_pNext->m_pNext;
					m_allocator.deallocate(reinterpret_cast<char*>(pNext), CHUNK_SIZE);
				}
			}
			void* Allocate(std::size_t) {
				if(m_pNext == 0)
					_ExpandList();
				_CMemoryPool*	pHead = m_pNext;
				m_pNext = pHead->m_pNext;
				return pHead;
			}
			void Deallocate(void* p) {
				_CMemoryPool*	pHead = static_cast<_CMemoryPool*>(p);
				pHead->m_pNext = m_pNext;
				m_pNext = pHead;
			}
		private:
			void _ExpandList() {
#if(_MSC_VER < 1300)
				const std::size_t	CHUNK_SIZE = std::max(sizeof(T), sizeof(_CMemoryPool*));
#endif /* _MSC_VER < 1300 */
				_CMemoryPool*	p = reinterpret_cast<_CMemoryPool*>(m_allocator.allocate(CHUNK_SIZE));

				m_pNext = p;
				for(int i = 0; i < nExpansionChunkCount; ++i) {
					p->m_pNext = reinterpret_cast<_CMemoryPool*>(m_allocator.allocate(CHUNK_SIZE));
					p = p->m_pNext;
				}
				p->m_pNext = 0;
			}

		private:
#if((!defined _MSC_VER) || (_MSC_VER >= 1300))
			enum {CHUNK_SIZE = (sizeof(T) > sizeof(_CMemoryPool*)) ? sizeof(T) : sizeof(_CMemoryPool*)};
#endif /* (!defined _MSC_VER) || (_MSC_VER >= 1300) */
			_CMemoryPool*	m_pNext;
		};
		static _CPoolHolder<_CMemoryPool>	m_pPool;
		static Allocator<char>				m_allocator;
	};

	template<class T, std::size_t nExpansionChunkCount, template<class> class Allocator>
#if(_MSC_VER < 1300)
	_CPoolHolder<CUseMemoryPool<T, nExpansionChunkCount, Allocator>::_CMemoryPool>
	CUseMemoryPool<T, nExpansionChunkCount, Allocator>::m_pPool;
#else
	_CPoolHolder<typename CUseMemoryPool<T, nExpansionChunkCount, Allocator>::_CMemoryPool>
	CUseMemoryPool<T, nExpansionChunkCount, Allocator>::m_pPool;
#endif /* _MSC_VER < 1300 */
	template<class T, std::size_t nSize, template<class> class Allocator>
	Allocator<char> CUseMemoryPool<T, nSize, Allocator>::m_allocator;


	namespace Windows {

		class CDumpContext : public CSelfAssertable {
			// constructor
		public:
			CDumpContext(const TCHAR* lpszFileName = 0) throw(std::exception);

			// operators
			template <typename T>
			CDumpContext&	operator <<(const T& rhs);

			// methods
		public:
			void	Flush() throw();
			void	HexDump(const TCHAR* lpszLine, uchar* pb, int nBytes, int nWidth = 0x10) throw();

			// data member
		private:
			TCHAR*	m_pszFileName;	// error log
		};

		namespace {
			CDumpContext	dout;
		}

} /* namespace Windows */
} /* namespace Manah */


// macros
/////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
#define ASSERT(exp)	assert(exp)
#define VERIFY(exp)	assert(exp)
#else
#define ASSERT(exp)
#define VERIFY(exp)	(exp)
#endif /* _DEBUG */

#define toBoolean(exp)	((exp) != 0)
#ifndef _countof
#define _countof(array)	(sizeof(array) / sizeof(array[0]))
#endif /* !_countof */
#ifndef _endof
#define _endof(array)	(array + _countof(array))
#endif /* !_endof */

template<class Base> struct AutoZero : public Base {
	AutoZero() {std::memset(this, 0, sizeof(Base));}
};
template<class Base, typename sizeMemberType, sizeMemberType(Base::*sizeMember)>
struct AutoZeroS : public AutoZero<Base> {
	AutoZeroS() {this->*sizeMember = sizeof(Base);}
};
template<class Base> struct AutoZeroCB : public AutoZeroS<Base, UINT, &Base::cbSize> {};
template<class Base> struct AutoZeroLS : public AutoZeroS<Base, DWORD, &Base::lStructSize> {};

#ifndef LVS_EX_LABELTIP	// VC6
	#define LVS_EX_LABELTIP	0x4000
#endif
#ifndef ListView_SetCheckState	// VC6
	#define ListView_SetCheckState(hwndLV, i, fCheck)	\
	ListView_SetItemState(hwndLV, i,					\
	INDEXTOSTATEIMAGEMASK((fCheck)? 2 : 1), LVIS_STATEIMAGEMASK)
#endif


// CDumpContext class implementation
/////////////////////////////////////////////////////////////////////////////

inline Manah::Windows::CDumpContext::CDumpContext(
		const TCHAR* lpszFileName /* = 0 */) throw(std::exception) {
	if(lpszFileName != 0)
		throw std::exception("File log is not supported!");
		// std::_tcscpy(m_szFileName, lpszFileName);
}

inline void Manah::Windows::CDumpContext::Flush() {
	AssertValid();
	/* not implemented */
}

inline void Manah::Windows::CDumpContext::HexDump(const TCHAR* lpszLine, uchar* pb, int nBytes, int nWidth /* = 0x10 */) {
	AssertValid();

	TCHAR*	pszOutput;	// all line
	TCHAR	szByte[4];

	pszOutput = new TCHAR[static_cast<std::size_t>(
		(std::_tcslen(lpszLine) + 3 * nWidth + 2) * static_cast<float>(nBytes / nWidth))];
	std::_tcscpy(pszOutput, lpszLine);
	for(int i = 0; i < nBytes; i++){
		wsprintf(szByte, _T(" %d"), pb);
		std::_tcscat(pszOutput, szByte);
		if(i % nWidth == 0){
			std::_tcscat(pszOutput, _T("\n"));
			std::_tcscat(pszOutput, lpszLine);
		}
	}
	::OutputDebugString(_T("\n>----Dump is started"));
	::OutputDebugString(pszOutput);
	::OutputDebugString(_T("\n>----Dump is done"));
	delete[] pszOutput;
}

template <typename T>
inline Manah::Windows::CDumpContext& Manah::Windows::CDumpContext::operator <<(const T& rhs) throw() {
	AssertValid();
	std::basic_ostringstream<TCHAR>	ss;
	ss << rhs;
	::OutputDebugString(ss.str().c_str());
	return *this;
}

#endif /* _OBJECT_H_ */

/* [EOF] */