// Splitter.h
// (c) 2005 exeal

#ifndef _SPLITTER_H_
#define _SPLITTER_H_
#include "Window.h"

namespace Manah {
namespace Windows {
namespace Controls {

enum ChildrenDestructionPolicy {
	CHILDRENDESTRUCTIONPOLICY_DELETE,						// pane objects' destructors
	CHILDRENDESTRUCTIONPOLICY_DONTDELETE,					// do nothing about panes
	CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL	// call SetParent(pane.GetWindow(), 0) when root window destroyed
};
enum SplitDirection {
	SPLITDIRECTION_NOSPLIT,
	SPLITDIRECTION_NS,
	SPLITDIRECTION_WE
};
enum PanePosition	{
	PANEPOSITION_LEFT = 0,
	PANEPOSITION_TOP = PANEPOSITION_LEFT,
	PANEPOSITION_RIGHT = 1,
	PANEPOSITION_BOTTOM = PANEPOSITION_RIGHT,
	PANEPOSITION_COUNT = 2
};


template<class Pane, ChildrenDestructionPolicy cdp> class _CSplitter;


class CAbstractPane {
public:
	virtual ~CAbstractPane() {}
public:
	virtual HWND GetWindow() const = 0;	// provides pane's window handle
};


template<class Pane /* implements CAbstractPane*/, ChildrenDestructionPolicy cdp = CHILDRENDESTRUCTIONPOLICY_DELETE>
/* final */ class CSplitterRoot : public CCustomControl<CSplitterRoot>, CNoncopyable {
public:
	class Iterator {	// once the structure is changed, iterator is inavailable
	private:
		typedef _CSplitter<Pane, cdp> Splitter;
		Iterator(const Splitter& root) : m_pParent(&root) {Reset();}
		Iterator(const Splitter& parent, Pane& pane) : m_pParent(&parent), m_pPane(&pane) {}
		operator =(const Iterator& rhs);	// not used
	public:
		Iterator(const Iterator& rhs) : m_pParent(rhs.m_pParent), m_pPane(rhs.m_pPane) {}
	public:
		Pane& Get() const {assert(m_pPane != 0); return *m_pPane;}
		bool IsEnd() const {return m_pPane == 0;}
		void Next() {
			Pane*		pNext = 0;
			Splitter*	pParent = 0;
			if(m_pParent->GetNextPane(*m_pPane, true, pNext, &pParent)) {
				m_pParent = pParent;
				m_pPane = pNext;
			} else	// at end
				m_pPane = 0;
		}
		void Reset() {
			while(m_pParent->m_pParent != 0)
				m_pParent = m_pParent->m_pParent;
			m_pParent->GetFirstPane(true, m_pPane, const_cast<Splitter**>(&m_pParent));
		}
	private:
		const Splitter* m_pParent;
		Pane* m_pPane;
		friend class CSplitterRoot<Pane, cdp>;
	};

	DEFINE_WINDOW_CLASS() {
		lpszName = _T("ManahSplitterRoot");
		bgColor = COLOR_BTNFACE;
		nStyle = CS_DBLCLKS;
	}

public:
	CSplitterRoot() : m_pDefaultActivePane(0),
			m_nFrameWidth(::GetSystemMetrics(SM_CXSIZEFRAME)), m_nFrameHeight(::GetSystemMetrics(SM_CYSIZEFRAME)),
			m_nMinimumPaneWidth(::GetSystemMetrics(SM_CXMIN)), m_nMinimumPaneHeight(::GetSystemMetrics(SM_CYMIN)),
			m_pDraggingSplitter(0) {}
public:
	void ActivateNextPane() {_ActivateNextPane(true);}
	void ActivatePreviousPane() {_ActivateNextPane(false);}
	void AdjustPanes() {
		RECT	rc;
		GetWindowRect(rc);
		::OffsetRect(&rc, -rc.left, -rc.top);
		m_root.AdjustPanes(rc, m_nFrameWidth, m_nFrameHeight);
		InvalidateRect(0);
	}
	bool Create(HWND hwndParent, const RECT& rect, DWORD dwStyle, DWORD dwExStyle, Pane& initialPane) {
		assert(toBoolean(::IsWindow(hwndParent)));
		if(!CCustomControl<CSplitterRoot>::Create(hwndParent, rect, 0, dwStyle, dwExStyle))
			return false;
		::SetParent(static_cast<CAbstractPane&>(initialPane).GetWindow(), GetSafeHwnd());
		m_root.m_children[PANEPOSITION_LEFT].type = Splitter::PANETYPE_SINGLE;
		m_root.m_children[PANEPOSITION_LEFT].body.pPane = m_pDefaultActivePane = &initialPane;
		return true;
	}
	Iterator EnumeratePanes() const {return Iterator(m_root);}
	Pane& GetActivePane() const throw(std::logic_error) {
		HWND	hwndFocused = ::GetFocus();
		for(Iterator it = EnumeratePanes(); !it.IsEnd(); it.Next()) {
			if(static_cast<CAbstractPane&>(it.Get()).GetWindow() == hwndFocused)
				return it.Get();
		}
		if(m_pDefaultActivePane != 0)
			return *m_pDefaultActivePane;
		else
			throw std::logic_error("There are no panes.");
	}
	uint GetSplitterSize(uint& nWidth, uint& nHeight) const {nWidth = m_nSplitterWidth; nHeight = m_nSplitterHeight;}
	bool IsSplit(const Pane& pane) const throw(std::invalid_argument) {
		if(Splitter* pParent = _FindPane(m_root, pane))
			return pParent->m_direction != SPLITDIRECTION_NOSPLIT;
		else
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
	}
	void RemoveActivePane() throw(std::logic_error) {Unsplit(GetActivePane());}
	void RemoveInactivePanes() throw(std::logic_error) {
		Pane&		activePane = GetActivePane();
		Splitter*	pParent = _FindPane(m_root, activePane);
		const bool	bHasFocus = IsChild(::GetFocus());

		// prevent active pane from deletion
		assert(pParent != 0);
		if(pParent->m_children[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SINGLE
				&& pParent->m_children[PANEPOSITION_LEFT].body.pPane == &activePane)
			pParent->m_children[PANEPOSITION_LEFT].type = Splitter::PANETYPE_EMPTY;
		else {
			assert(pParent->m_children[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SINGLE
				&& pParent->m_children[PANEPOSITION_RIGHT].body.pPane == &activePane);
			pParent->m_children[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
		}

		if(m_root.m_children[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SINGLE
				&& cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
			delete m_root.m_children[PANEPOSITION_LEFT].body.pPane;
		else if(m_root.m_children[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SPLITTER)
			delete m_root.m_children[PANEPOSITION_LEFT].body.pSplitter;
		if(m_root.m_direction != SPLITDIRECTION_NOSPLIT) {
			if(m_root.m_children[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SINGLE
					&& cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
				delete m_root.m_children[PANEPOSITION_RIGHT].body.pPane;
			else if(m_root.m_children[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SPLITTER)
				delete m_root.m_children[PANEPOSITION_RIGHT].body.pSplitter;
		}

		// same as construction...
		m_root.m_children[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
		m_root.m_direction = SPLITDIRECTION_NOSPLIT;
		m_root.m_nFirstPaneSizeRatio = 0.5F;
		m_root.m_children[PANEPOSITION_LEFT] = Splitter::Child(activePane);
		m_pDefaultActivePane = &activePane;
		if(bHasFocus && static_cast<CAbstractPane&>(activePane).GetWindow() != ::GetFocus())
			::SetFocus(static_cast<CAbstractPane&>(activePane).GetWindow());
		AdjustPanes();
	}
	void SetDefaultActivePane(Pane& pane) throw(std::invalid_argument) {
		if(_FindPane(m_root, pane) == 0)
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
		m_pDefaultActivePane = &pane;
	}
	void SetPaneMinimumSize(uint nWidth, uint nHeight) {m_nMinimumPaneWidth = nWidth; m_nMinimumPaneHeight = nHeight;}
	void SetSplitterSize(uint nWidth, uint nHeight) {
		m_nSplitterWidth = nWidth;
		m_nSplitterHeight = nHeight;
		AdjustPanes();
	}
	void SplitNS(Pane& pane, Pane& clone) throw(std::invalid_argument) {_Split(pane, clone, true);}
	void SplitWE(Pane& pane, Pane& clone) throw(std::invalid_argument) {_Split(pane, clone, false);}
	void Unsplit(Pane& pane) throw(std::invalid_argument) {
		Splitter*	pParent = _FindPane(m_root, pane);

		if(pParent == 0)
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
		else if(pParent->m_direction == SPLITDIRECTION_NOSPLIT)
			throw std::invalid_argument("Specified pane is not split.");

		Splitter::Child&	leftTop = pParent->m_children[PANEPOSITION_LEFT];
		Splitter::Child&	rightBottom = pParent->m_children[PANEPOSITION_RIGHT];
		const bool	bRemovedPaneWasDefaultActive = &pane == m_pDefaultActivePane;
		const bool	bRemovedPaneHadFocus = static_cast<CAbstractPane&>(pane).GetWindow() == ::GetFocus();

		// swap panes if deleting pane is left/top
		if(leftTop.type == Splitter::PANETYPE_SINGLE && leftTop.body.pPane == &pane)
			std::swap(leftTop, rightBottom);
		assert(rightBottom.type == Splitter::PANETYPE_SINGLE && rightBottom.body.pPane == &pane);

		Pane*	pNextFirstPane = 0;
		if(bRemovedPaneWasDefaultActive || bRemovedPaneHadFocus) {
			if(leftTop.type == Splitter::PANETYPE_SINGLE)
				pNextFirstPane = leftTop.body.pPane;
			else
				leftTop.body.pSplitter->GetFirstPane(true, pNextFirstPane);
		}

		if(cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
			delete rightBottom.body.pPane;

		// there are 4 scenarios
		if(pParent == &m_root) {	// parent is root
			if(leftTop.type == Splitter::PANETYPE_SINGLE) {	// live pane is a single pane
				rightBottom = Splitter::Child();
				m_root.m_direction = SPLITDIRECTION_NOSPLIT;
				m_root.m_nFirstPaneSizeRatio = 0.5F;
			} else {	// live pane is a splitter
				Splitter&	liveSplitter = *leftTop.body.pSplitter;
				m_root.m_direction = liveSplitter.m_direction;
				m_root.m_nFirstPaneSizeRatio = liveSplitter.m_nFirstPaneSizeRatio;
				leftTop.type = liveSplitter.m_children[PANEPOSITION_LEFT].type;
				leftTop.body = liveSplitter.m_children[PANEPOSITION_LEFT].body;
				rightBottom.type = liveSplitter.m_children[PANEPOSITION_RIGHT].type;
				rightBottom.body = liveSplitter.m_children[PANEPOSITION_RIGHT].body;

				if(liveSplitter.m_children[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SPLITTER)
					liveSplitter.m_children[PANEPOSITION_LEFT].body.pSplitter->m_pParent = &m_root;
				if(liveSplitter.m_children[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SPLITTER)
					liveSplitter.m_children[PANEPOSITION_RIGHT].body.pSplitter->m_pParent = &m_root;
				liveSplitter.m_children[PANEPOSITION_LEFT].type
					= liveSplitter.m_children[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
				delete &liveSplitter;	// this splitter should delete no children
			}
		} else {	// parent is not root
			Splitter&	grandParent = *pParent->m_pParent;
			const bool	bParentIsLeftChildOfGrandParent =
				pParent == grandParent.m_children[PANEPOSITION_LEFT].body.pSplitter;
			Splitter::Child&	replacedChild =
				grandParent.m_children[bParentIsLeftChildOfGrandParent ? PANEPOSITION_LEFT : PANEPOSITION_RIGHT];
			if(leftTop.type == Splitter::PANETYPE_SINGLE) {	// live pane is a single pane
				replacedChild.type = Splitter::PANETYPE_SINGLE;
				replacedChild.body.pPane = leftTop.body.pPane;
			} else {	// live pane is a splitter
				replacedChild.type = Splitter::PANETYPE_SPLITTER;
				replacedChild.body.pSplitter = leftTop.body.pSplitter;
				leftTop.body.pSplitter->m_pParent = &grandParent;
			}
			pParent->m_children[PANEPOSITION_LEFT].type
				= pParent->m_children[PANEPOSITION_RIGHT].type = Splitter::PANETYPE_EMPTY;
			delete pParent;	// this splitter should delete no children
		}

		AdjustPanes();

		if(bRemovedPaneWasDefaultActive)	m_pDefaultActivePane = pNextFirstPane;
		if(bRemovedPaneHadFocus)			::SetFocus(static_cast<CAbstractPane*>(pNextFirstPane)->GetWindow());
	}
protected:
	LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
		if(message == WM_CAPTURECHANGED)
			OnCaptureChanged(reinterpret_cast<HWND>(lParam));
		return CCustomControl<CSplitterRoot>::DispatchEvent(message, wParam, lParam);
	}
//	friend class Manah::Windows::Controls::CCustomControl<CSplitterRoot>;
	void OnCaptureChanged(HWND hwndNew) {
		if(m_pDraggingSplitter != 0) {
			m_root.SendMessageToChildren(WM_EXITSIZEMOVE);
			if(m_nSizingFirstPaneSize != -1) {
				const RECT&	rcSplitter = m_pDraggingSplitter->m_rect;
				_DrawSizingSplitterXorBar();	// erase ghost bar
				m_pDraggingSplitter->m_nFirstPaneSizeRatio =
					(m_pDraggingSplitter->m_direction == SPLITDIRECTION_NS) ?
					(m_nSizingFirstPaneSize + m_nFrameHeight / 2) / static_cast<double>(rcSplitter.bottom - rcSplitter.top)
					: (m_nSizingFirstPaneSize + m_nFrameWidth / 2) / static_cast<double>(rcSplitter.right - rcSplitter.left);
				m_pDraggingSplitter->AdjustPanes(rcSplitter, m_nFrameWidth, m_nFrameHeight);
			}
			m_pDraggingSplitter = 0;
		}
	}
	void OnDestroy() {
		if(cdp == CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL) {
			for(Iterator it = EnumeratePanes(); !it.IsEnd(); it.Next())
				::SetParent(static_cast<CAbstractPane&>(it.Get()).GetWindow(), 0);
		}
	}
	void OnLButtonDown(UINT nFlags, POINT pt) {
		// begin sizing
		if(m_root.m_direction == SPLITDIRECTION_NOSPLIT)
			return;
		else if(m_pDraggingSplitter = m_root.HitTest(pt)) {
			const RECT&	rcSplitter = m_pDraggingSplitter->m_rect;
			BOOL		bFullDraggingEnabled;

			::SystemParametersInfo(SPI_GETDRAGFULLWINDOWS, 0, &bFullDraggingEnabled, 0);
			if(toBoolean(bFullDraggingEnabled))
				m_nSizingFirstPaneSize = -1;
			else if(m_pDraggingSplitter->m_direction == SPLITDIRECTION_NS) {
				m_nSizingFirstPaneSize =
					static_cast<uint>((rcSplitter.bottom - rcSplitter.top) * m_pDraggingSplitter->m_nFirstPaneSizeRatio)
					- m_nFrameHeight / 2;
				_DrawSizingSplitterXorBar();
			} else {
				assert(m_pDraggingSplitter->m_direction == SPLITDIRECTION_WE);
				m_nSizingFirstPaneSize =
					static_cast<uint>((rcSplitter.right - rcSplitter.left) * m_pDraggingSplitter->m_nFirstPaneSizeRatio)
					- m_nFrameWidth / 2;
				_DrawSizingSplitterXorBar();
			}
			SetCapture();
			m_root.SendMessageToChildren(WM_ENTERSIZEMOVE);
		}
	}
	void OnLButtonDblClk(UINT nFlags, POINT pt) {
		// double click splitter bar to unsplit
		if(m_root.m_direction == SPLITDIRECTION_NOSPLIT)
			return;
		else if(Splitter* pSplitter = m_root.HitTest(pt)) {
			if(pSplitter->m_children[PANEPOSITION_LEFT].type == Splitter::PANETYPE_SINGLE)
				Unsplit(*pSplitter->m_children[PANEPOSITION_LEFT].body.pPane);
			else if(pSplitter->m_children[PANEPOSITION_RIGHT].type == Splitter::PANETYPE_SINGLE)
				Unsplit(*pSplitter->m_children[PANEPOSITION_RIGHT].body.pPane);
		}
	}
	void OnLButtonUp(UINT nFlags, POINT pt) {ReleaseCapture();}
	void OnMouseMove(UINT nFlags, POINT pt) {
		if(m_pDraggingSplitter == 0)
			return;
		const RECT&	rc = m_pDraggingSplitter->m_rect;
		double		ratio;

		if(m_pDraggingSplitter->m_direction == SPLITDIRECTION_NS) {
			if(static_cast<uint>(rc.bottom - rc.top) <= m_nMinimumPaneHeight * 2 + m_nFrameHeight)	// too short
				ratio = 0.5F;
			else {
				pt.y = std::max(pt.y, rc.top + static_cast<long>(m_nMinimumPaneHeight));
				pt.y = std::min(pt.y, rc.bottom - static_cast<long>(m_nMinimumPaneHeight + m_nFrameHeight));
				ratio = (rc.top != rc.bottom) ?
					static_cast<double>(pt.y - rc.top) / static_cast<double>(rc.bottom - rc.top) : 0.5F;
			}
		} else {
			assert(m_pDraggingSplitter->m_direction == SPLITDIRECTION_WE);
			if(static_cast<uint>(rc.right - rc.left) <= m_nMinimumPaneWidth * 2 + m_nFrameWidth)	// too narrow
				ratio = 0.5F;
			else {
				pt.x = std::max(pt.x, rc.left + static_cast<long>(m_nMinimumPaneWidth));
				pt.x = std::min(pt.x, rc.right - static_cast<long>(m_nMinimumPaneWidth + m_nFrameWidth));
				ratio = (rc.left != rc.right) ?
					static_cast<double>(pt.x - rc.left) / static_cast<double>(rc.right - rc.left) : 0.5F;
			}
		}

		if(m_nSizingFirstPaneSize == -1) {	// dynamic sizing
			m_pDraggingSplitter->m_nFirstPaneSizeRatio = ratio;
			m_pDraggingSplitter->AdjustPanes(rc, m_nFrameWidth, m_nFrameHeight);
			InvalidateRect(0);
		} else {	// static sizing
			_DrawSizingSplitterXorBar();	// erase previous ghost
			m_nSizingFirstPaneSize = (m_pDraggingSplitter->m_direction == SPLITDIRECTION_NS) ?
				static_cast<uint>((rc.bottom - rc.top) * ratio) : static_cast<uint>((rc.right - rc.left) * ratio);
			_DrawSizingSplitterXorBar();	// draw new ghost
		}
	}
	void OnPaint(GDI::CPaintDC& dc) {m_root.Draw(dc, m_nFrameWidth, m_nFrameHeight);}
	bool OnSetCursor(HWND hWnd, UINT nHitTest, UINT message) {
		if(hWnd == m_hWnd && nHitTest == HTCLIENT) {
			POINT	pt;
			if(toBoolean(::GetCursorPos(&pt))) {
				ScreenToClient(pt);
				if(Splitter* pSplitter = m_root.HitTest(pt)) {
					if(pSplitter->m_direction != SPLITDIRECTION_NOSPLIT) {
						::SetCursor(::LoadCursor(0,
							(pSplitter->m_direction == SPLITDIRECTION_NS) ? IDC_SIZENS : IDC_SIZEWE));
						return true;
					}
				}
			}
		}
		return false;
	}
	void OnSetFocus(HWND hwndOld) {
		if(m_pDefaultActivePane != 0)
			::SetFocus(static_cast<CAbstractPane*>(m_pDefaultActivePane)->GetWindow());
	}
	void OnSize(UINT nType, int cx, int cy) {
		HWND	hWnd = m_hWnd, hwndParent = ::GetParent(m_hWnd);
		do {
			if(toBoolean(::IsIconic(hWnd)))	// ignore if the window is iconic
				return;
			hWnd = hwndParent;
			hwndParent = ::GetParent(hWnd);
		} while(hwndParent != 0);

		RECT	rc = {0, 0, cx, cy};
		m_root.AdjustPanes(rc, m_nFrameWidth, m_nFrameHeight);
		InvalidateRect(0);
	}
private:
	typedef _CSplitter<Pane, cdp>	Splitter;
	void _ActivateNextPane(bool bNext) {
		Pane&	activePane = GetActivePane();

		if(Splitter* pParent = _FindPane(m_root, activePane)) {
			Pane*	pNextPane = 0;
			if(pParent->GetNextPane(activePane, bNext, pNextPane))
				m_pDefaultActivePane = pNextPane;
			else	// current active pane is end/first
				m_root.GetFirstPane(bNext, m_pDefaultActivePane);
			if(HWND hWnd = static_cast<CAbstractPane*>(m_pDefaultActivePane)->GetWindow())
				::SetFocus(hWnd);
		}
	}
	void _DrawSizingSplitterXorBar() {
		RECT		rect;
		const RECT&	rcSplitter = m_pDraggingSplitter->m_rect;
		Manah::Windows::GDI::CWindowDC	dc = GetWindowDC();

		// create half tone brush (from MFC)
		ushort	grayPattern[8];
		for(int i = 0; i < _countof(grayPattern); ++i)
			grayPattern[i] = static_cast<ushort>(0x5555 << (i & 1));
		HBITMAP	hBitmap = ::CreateBitmap(_countof(grayPattern), _countof(grayPattern), 1, 1, grayPattern);
		HBRUSH	hBrush = ::CreatePatternBrush(hBitmap), hOldBrush;

		assert(m_pDraggingSplitter != 0 && m_nSizingFirstPaneSize != -1);
		if(m_pDraggingSplitter->m_direction == SPLITDIRECTION_NS)
			::SetRect(&rect, rcSplitter.left, rcSplitter.top + m_nSizingFirstPaneSize,
				rcSplitter.right, rcSplitter.top + m_nSizingFirstPaneSize + m_nFrameHeight);
		else {
			assert(m_pDraggingSplitter->m_direction == SPLITDIRECTION_WE);
			::SetRect(&rect, rcSplitter.left + m_nSizingFirstPaneSize, rcSplitter.top,
				rcSplitter.left + m_nSizingFirstPaneSize + m_nFrameWidth, rcSplitter.bottom);
		}
		hOldBrush = dc.SelectObject(hBrush);
		dc.PatBlt(rect, PATINVERT);
		dc.SelectObject(hOldBrush);
		::DeleteObject(hBrush);
		::DeleteObject(grayPattern);
	}
	static Splitter* _FindPane(const Splitter& splitter, const Pane& pane) {	// recursive method which returns pane's parent
		const Splitter::Child&	leftTop = splitter.m_children[PANEPOSITION_LEFT];
		const Splitter::Child&	rightBottom = splitter.m_children[PANEPOSITION_RIGHT];

		if(leftTop.type == Splitter::PANETYPE_EMPTY)
			return 0;
		else if(leftTop.type == Splitter::PANETYPE_SINGLE && leftTop.body.pPane == &pane)
			return const_cast<Splitter*>(&splitter);
		else if(leftTop.type == Splitter::PANETYPE_SPLITTER) {
			if(Splitter* p = _FindPane(*leftTop.body.pSplitter, pane))
				return p;
		}

		if(rightBottom.type == Splitter::PANETYPE_EMPTY)
			return 0;
		else if(rightBottom.type == Splitter::PANETYPE_SINGLE && rightBottom.body.pPane == &pane)
			return const_cast<Splitter*>(&splitter);
		else if(rightBottom.type == Splitter::PANETYPE_SPLITTER) {
			if(Splitter* p = _FindPane(*rightBottom.body.pSplitter, pane))
				return p;
		}

		return 0;
	}
	void _Split(Pane& pane, Pane& clone, bool bNS) throw(std::invalid_argument) {
		Splitter* const	pParent = _FindPane(m_root, pane);
		const pos1 = bNS ? PANEPOSITION_TOP : PANEPOSITION_LEFT;	// no matter...
		const pos2 = bNS ? PANEPOSITION_BOTTOM : PANEPOSITION_RIGHT;

		if(pParent == 0)
			throw std::invalid_argument("Specified pane does not belong to this splitter.");
		if(pParent->m_children[pos2].type == Splitter::PANETYPE_EMPTY) {
			// there is another empty pane
			assert(pParent->m_direction == SPLITDIRECTION_NOSPLIT);
			assert(pParent->m_children[pos1].type == Splitter::PANETYPE_SINGLE);
			assert(pParent->m_children[pos1].body.pPane == &pane);
			pParent->m_children[pos2] = Splitter::Child(clone);
			pParent->m_direction = bNS ? SPLITDIRECTION_NS : SPLITDIRECTION_WE;
		} else {
			// this splitter is already split
			Splitter::Child*	pChild = 0;

			assert(pParent->m_direction != SPLITDIRECTION_NOSPLIT);
			if(pParent->m_children[pos1].body.pPane == &pane)
				pChild = &pParent->m_children[pos1];
			else if(pParent->m_children[pos2].body.pPane == &pane)
				pChild = &pParent->m_children[pos2];

			assert(pChild != 0);
			assert(pChild->type == Splitter::PANETYPE_SINGLE);
			pChild->type = Splitter::PANETYPE_SPLITTER;
			pChild->body.pSplitter = new Splitter;
			pChild->body.pSplitter->m_pParent = pParent;
			pChild->body.pSplitter->m_direction = bNS ? SPLITDIRECTION_NS : SPLITDIRECTION_WE;
			pChild->body.pSplitter->m_children[PANEPOSITION_LEFT] = Splitter::Child(pane);
			pChild->body.pSplitter->m_children[PANEPOSITION_RIGHT] = Splitter::Child(clone);
		}
		AdjustPanes();
	}
private:
	Splitter	m_root;
	Pane*		m_pDefaultActivePane;
	uint		m_nFrameWidth, m_nFrameHeight;
	uint		m_nMinimumPaneWidth, m_nMinimumPaneHeight;

	Splitter*	m_pDraggingSplitter;	// sizing splitter
	uint		m_nSizingFirstPaneSize;	// first pane size of m_pDraggingSplitter (-1 when full dragging is enabled)
};


template<class Pane /* implements CAbstractPane */, ChildrenDestructionPolicy cdp>
class _CSplitter : public CNoncopyable {
private:
	_CSplitter() : m_pParent(0), m_direction(SPLITDIRECTION_NOSPLIT), m_nFirstPaneSizeRatio(0.5F) {}
	~_CSplitter() {
		Child&	leftTop = m_children[PANEPOSITION_LEFT];
		Child&	rightBottom = m_children[PANEPOSITION_RIGHT];

		if(leftTop.type == PANETYPE_SINGLE) {
			if(cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
				delete leftTop.body.pPane;
			else if(cdp == CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL)
				::SetParent(static_cast<CAbstractPane*>(leftTop.body.pPane)->GetWindow(), 0);
		} else if(leftTop.type == PANETYPE_SPLITTER)
			delete leftTop.body.pSplitter;
		if(rightBottom.type == PANETYPE_SINGLE) {
			if(cdp == CHILDRENDESTRUCTIONPOLICY_DELETE)
				delete rightBottom.body.pPane;
			else if(cdp == CHILDRENDESTRUCTIONPOLICY_DONTDELETEANDSETPARENTTONULL)
				::SetParent(static_cast<CAbstractPane*>(rightBottom.body.pPane)->GetWindow(), 0);
		} else if(rightBottom.type == PANETYPE_SPLITTER)
			delete rightBottom.body.pSplitter;
	}
	enum PaneType {PANETYPE_EMPTY, PANETYPE_SINGLE, PANETYPE_SPLITTER};
private:
	void AdjustPanes(const RECT& newRect, uint nFrameWidth, uint nFrameHeight) {
		m_rect = newRect;
		if(m_direction == SPLITDIRECTION_NOSPLIT) {	// is not split
			if(m_children[PANEPOSITION_LEFT].type != PANETYPE_EMPTY)
				::MoveWindow(
					static_cast<CAbstractPane&>(*m_children[PANEPOSITION_LEFT].body.pPane).GetWindow(),
					newRect.left, newRect.top,
					newRect.right - newRect.left, newRect.bottom - newRect.top, true);
		} else {
			RECT	rc = newRect;

			if(m_direction == SPLITDIRECTION_NS)
				rc.bottom = newRect.top +
					static_cast<long>((newRect.bottom - newRect.top) * m_nFirstPaneSizeRatio) - nFrameHeight / 2;
			else
				rc.right = newRect.left +
					static_cast<long>((newRect.right - newRect.left) * m_nFirstPaneSizeRatio) - nFrameWidth / 2;
			if(m_children[PANEPOSITION_LEFT].type == PANETYPE_SINGLE)
				::MoveWindow(static_cast<CAbstractPane*>(m_children[PANEPOSITION_LEFT].body.pPane)->GetWindow(),
					rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top, true);
			else
				m_children[PANEPOSITION_LEFT].body.pSplitter->AdjustPanes(rc, nFrameWidth, nFrameHeight);

			if(m_direction == SPLITDIRECTION_NS) {
				rc.top = rc.bottom + nFrameHeight;
				rc.bottom = newRect.bottom;
			} else {
				rc.left = rc.right + nFrameWidth;
				rc.right = newRect.right;
			}
			if(m_children[PANEPOSITION_RIGHT].type == PANETYPE_SINGLE)
				::MoveWindow(static_cast<CAbstractPane*>(m_children[PANEPOSITION_RIGHT].body.pPane)->GetWindow(),
					rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top, true);
			else
				m_children[PANEPOSITION_RIGHT].body.pSplitter->AdjustPanes(rc, nFrameWidth, nFrameHeight);
		}
	}
	void Draw(Manah::Windows::GDI::CPaintDC& dc, uint nFrameWidth, uint nFrameHeight) {
		if(m_direction == SPLITDIRECTION_NOSPLIT)
			return;

		RECT	rect;

		rect.left = (m_direction == SPLITDIRECTION_NS) ? m_rect.left :
			m_rect.left + static_cast<long>((m_rect.right - m_rect.left) * m_nFirstPaneSizeRatio) - nFrameWidth / 2;
		rect.top = (m_direction == SPLITDIRECTION_WE) ? m_rect.top :
			m_rect.top + static_cast<long>((m_rect.bottom - m_rect.top) * m_nFirstPaneSizeRatio) - nFrameHeight / 2;
		rect.right = (m_direction == SPLITDIRECTION_NS) ? m_rect.right : rect.left + nFrameWidth;
		rect.bottom = (m_direction == SPLITDIRECTION_WE) ? m_rect.bottom : rect.top + nFrameHeight;
		dc.FillSolidRect(rect, ::GetSysColor(COLOR_3DFACE));

		if(m_children[PANEPOSITION_LEFT].type == PANETYPE_SPLITTER)
			m_children[PANEPOSITION_LEFT].body.pSplitter->Draw(dc, nFrameWidth, nFrameHeight);
		if(m_children[PANEPOSITION_RIGHT].type == PANETYPE_SPLITTER)
			m_children[PANEPOSITION_RIGHT].body.pSplitter->Draw(dc, nFrameWidth, nFrameHeight);
	}
	void GetFirstPane(bool bLeftTop, Pane*& pPane, _CSplitter<Pane, cdp>** ppParent = 0) const {
		_CSplitter<Pane, cdp>*	pParent = const_cast<_CSplitter<Pane, cdp>*>(this);
		const PanePosition		seekDirection = bLeftTop ? PANEPOSITION_LEFT : PANEPOSITION_RIGHT;
		pPane = 0;
		do {
			if(pParent->m_children[seekDirection].type == PANETYPE_SINGLE) {
				pPane = pParent->m_children[seekDirection].body.pPane;
				assert(pPane != 0);
			} else {
				assert(pParent->m_children[seekDirection].type == PANETYPE_SPLITTER);
				pParent = pParent->m_children[seekDirection].body.pSplitter;
			}
		} while(pPane == 0);
		if(ppParent != 0)
			*ppParent = pParent;
	}
	bool GetNextPane(const Pane& pane, bool bNext, Pane*& pNext, _CSplitter<Pane, cdp>** ppParent = 0) const {
		// returns false if pane is rightmost
		const PanePosition	forwardPosition = bNext ? PANEPOSITION_RIGHT : PANEPOSITION_LEFT;
		const PanePosition	backPosition = bNext ? PANEPOSITION_LEFT : PANEPOSITION_RIGHT;

		if(m_children[backPosition].type == PANETYPE_SINGLE && m_children[backPosition].body.pPane == &pane) {
			if(m_children[forwardPosition].type == PANETYPE_SINGLE) {
				pNext = m_children[forwardPosition].body.pPane;
				if(ppParent != 0)	*ppParent = const_cast<_CSplitter<Pane, cdp>*>(this);
				return true;
			} else if(m_children[forwardPosition].type == PANETYPE_SPLITTER) {
				m_children[forwardPosition].body.pSplitter->GetFirstPane(bNext, pNext, ppParent);
				return true;
			} else {
				assert(m_children[forwardPosition].type == PANETYPE_EMPTY);
				return false;
			}
		}

		// up one level
		assert(m_children[forwardPosition].type == PANETYPE_SINGLE && m_children[forwardPosition].body.pPane == &pane);
		_CSplitter*	pChildSplitter = const_cast<_CSplitter<Pane, cdp>*>(this);
		_CSplitter*	pParent = m_pParent;

		if(pParent == 0)	// `this' is root
			return false;
		while(pParent != 0) {
			if(pParent->m_children[backPosition].type == PANETYPE_SPLITTER
					&& pParent->m_children[backPosition].body.pSplitter == pChildSplitter) {
				if(pParent->m_children[forwardPosition].type == PANETYPE_SINGLE) {
					pNext = pParent->m_children[forwardPosition].body.pPane;
					if(ppParent != 0)	*ppParent = pParent;
					return true;
				} else if(pParent->m_children[forwardPosition].type == PANETYPE_SPLITTER) {
					pParent->m_children[forwardPosition].body.pSplitter->GetFirstPane(bNext, pNext, ppParent);
					return true;
				} else {
					assert(pParent->m_children[forwardPosition].type == PANETYPE_EMPTY);
					break;
				}
			} else {
				// up one level
				assert(pParent->m_children[forwardPosition].type == PANETYPE_SPLITTER
					&& pParent->m_children[forwardPosition].body.pSplitter == pChildSplitter);
				pChildSplitter = pParent;
				pParent = pParent->m_pParent;
			}
		}
		return false;
	}
	_CSplitter<Pane, cdp>* HitTest(const POINT& pt) const {
		if(!toBoolean(::PtInRect(&m_rect, pt)))
			return 0;
		if(m_children[PANEPOSITION_LEFT].type == PANETYPE_SPLITTER) {
			if(_CSplitter<Pane, cdp>* p = m_children[PANEPOSITION_LEFT].body.pSplitter->HitTest(pt))
				return p;
		}
		if(m_children[PANEPOSITION_RIGHT].type == PANETYPE_SPLITTER) {
			if(_CSplitter<Pane, cdp>* p = m_children[PANEPOSITION_RIGHT].body.pSplitter->HitTest(pt))
				return p;
		}
		return const_cast<_CSplitter<Pane, cdp>*>(this);
	}
	void SendMessageToChildren(UINT message) {
		Child&	leftTop = m_children[PANEPOSITION_LEFT];
		Child&	rightBottom = m_children[PANEPOSITION_RIGHT];

		if(leftTop.type == PANETYPE_SINGLE)
			::SendMessage(static_cast<CAbstractPane*>(leftTop.body.pPane)->GetWindow(), message, 0, 0L);
		else if(leftTop.type == PANETYPE_SPLITTER)
			leftTop.body.pSplitter->SendMessageToChildren(message);
		if(rightBottom.type == PANETYPE_SINGLE)
			::SendMessage(static_cast<CAbstractPane*>(rightBottom.body.pPane)->GetWindow(), message, 0, 0L);
		else if(rightBottom.type == PANETYPE_SPLITTER)
			rightBottom.body.pSplitter->SendMessageToChildren(message);
	}
private:
	struct Child {
		PaneType type;
		union {
			Pane*					pPane;
			_CSplitter<Pane, cdp>*	pSplitter;
		} body;
		Child() : type(PANETYPE_EMPTY) {}
		Child(Pane& pane) : type(PANETYPE_SINGLE) {body.pPane = &pane;}
	} m_children[PANEPOSITION_COUNT];	// children ([RIGHT] is EMPTY if this pane is not split)
	_CSplitter<Pane, cdp>*	m_pParent;		// parent splitter
	SplitDirection			m_direction;	// SPLITDIRECTION_NOSPLIT for only root
	RECT					m_rect;
	double					m_nFirstPaneSizeRatio;	// (left or top pane's size) / (whole splitter size)

	friend class CSplitterRoot<Pane, cdp>;
	friend class CSplitterRoot<Pane, cdp>::Iterator;
};

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _SPLITTER_WINDOW_H_ */

/* [EOF] */