// Window.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _WINDOW_H_
#define _WINDOW_H_

#include "DC.h"

#ifndef WS_EX_LAYERED
#define WS_EX_LAYERED	0x00080000
#define LWA_COLORKEY	0x00000001
#define LWA_ALPHA		0x00000002
/*typedef struct _BLENDFUNCTION {
	BYTE	BlendOp;
	BYTE	BlendFlags;
	BYTE	SourceConstantAlpha;
	BYTE	AlphaFormat;
} BLENDFUNCTION, *PBLENDFUNCTION;*/
#endif /* !WS_EX_LAYERED */

#ifndef AW_SLIDE
#define AW_HOR_POSITIVE	0x00000001
#define AW_HOR_NEGATIVE	0x00000002
#define AW_VER_POSITIVE	0x00000004
#define AW_VER_NEGATIVE	0x00000008
#define AW_CENTER		0x00000010
#define AW_HIDE			0x00010000
#define AW_ACTIVATE		0x00020000
#define AW_SLIDE		0x00040000
#define AW_BLEND		0x00080000
#endif /* !AW_SLIDE */

namespace {
#if(_MSC_VER < 1300)
	template<class T, class U> struct SameTypes {
		template<class V> struct _ {enum {result = false};};
		template<> struct _<U> {enum {result = true};};
		enum {result = _<T>::result};
	};
#else
	template<class T1, class T2> struct SameTypes {enum {result = false};};
	template<class T> struct SameTypes<T, T> {enum {result = true};};
#endif /* _MSC_VER < 1300 */

	template<bool> struct TimerIdTypeSelector {typedef UINT type;};
	template<> struct TimerIdTypeSelector<true> {typedef UINT_PTR type;};
} // namespace `anonymous'

// WIN32 API Timer ID type
typedef TimerIdTypeSelector<
	SameTypes<TIMERPROC, void(*)(HWND, UINT, UINT_PTR, DWORD)>::result
>::type timerid_t;


// CWindow class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

// convenient types from ATL
struct MenuHandleOrControlId {
	MenuHandleOrControlId(HMENU hMenu) : m_hMenu(hMenu) {}
	MenuHandleOrControlId(UINT id) : m_hMenu(reinterpret_cast<HMENU>(id)) {}
	HMENU	m_hMenu;
};

// others used by (CCustomControl derevied)::GetWindowClass
struct BrushHandleOrColor {
	BrushHandleOrColor() : m_hBrush(0) {}
	BrushHandleOrColor(HBRUSH hBrush) : m_hBrush(hBrush) {}
	BrushHandleOrColor(COLORREF color) : m_hBrush(reinterpret_cast<HBRUSH>(color + 1)) {}
	BrushHandleOrColor& operator =(HBRUSH hBrush) {m_hBrush = hBrush; return *this;}
	BrushHandleOrColor& operator =(COLORREF color) {m_hBrush = reinterpret_cast<HBRUSH>(color + 1); return *this;}
	HBRUSH	m_hBrush;
};
struct CursorHandleOrId {
	CursorHandleOrId() : m_hCursor(0) {}
	CursorHandleOrId(HCURSOR hCursor) : m_hCursor(hCursor) {}
	CursorHandleOrId(const TCHAR* lpszSystemCursorId) : m_hCursor(::LoadCursor(0, lpszSystemCursorId)) {}
	CursorHandleOrId& operator =(HCURSOR hCursor) {m_hCursor = hCursor; return *this;}
	CursorHandleOrId& operator =(const TCHAR* lpszSystemCursorId) {m_hCursor = ::LoadCursor(0, lpszSystemCursorId); return *this;}
	HCURSOR	m_hCursor;
};

struct DefaultWindowRect : public tagRECT {
	DefaultWindowRect() {left = top = right = bottom = CW_USEDEFAULT;}
};

template<class Window> class CSubclassable;

class CWindow : public CSelfAssertable {
	// RXgN^
public:
	explicit CWindow(HWND hWnd = 0);	// attach
	CWindow(const CWindow& rhs);
	virtual ~CWindow();

	// Zq
public:
	bool	operator ==(const CWindow& rhs) const {return m_hWnd == rhs.m_hWnd;}
			operator HWND() const {return GetSafeHwnd();}
private:
			operator =(const CWindow& rhs);	// never been invoked

	// \bh
public:
	HWND	GetSafeHwnd() const;

	/* \zƔj */
	bool	Create(const TCHAR* lpszClassName, HWND hwndParentOrHInstance,
				const RECT& rect = DefaultWindowRect(), const TCHAR* lpszWindowName = 0,
				DWORD dwStyle = 0UL, DWORD dwExStyle = 0UL, HMENU hMenu = 0, void* lpParam = 0);
	bool	DestroyWindow();

	/* A^b` */
	virtual bool	Attach(HWND hWnd);
	virtual HWND	Detach();

	/* X^C */
	DWORD	GetExStyle() const;
	DWORD	GetStyle() const;
	bool	ModifyStyle(DWORD dwRemove, DWORD dwAdd);
	bool	ModifyStyleEx(DWORD dwRemove, DWORD dwAdd);

	/* EBhENX */
	DWORD	GetClassLong(int nIndex) const;
	int		GetClassName(TCHAR* lpszClassName, int cchMax) const;
	LONG	GetWindowLong(int nIndex) const;
	DWORD	SetClassLong(int nIndex, DWORD dwNewLong);
	LONG	SetWindowLong(int nIndex, LONG dwNewLong);
#ifdef _WIN64
	ULONG_PTR	GetClassLongPtr(int nIndex) const;
	LONG_PTR	GetWindowLongPtr(int nIndex) const;
	ULONG_PTR	SetClassLongPtr(int nIndex, ULONG_PTR dwNewLong);
	LONG_PTR	SetWindowLongPtr(int nIndex, LONG_PTR dwNewLong);
#endif /* _WIN64 */

	/*  */
	bool			EnableWindow(bool bEnable = true);
	static CWindow	GetActiveWindow();
	static CWindow	GetCapture();
	static CWindow	GetDesktopWindow();
	static CWindow	GetFocus();
	static CWindow	GetForegroundWindow();
	HICON			GetIcon(bool bBigIcon = true) const;
	bool			HasFocus() const;
	bool			IsWindow() const;
	bool			IsWindowEnabled() const;
	bool			IsWindowUnicode() const;
	static bool		ReleaseCapture();
	CWindow			SetActiveWindow();
	CWindow			SetCapture();
	CWindow			SetFocus();
	bool			SetForegroundWindow();
	HICON			SetIcon(HICON hIcon, bool bBigIcon  = true);

	/* TCYƈʒu */
	UINT	ArrangeIconicWindows();
	void	BringWindowToTop();
	void	GetClientRect(RECT& rect) const;
	bool	GetWindowPlacement(WINDOWPLACEMENT& wndpl) const;
	void	GetWindowRect(RECT& rect) const;
	int		GetWindowRgn(HRGN hRgn) const;
	bool	IsIconic() const;
	bool	IsZoomed() const;
	void	MoveWindow(int x, int y, int nWidth, int nHeight, bool bRepaint = true);
	void	MoveWindow(const RECT& rect, bool bRepaint = true);
	bool	SetWindowPlacement(const WINDOWPLACEMENT& wndpl);
	bool	SetWindowPos(HWND hwndInsertAfter, int x, int y, int cx, int cy, UINT nFlags);
	bool	SetWindowPos(HWND hwndInsertAfter, const RECT& rect, UINT nFlags);
	int		SetWindowRgn(const HRGN hRgn, bool bRedraw = true);

	/* EBhEANZX */
	void			CenterWindow(HWND hwndAlternate = 0);
	CWindow			ChildWindowFromPoint(const POINT& pt) const;
	CWindow			ChildWindowFromPointEx(const POINT& pt, UINT nFlags) const;
	static CWindow	FindWindow(const TCHAR* lpszClassName, const TCHAR* lpszWindowName);
	int				GetDlgCtrlID() const;
	CWindow			GetLastActivePopup() const;
	CWindow			GetNext(UINT nFlag = GW_HWNDNEXT) const;
	CWindow			GetOwner() const;
	CWindow			GetParent() const;
	CWindow			GetTopWindow() const;
	CWindow			GetWindow(UINT nCmd) const;
	bool			IsChild(HWND hWnd) const;
	int				SetDlgCtrlID(int nID);
	CWindow			SetParent(HWND hwndNewParent);
	static CWindow	WindowFromPoint(const POINT& pt);

	/* XVƕ` */
	GDI::CPaintDC	BeginPaint(PAINTSTRUCT& paint);
	bool			EnableScrollBar(int nSBFlags, UINT nArrowFlags = ESB_ENABLE_BOTH);
	void			EndPaint(const PAINTSTRUCT& paint);
	GDI::CClientDC	GetDC();
	GDI::CClientDC	GetDCEx(HRGN hrgnClip, DWORD dwFlags);
	bool			GetUpdateRect(RECT& rect, bool bErase = false);
	int				GetUpdateRgn(HRGN hRgn, bool bErase = false);
	GDI::CWindowDC	GetWindowDC();
	bool			LockWindowUpdate();
	void			Invalidate(bool bErase = true);
	void			InvalidateRect(const RECT* lpRect, bool bErase = true);
	void			InvalidateRgn(HRGN hRgn, bool bErase = true);
	bool			IsWindowVisible() const;
	void			Print(HDC hDC, DWORD dwFlags) const;
	void			PrintClient(HDC hDC, DWORD dwFlags) const;
	bool			RedrawWindow(RECT* lpRectUpdate = 0, HRGN hrgnClip = 0,
						UINT nFlags = RDW_INVALIDATE | RDW_UPDATENOW | RDW_ERASE);
	int				ReleaseDC(HDC hDC);
	void			SetRedraw(bool bRedraw = true);
	void			ShowOwnedPopups(bool bShow = true);
	bool			ShowWindow(UINT nCmdShow);
	void			UnlockWindowUpdate();
	void			UpdateWindow();
	void			ValidateRect(const RECT* lpRect);
	void			ValidateRgn(HRGN hRgn);

	/* W}bsO */
	void	ClientToScreen(POINT& pt) const;
	void	ClientToScreen(RECT& rect) const;
	void	MapWindowPoints(HWND hWndTo, RECT& rect) const;
	void	MapWindowPoints(HWND hWndTo, POINT* lpPoint, UINT nCount) const;
	void	ScreenToClient(POINT& pt) const;
	void	ScreenToClient(RECT& rect) const;

	/* EBhEeLXg */
	int		GetWindowText(TCHAR* lpszText, int nMaxCount) const;
	int		GetWindowTextLength() const;
	void	SetWindowText(const TCHAR* lpszText);

	/* tHg */
	HFONT	GetFont() const;
	void	SetFont(HFONT hFont, bool bRedraw = true);

	/* vpeB */
	int		EnumProps(PROPENUMPROC lpEnumFunc);
	int		EnumPropsEx(PROPENUMPROCEX lpEnumFunc, LPARAM lParam);
	HANDLE	GetProp(const TCHAR* lpsz) const;
	HANDLE	RemoveProp(const TCHAR* lpsz);
	bool	SetProp(const TCHAR* lpsz, HANDLE hData);

	/* wv */
	DWORD	GetWindowContextHelpId() const;
	bool	SetWindowContextHelpId(DWORD dwContextHelpId);
	bool	WinHelp(const TCHAR* lpszHelp, UINT nCommand = HELP_CONTEXT, DWORD dwData = 0);

	/* XN[ */
	bool	GetScrollInfo(int nBar, SCROLLINFO& scrollInfo, UINT nMask = SIF_ALL) const;
	int		GetScrollLimit(int nBar) const;
	int		GetScrollPos(int nBar) const;
	void	GetScrollRange(int nBar, int* nMinPos, int* nMaxPos) const;
	int		GetScrollTrackPos(int nBar) const;
	void	ScrollWindow(int xAmount, int yAmount, RECT* lpRect = 0, RECT* lpClipRect = 0);
	int		ScrollWindowEx(int dx, int dy, RECT* lpScrollRect, RECT* lpClipRect,
				HRGN hrgnUpdate, RECT* lpUpdateRect, UINT nFlags);
	bool	SetScrollInfo(int nBar, const SCROLLINFO& scrollInfo, bool bRedraw = true);
	int		SetScrollPos(int nBar, int nPos, bool bRedraw = true);
	void	SetScrollRange(int nBar, int nMinPos, int nMaxPos, bool bRedraw = true);
	void	ShowScrollBar(int nBar, bool bShow = true);

	/* Nbv{[hr[ */
	bool	ChangeClipboardChain(HWND hwndNewNext);
	CWindow	SetClipboardViewer();

	/* hbOAhhbv */
	void	DragAcceptFiles(bool bAccept = true);
	HRESULT	RegisterDragDrop(IDropTarget& dropTarget);
	HRESULT	RevokeDragDrop();

	/* Lbg */
	bool			CreateCaret(HBITMAP hBitmap, int nWidth, int nHeight);
	bool			CreateSolidCaret(int nWidth, int nHeight);
	bool			CreateGrayCaret(int nWidth, int nHeight);
	static LPPOINT	GetCaretPos();
	void			HideCaret();
	static void		SetCaretPos(POINT pt);
	void			ShowCaret();

	/* j[ */
	void	DrawMenuBar();
	HMENU	GetMenu() const;
	HMENU	GetSystemMenu(bool bRevert) const;
	bool	HiliteMenuItem(HMENU hMenu, UINT nItem, UINT nFlags);
	bool	SetMenu(HMENU hMenu);

	/* zbgL[ */
	DWORD	GetHotKey() const;
	int		SetHotKey(WORD nVirtualKeyCode, WORD nModifiers);

	/* ^C} */
	bool		KillTimer(timerid_t nIDEvent);
	timerid_t	SetTimer(timerid_t nIDEvent, UINT nElapse,
					void (CALLBACK* lpfnTimer)(HWND, UINT, timerid_t, DWORD) = 0);

	/* x */
	bool	FlashWindow(bool bInvert);
	int		MessageBox(const TCHAR* lpszText, const TCHAR* lpszCaption = 0, UINT nType = MB_OK);

	/* EBhEbZ[W */
	virtual LRESULT	DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam);
	LRESULT			SendMessage(UINT message, WPARAM wParam = 0, LPARAM lParam = 0L);
	bool			SendNotifyMessage(UINT message, WPARAM wParam, LPARAM lParam);
	LRESULT			PostMessage(UINT message, WPARAM wParam = 0, LPARAM lParam = 0L);

	/* vZXƃXbh */
	DWORD	GetWindowProcessId() const;
	DWORD	GetWindowThreadId() const;

	/* I[o[Ch\ȃbZ[Wnh */
protected:
	/* need to any implement */
	virtual bool	OnCommand(WORD wID, WORD wNotifyCode, HWND hwndCtrl);		// WM_COMMAND
	virtual void	OnClose();													// WM_CLOSE
	virtual void	OnDestroy();												// WM_DESTROY
	virtual bool	OnNotify(int idCtrl, LPNMHDR lpNMHDR);						// WM_NOTIFY
	virtual bool	OnSetCursor(HWND hWnd, UINT nHitTest, UINT message);		// WM_SETCURSOR
	/* noneed to implement */
	virtual void	OnActivate(UINT nState, HWND hwndPrevious, bool bMinimized);// WM_ACTIVATE
	virtual bool	OnContextMenu(HWND hWnd, POINT pt);							// WM_CONTEXTMENU
	virtual void	OnDrawItem(UINT idCtrl, LPDRAWITEMSTRUCT lpDrawItem);		// WM_DRAWITEM
	virtual void	OnKillFocus(HWND hwndNew);									// WM_KILLFOCUS
	virtual void	OnLButtonDown(UINT nFlags, POINT pt);						// WM_LBUTTONDOWN
	virtual void	OnLButtonDblClk(UINT nFlags, POINT pt);						// WM_LBUTTONDBLCLK
	virtual void	OnLButtonUp(UINT nFlags, POINT pt);							// WM_LBUTTONUP
	virtual void	OnMeasureItem(UINT idCtrl, LPMEASUREITEMSTRUCT lpMI);		// WM_MEASUREITEM
	virtual void	OnMouseMove(UINT nFlags, POINT pt);							// WM_MOUSEMOVE
	virtual void	OnSetFocus(HWND hwndOld);									// WM_SETFOCUS
	virtual void	OnShowWindow(bool bShow, UINT nStatus);						// WM_SHOWWINDOW
	virtual void	OnSize(UINT nType, int cx, int cy);							// WM_SIZE
	virtual void	OnSysColorChange();											// WM_SYSCOLORCHANGE
	virtual void	OnTimer(UINT nIDEvent);										// WM_TIMER

protected:
	virtual LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);
	virtual bool	PreTranslateMessage(UINT message, WPARAM wParam, LPARAM lParam);	/* called by only DE */

	/* fobO */
protected:
#ifdef _DEBUG
	virtual void AssertValid() const {CSelfAssertable::AssertValid();}
	virtual void AssertValidAsWindow() const {CSelfAssertable::AssertValid(); assert(IsWindow());}
#else
	virtual void AssertValid() const {}
	virtual void AssertValidAsWindow() const {}
#endif /* _DEBUG */

	// f[^o
protected:
	HWND	m_hWnd;
	bool	m_bAttached;

	friend class CSubclassable;
};


#ifdef SubclassWindow
#undef SubclassWindow
#endif /* SubclassWindow */

template<class Window = CWindow> class CSubclassable : public Window, public CNoncopyable {
public:
	CSubclassable() : m_pfnOldProc(0) {}
	virtual ~CSubclassable() {if(IsWindow()) UnsubclassWindow();}
public:
	bool			Attach(HWND hWnd, bool bSubclass = false);
	virtual LRESULT	DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam);
	HWND			Detach();
	virtual LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);
	bool			IsSubclassed() const {return m_pfnOldProc != 0;}
	bool			SubclassWindow();
	bool			UnsubclassWindow();
protected:
	virtual void	OnDestroy();
	static LRESULT CALLBACK	WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
private:
	WNDPROC	m_pfnOldProc;	// original procedure
};

typedef CSubclassable<CWindow>	CSubclassableWindow;


template<class Window = CWindow> class CLayered : public Window {
public:
	CLayered() : m_hUser32Dll(0), m_pfnAnimateWindow(0),
			m_pfnSetLayeredWindowAttributes(0), m_pfnUpdateLayeredWindow(0) {}
	virtual ~CLayered() {if(m_hUser32Dll != 0) ::FreeLibrary(m_hUser32Dll);}
public:
	bool	AnimateWindow(DWORD dwTime, DWORD dwFlags, bool bCatchError = true);
	bool	SetLayeredWindowAttributes(COLORREF clrKey, BYTE bAlpha, DWORD dwFlags);
	bool	UpdateLayeredWindow(HDC hDest, POINT* pptDest,
				SIZE* pSize, HDC hSrc, POINT* pptSrc, COLORREF clrKey,
				BLENDFUNCTION* pBlend, DWORD dwFlags);
private:
	typedef BOOL(WINAPI *AW)(HWND, DWORD, DWORD);
	typedef BOOL(WINAPI *SLWA)(HWND, COLORREF, BYTE, DWORD);
	typedef BOOL(WINAPI *ULW)(HWND, HDC, POINT*, SIZE*, HDC, POINT*, COLORREF, BLENDFUNCTION*, DWORD);
private:
	HMODULE	m_hUser32Dll;	// loaded dynamically to use layered window APIs
	AW		m_pfnAnimateWindow;
	SLWA	m_pfnSetLayeredWindowAttributes;
	ULW		m_pfnUpdateLayeredWindow;
};

typedef CLayered<CWindow>	CLayeredWindow;


// Control must be define GetClassName static method returns const TCHAR* and takes no parameters
#define CCommonControl	CStandardControl
#define DEFINE_CLASS_NAME(name)	public: static const TCHAR* GetClassName() {return name;}
template<class Control> class CStandardControl : public CSubclassableWindow {
public:
	CStandardControl() {}
	virtual ~CStandardControl() {}
public:
	bool	Create(HWND hwndParent, const RECT& rect = DefaultWindowRect(),
				const TCHAR* lpszWindowName = 0, int id = 0,
				DWORD dwStyle = DefaultStyle, DWORD dwExStyle = 0UL);
	enum {DefaultStyle = WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN | WS_CLIPSIBLINGS};
};


// Control must define one static method:
// void GetClass(GET_CLASS_PARAM_LIST)
#define GET_CLASS_PARAM_LIST	const TCHAR*& lpszName,											\
	HINSTANCE& hInstance, UINT& nStyle, Manah::Windows::Controls::BrushHandleOrColor& bgColor,	\
	Manah::Windows::Controls::CursorHandleOrId& cursor,	HICON& hIcon, HICON& hSmallIcon, int& cbClsExtra, int& cbWndExtra
#define DEFINE_WINDOW_CLASS()	public: static void GetClass(GET_CLASS_PARAM_LIST)
template<class Control, class BaseWindow = CWindow> class CCustomControl : public BaseWindow {
public:
	CCustomControl() {}
	CCustomControl(const CCustomControl<Control, BaseWindow>& rhs) : BaseWindow(rhs) {}
	virtual ~CCustomControl();
	bool	Create(HWND hwndParent, const RECT& rect = DefaultWindowRect(),
				const TCHAR* lpszWindowName = 0, DWORD dwStyle = 0UL, DWORD dwExStyle = 0UL);
protected:
	static LRESULT CALLBACK	WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
	virtual void	OnPaint(GDI::CPaintDC& dc) = 0;	// WM_PAINT
};


// write one of these in your class definition if override DispatchEvent.
#define OVERRIDE_DISPATCH_EVENT(Control)								\
	LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);	\
	friend class Manah::Windows::Controls::CCustomControl<Control>
#define OVERRIDE_DISPATCH_EVENT_(Control, BaseWindow)					\
	LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);	\
	friend class Manah::Windows::Controls::CCustomControl<Control, BaseWindow>


#define UNCONST_THIS	const_cast<CWindow*>(this)

inline CWindow::CWindow(HWND hWnd /* = 0 */) : m_hWnd(0), m_bAttached(false) {
	if(hWnd != 0)
		Attach(hWnd);
}

inline CWindow::CWindow(const CWindow& rhs) : m_hWnd(0), m_bAttached(false) {
	if(rhs.m_bAttached)
		Attach(rhs.m_hWnd);
}

inline CWindow::~CWindow() {
	if(IsWindow() && !m_bAttached)
		DestroyWindow();
}

inline UINT CWindow::ArrangeIconicWindows() {
	AssertValid();
	return ::ArrangeIconicWindows(m_hWnd);
}

inline bool CWindow::Attach(HWND hWnd) {
	AssertValid();

	if(IsWindow() || !toBoolean(::IsWindow(hWnd)))
		return false;
	m_hWnd = hWnd;
	m_bAttached = true;

	return true;
}

inline GDI::CPaintDC CWindow::BeginPaint(PAINTSTRUCT& paint) {
	AssertValidAsWindow();
	::BeginPaint(m_hWnd, &paint);
	return GDI::CPaintDC(m_hWnd, paint);
}

inline void CWindow::BringWindowToTop() {
	AssertValidAsWindow();
	::BringWindowToTop(m_hWnd);
}

inline void CWindow::CenterWindow(HWND hwndAlternate /* = 0 */) {
	AssertValidAsWindow();

	RECT rectWin, rectAlt;

	if(hwndAlternate == 0){
		hwndAlternate = ::GetParent(m_hWnd);
		if(hwndAlternate == 0)
			hwndAlternate = ::GetDesktopWindow();
	}
	assert(::IsWindow(hwndAlternate));

	GetWindowRect(rectWin);
	::GetWindowRect(hwndAlternate, &rectAlt);
	SetWindowPos(0, (rectAlt.right - rectAlt.left) / 2 - (rectWin.right - rectWin.left) / 2,
		(rectAlt.bottom - rectAlt.top) / 2 - (rectWin.bottom - rectWin.top) / 2,
		0, 0, SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOZORDER);
}

inline bool CWindow::ChangeClipboardChain(HWND hwndNewNext) {
	AssertValidAsWindow();
	return toBoolean(::ChangeClipboardChain(m_hWnd, hwndNewNext));
}

inline CWindow CWindow::ChildWindowFromPoint(const POINT& pt) const {
	AssertValidAsWindow();
	return CWindow(::ChildWindowFromPoint(m_hWnd, pt));
}

inline CWindow CWindow::ChildWindowFromPointEx(const POINT& pt, UINT nFlags) const {
	AssertValidAsWindow();
	return CWindow(::ChildWindowFromPointEx(m_hWnd, pt, nFlags));
}

inline void CWindow::ClientToScreen(RECT& rect) const {
	AssertValidAsWindow();

	POINT	point[2];

	point[0].x = rect.left;
	point[0].y = rect.top;
	point[1].x = rect.right;
	point[1].y = rect.bottom;
	::ClientToScreen(m_hWnd, &point[0]);
	::ClientToScreen(m_hWnd, &point[1]);
	::SetRect(&rect, point[0].x, point[0].y, point[1].x, point[1].y);
}

inline void CWindow::ClientToScreen(POINT& pt) const {
	AssertValidAsWindow();
	::ClientToScreen(m_hWnd, &pt);
}

inline bool CWindow::Create(const TCHAR* lpszClassName, HWND hwndParentOrHInstance,
		const RECT& rect /* = DefaultWindowRect() */, const TCHAR* lpszWindowName /* = 0 */,
		DWORD dwStyle /* = 0UL */, DWORD dwExStyle /* = 0UL */, HMENU hMenu /* = 0 */, void* lpParam /* = 0 */) {
	AssertValid();

	if(IsWindow())
		return false;

	HWND		hwndParent = toBoolean(::IsWindow(hwndParentOrHInstance)) ? hwndParentOrHInstance : 0;
	HINSTANCE	hInstance = (hwndParent != 0) ?
#ifdef _WIN64
					reinterpret_cast<HINSTANCE>(::GetWindowLongPtr(hwndParent, GWLP_HINSTANCE))
#else
					reinterpret_cast<HINSTANCE>(::GetWindowLong(hwndParent, GWL_HINSTANCE))
#endif /* _WIN64 */
					: reinterpret_cast<HINSTANCE>(hwndParentOrHInstance);

	return toBoolean(m_hWnd = ::CreateWindowEx(dwExStyle,
		lpszClassName, lpszWindowName, dwStyle, rect.left, rect.top,
		rect.right - rect.left, rect.bottom - rect.top, hwndParent, hMenu, hInstance, lpParam));
}

inline bool CWindow::CreateCaret(HBITMAP hBitmap, int nWidth, int nHeight) {
	AssertValidAsWindow();
	return toBoolean(::CreateCaret(m_hWnd, hBitmap, nWidth, nHeight));
}

inline bool CWindow::CreateGrayCaret(int nWidth, int nHeight) {
	AssertValidAsWindow();
	return toBoolean(::CreateCaret(m_hWnd, reinterpret_cast<HBITMAP>(1), nWidth, nHeight));
}

inline bool CWindow::CreateSolidCaret(int nWidth, int nHeight) {
	AssertValidAsWindow();
	return toBoolean(::CreateCaret(m_hWnd, 0, nWidth, nHeight));
}

inline LRESULT CWindow::DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValidAsWindow();
	return ::DefWindowProc(m_hWnd, message, wParam, lParam);
}

inline bool CWindow::DestroyWindow() {
	AssertValid();
	return IsWindow() ? toBoolean(::DestroyWindow(m_hWnd)) : false;
}

inline HWND CWindow::Detach() {
	AssertValid();

	if(!m_bAttached)
		return 0;

	HWND	hWnd = m_hWnd;

	m_hWnd = 0;
	m_bAttached = false;

	return hWnd;
}

inline LRESULT CWindow::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	POINT pt;

	if(PreTranslateMessage(message, wParam, lParam))
		return false;

	switch(message) {
	case WM_ACTIVATE:
		OnActivate(LOWORD(wParam), reinterpret_cast<HWND>(lParam), toBoolean(HIWORD(wParam)));
		break;
	case WM_CLOSE:
		OnClose();
		return 0L;
	case WM_COMMAND:
		return OnCommand(LOWORD(wParam), HIWORD(wParam), reinterpret_cast<HWND>(lParam));
	case WM_CONTEXTMENU:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		if(OnContextMenu(reinterpret_cast<HWND>(wParam), pt))
			return 0L;
		break;
	case WM_DESTROY:
		OnDestroy();
		return 0L;
	case WM_DRAWITEM:
		OnDrawItem(static_cast<UINT>(wParam), reinterpret_cast<LPDRAWITEMSTRUCT>(lParam));
		break;
	case WM_KILLFOCUS:
		OnKillFocus(reinterpret_cast<HWND>(wParam));
		return 0L;
	case WM_LBUTTONDOWN:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonDown(wParam, pt);
		break;
	case WM_LBUTTONDBLCLK:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonDblClk(wParam, pt);
		break;
	case WM_LBUTTONUP:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonUp(wParam, pt);
		break;
	case WM_MEASUREITEM:
		OnMeasureItem(static_cast<UINT>(wParam), reinterpret_cast<LPMEASUREITEMSTRUCT>(lParam));
		break;
	case WM_MOUSEMOVE:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnMouseMove(wParam, pt);
		break;
	case WM_NOTIFY:
		return OnNotify(wParam, reinterpret_cast<LPNMHDR>(lParam));
	case WM_SETCURSOR:
		if(OnSetCursor(reinterpret_cast<HWND>(wParam),
			static_cast<UINT>(LOWORD(lParam)), static_cast<UINT>(HIWORD(lParam))))
			return false;
		break;
	case WM_SETFOCUS:
		OnSetFocus(reinterpret_cast<HWND>(wParam));
		return 0L;
	case WM_SHOWWINDOW:
		OnShowWindow(toBoolean(wParam), lParam);
		return 0L;
	case WM_SIZE:
		OnSize(wParam, LOWORD(lParam), HIWORD(lParam));
		break;
	case WM_SYSCOLORCHANGE:
		OnSysColorChange();
		break;
	case WM_TIMER:
		OnTimer(wParam);
		break;
	default:
		break;
	}

	return DefWindowProc(message, wParam, lParam);
}

inline void CWindow::DragAcceptFiles(bool bAccept /* = true */) {
	AssertValidAsWindow();
	::DragAcceptFiles(m_hWnd, bAccept);
}

inline void CWindow::DrawMenuBar() {
	AssertValidAsWindow();
	::DrawMenuBar(m_hWnd);
}

inline bool CWindow::EnableScrollBar(int nSBFlags, UINT nArrowFlags /* = ESB_ENABLE_BOTH */) {
	AssertValidAsWindow();
	return toBoolean(::EnableScrollBar(m_hWnd, nSBFlags, nArrowFlags));
}

inline bool CWindow::EnableWindow(bool bEnable /* = true */) {
	AssertValidAsWindow();
	return toBoolean(::EnableWindow(m_hWnd, bEnable));
}

inline void CWindow::EndPaint(const PAINTSTRUCT& paint) {
	AssertValidAsWindow();
	::EndPaint(m_hWnd, &paint);
}

inline int CWindow::EnumProps(PROPENUMPROC lpEnumFunc) {
	AssertValidAsWindow();
	return ::EnumProps(m_hWnd, lpEnumFunc);
}

inline int CWindow::EnumPropsEx(PROPENUMPROCEX lpEnumProc, LPARAM lParam) {
	AssertValidAsWindow();
	return ::EnumPropsEx(m_hWnd, lpEnumProc, lParam);
}

inline CWindow CWindow::FindWindow(const TCHAR* lpszClassName, const TCHAR* lpszWindowName) {
	return CWindow(::FindWindow(lpszClassName, lpszWindowName));
}

inline bool CWindow::FlashWindow(bool bInvert) {
	AssertValidAsWindow();
	return toBoolean(::FlashWindow(m_hWnd, bInvert));
}

inline CWindow CWindow::GetActiveWindow() {
	return CWindow(::GetActiveWindow());
}

inline CWindow CWindow::GetCapture() {
	return CWindow(::GetCapture());
}

inline LPPOINT CWindow::GetCaretPos() {
	static POINT pt;

	::GetCaretPos(&pt);
	return &pt;
}

inline DWORD CWindow::GetClassLong(int nIndex) const {
	AssertValidAsWindow();
	return ::GetClassLong(m_hWnd, nIndex);
}

#ifdef _WIN64
inline ULONG_PTR CWindow::GetClassLongPtr(int nIndex) const {
	AssertValidAsWindow();
	return ::GetClassLongPtr(m_hWnd, nIndex);
}
#endif /* _WIN64 */

inline int CWindow::GetClassName(TCHAR* lpszClassName, int cchMax) const {
	AssertValidAsWindow();
	return ::GetClassName(m_hWnd, lpszClassName, cchMax);
}

inline void CWindow::GetClientRect(RECT& rect) const {
	AssertValidAsWindow();
	::GetClientRect(m_hWnd, &rect);
}

inline GDI::CClientDC CWindow::GetDC() {
	AssertValidAsWindow();
	return GDI::CClientDC(m_hWnd);
}

inline GDI::CClientDC CWindow::GetDCEx(HRGN hrgnClip, DWORD dwFlags) {
	AssertValidAsWindow();
	return GDI::CClientDC(m_hWnd, hrgnClip, dwFlags);
}

inline CWindow CWindow::GetDesktopWindow() {
	return CWindow(::GetDesktopWindow());
}

inline int CWindow::GetDlgCtrlID() const {
	AssertValidAsWindow();
	return ::GetDlgCtrlID(m_hWnd);
}

inline DWORD CWindow::GetExStyle() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::GetWindowLong(m_hWnd, GWL_EXSTYLE));
}

inline CWindow CWindow::GetFocus() {
	return CWindow(::GetFocus());
}

inline HFONT CWindow::GetFont() const {
	AssertValidAsWindow();
	return reinterpret_cast<HFONT>(UNCONST_THIS->SendMessage(WM_GETFONT));
}

inline CWindow CWindow::GetForegroundWindow() {
	return CWindow(::GetForegroundWindow());
}

inline DWORD CWindow::GetHotKey() const {
	AssertValidAsWindow();
	return UNCONST_THIS->SendMessage(WM_GETHOTKEY);
}

inline HICON CWindow::GetIcon(bool bBigIcon /* = true */) const {
	AssertValidAsWindow();
	return reinterpret_cast<HICON>(UNCONST_THIS->SendMessage(WM_GETICON, bBigIcon ? ICON_BIG : ICON_SMALL, 0L));
}

inline CWindow CWindow::GetLastActivePopup() const {
	AssertValidAsWindow();
	return CWindow(::GetLastActivePopup(m_hWnd));
}

inline HMENU CWindow::GetMenu() const {
	AssertValidAsWindow();
	return ::GetMenu(m_hWnd);
}

inline CWindow CWindow::GetNext(UINT nFlag /* = GW_HWNDNEXT */) const {
	AssertValidAsWindow();
	return CWindow(::GetNextWindow(m_hWnd, nFlag));
}

inline CWindow CWindow::GetOwner() const {
	AssertValidAsWindow();
	return CWindow(::GetWindow(m_hWnd, GW_OWNER));
}

inline CWindow CWindow::GetParent() const {
	AssertValidAsWindow();
	return CWindow(::GetParent(m_hWnd));
}

inline HANDLE CWindow::GetProp(const TCHAR* lpsz) const {
	AssertValidAsWindow();
	return ::GetProp(m_hWnd, lpsz);
}

inline HWND CWindow::GetSafeHwnd() const {
	return (this != 0) ? m_hWnd : 0;
}

inline bool CWindow::GetScrollInfo(int nBar, SCROLLINFO& scrollInfo, UINT nMask /* = SIF_ALL */) const {
	AssertValidAsWindow();

	scrollInfo.cbSize = sizeof(SCROLLINFO);
	scrollInfo.fMask = nMask;

	return toBoolean(::GetScrollInfo(m_hWnd, nBar, &scrollInfo));
}

inline int CWindow::GetScrollLimit(int nBar) const {
	AssertValidAsWindow();

	int nLimit, nDummy;

	::GetScrollRange(m_hWnd, nBar, &nDummy, &nLimit);
	return nLimit;
}

inline int CWindow::GetScrollPos(int nBar) const {
	AssertValidAsWindow();
	return ::GetScrollPos(m_hWnd, nBar);
}

inline void CWindow::GetScrollRange(int nBar, int* pMinPos, int* pMaxPos) const {
	AssertValidAsWindow();
	::GetScrollRange(m_hWnd, nBar, pMinPos, pMaxPos);
}

inline int CWindow::GetScrollTrackPos(int nBar) const {
	AssertValidAsWindow();
	SCROLLINFO	si;
	return (GetScrollInfo(nBar, si, SIF_TRACKPOS)) ? si.nTrackPos : -1;
}

inline DWORD CWindow::GetStyle() const {
	AssertValidAsWindow();
	return static_cast<DWORD>(::GetWindowLong(m_hWnd, GWL_STYLE));
}

inline HMENU CWindow::GetSystemMenu(bool bRevert) const {
	AssertValidAsWindow();
	return ::GetSystemMenu(m_hWnd, bRevert);
}

inline CWindow CWindow::GetTopWindow() const {
	AssertValidAsWindow();
	return CWindow(::GetTopWindow(m_hWnd));
}

inline bool CWindow::GetUpdateRect(RECT& rect, bool bErase /* = false */) {
	AssertValidAsWindow();
	return toBoolean(::GetUpdateRect(m_hWnd, &rect, bErase));
}

inline int CWindow::GetUpdateRgn(HRGN hRgn, bool bErase /* = false */) {
	AssertValidAsWindow();
	return ::GetUpdateRgn(m_hWnd, hRgn, bErase);
}

inline CWindow CWindow::GetWindow(UINT nCmd) const {
	AssertValidAsWindow();
	return CWindow(::GetWindow(m_hWnd, nCmd));
}

inline DWORD CWindow::GetWindowContextHelpId() const {
	AssertValidAsWindow();
	return ::GetWindowContextHelpId(m_hWnd);
}

inline GDI::CWindowDC CWindow::GetWindowDC() {
	AssertValidAsWindow();
	return GDI::CWindowDC(m_hWnd);
}

inline LONG CWindow::GetWindowLong(int nIndex) const {
	AssertValidAsWindow();
	return ::GetWindowLong(m_hWnd, nIndex);
}

#ifdef _WIN64
inline LONG_PTR CWindow::GetWindowLongPtr(int nIndex) const {
	AssertValidAsWindow();
	return ::GetWindowLongPtr(m_hWnd, nIndex);
}
#endif /* _WIN64 */

inline bool CWindow::GetWindowPlacement(WINDOWPLACEMENT& wndpl) const {
	AssertValidAsWindow();
	return toBoolean(::GetWindowPlacement(m_hWnd, &wndpl));
}

inline DWORD CWindow::GetWindowProcessId() const {
	AssertValidAsWindow();
	DWORD	id;
	::GetWindowThreadProcessId(m_hWnd, &id);
	return id;
}

inline void CWindow::GetWindowRect(RECT& rect) const {
	AssertValidAsWindow();
	::GetWindowRect(m_hWnd, &rect);
}

inline int CWindow::GetWindowRgn(HRGN hRgn) const {
	AssertValidAsWindow();
	return ::GetWindowRgn(m_hWnd, hRgn);
}

inline int CWindow::GetWindowText(TCHAR* lpszText, int nMaxCount) const {
	AssertValidAsWindow();
	return ::GetWindowText(m_hWnd, lpszText, nMaxCount);
}

inline int CWindow::GetWindowTextLength() const {
	AssertValidAsWindow();
	return ::GetWindowTextLength(m_hWnd);
}

inline DWORD CWindow::GetWindowThreadId() const {
	AssertValidAsWindow();
	return ::GetWindowThreadProcessId(m_hWnd, 0);
}

inline bool CWindow::HasFocus() const {
	AssertValid();
	return ::GetFocus() == m_hWnd;
}

inline void CWindow::HideCaret() {
	AssertValidAsWindow();
	::HideCaret(m_hWnd);
}

inline bool CWindow::HiliteMenuItem(HMENU hMenu, UINT nItem, UINT nFlags) {
	AssertValidAsWindow();
	return toBoolean(::HiliteMenuItem(m_hWnd, hMenu, nItem, nFlags));
}

inline void CWindow::Invalidate(bool bErase /* = true */) {
	AssertValidAsWindow();
	::InvalidateRect(m_hWnd, 0, bErase);
}

inline void CWindow::InvalidateRect(const RECT* lpRect, bool bErase /* = true */) {
	AssertValidAsWindow();
	::InvalidateRect(m_hWnd, lpRect, bErase);
}

inline void CWindow::InvalidateRgn(HRGN hRgn, bool bErase /* = true */) {
	AssertValidAsWindow();
	::InvalidateRgn(m_hWnd, hRgn, bErase);
}

inline bool CWindow::IsChild(HWND hWnd) const {
	AssertValidAsWindow();
	return toBoolean(::IsChild(m_hWnd, hWnd));
}

inline bool CWindow::IsIconic() const {
	AssertValidAsWindow();
	return toBoolean(::IsIconic(m_hWnd));
}

inline bool CWindow::IsWindow() const {
	AssertValid();
	return toBoolean(::IsWindow(m_hWnd));
}

inline bool CWindow::IsWindowEnabled() const {
	AssertValidAsWindow();
	return toBoolean(::IsWindowEnabled(m_hWnd));
}

inline bool CWindow::IsWindowUnicode() const {
	AssertValidAsWindow();
	return toBoolean(::IsWindowUnicode(m_hWnd));
}

inline bool CWindow::IsWindowVisible() const {
	AssertValid();
	return toBoolean(::IsWindowVisible(m_hWnd));
}

inline bool CWindow::IsZoomed() const {
	AssertValidAsWindow();
	return toBoolean(::IsZoomed(m_hWnd));
}

inline bool CWindow::KillTimer(UINT nIDEvent) {
	AssertValidAsWindow();
	return toBoolean(::KillTimer(m_hWnd, nIDEvent));
}

inline bool CWindow::LockWindowUpdate() {
	AssertValidAsWindow();
	return toBoolean(::LockWindowUpdate(m_hWnd));
}

inline void CWindow::MapWindowPoints(HWND hWndTo, RECT& rect) const {
	AssertValidAsWindow();

	POINT point[2];

	point[0].x = rect.left;
	point[0].y = rect.top;
	point[1].x = rect.right;
	point[1].y = rect.bottom;
	::MapWindowPoints(m_hWnd, hWndTo, point, 2);
	::SetRect(&rect, point[0].x, point[0].y, point[1].x, point[1].y);
}

inline void CWindow::MapWindowPoints(HWND hWndTo, POINT* lpPoint, UINT nCount) const {
	AssertValidAsWindow();
	::MapWindowPoints(m_hWnd, hWndTo, lpPoint, nCount);
}

inline int CWindow::MessageBox(const TCHAR* lpszText, const TCHAR* lpszCaption /* = 0 */, UINT nType /* = MB_OK */) {
	AssertValidAsWindow();
	return ::MessageBox(m_hWnd, lpszText, lpszCaption, nType);
}

inline bool CWindow::ModifyStyle(DWORD dwRemove, DWORD dwAdd) {
	AssertValidAsWindow();

	DWORD dwStyle;

	dwStyle = GetStyle();
	dwStyle &= ~dwRemove;
	dwStyle |= dwAdd;
	if(SetWindowLong(GWL_STYLE, dwStyle) == 0)
		return false;
	return true;
}

inline bool CWindow::ModifyStyleEx(DWORD dwRemove, DWORD dwAdd) {
	AssertValidAsWindow();

	DWORD dwExStyle;

	dwExStyle = GetExStyle();
	dwExStyle &= ~dwRemove;
	dwExStyle |= dwAdd;
	if(SetWindowLong(GWL_EXSTYLE, dwExStyle) == 0)
		return false;
	return true;
}

inline void CWindow::MoveWindow(int x, int y, int nWidth, int nHeight, bool bRepaint /* = true */) {
	AssertValidAsWindow();
	::MoveWindow(m_hWnd, x, y, nWidth, nHeight, bRepaint);
}

inline void CWindow::Print(HDC hDC, DWORD dwFlags) const {
	AssertValidAsWindow();
	UNCONST_THIS->SendMessage(WM_PRINT, reinterpret_cast<WPARAM>(hDC), dwFlags);
}

inline void CWindow::PrintClient(HDC hDC, DWORD dwFlags) const {
	AssertValidAsWindow();
	UNCONST_THIS->SendMessage(WM_PRINTCLIENT, reinterpret_cast<WPARAM>(hDC), dwFlags);
}

inline void CWindow::MoveWindow(const RECT& rect, bool bRepaint /* = true */) {
	AssertValidAsWindow();
	::MoveWindow(m_hWnd, rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top, bRepaint);
}

inline LRESULT CWindow::PostMessage(UINT message, WPARAM wParam /* = 0 */, LPARAM lParam /* = 0L */) {
	AssertValidAsWindow();
	return ::PostMessage(m_hWnd, message, wParam, lParam);
}

inline bool CWindow::PreTranslateMessage(UINT message, WPARAM wParam, LPARAM lParam) {
	return false;
}

inline bool CWindow::RedrawWindow(
		LPRECT lpRectUpdate /* = 0 */, HRGN hrgnClip /* = 0 */,
		UINT nFlags /* = RDW_INVALIDATE | RDW_UPDATENOW | RDW_ERASE */) {
	AssertValidAsWindow();
	return toBoolean(::RedrawWindow(m_hWnd, lpRectUpdate, hrgnClip, nFlags));
}

inline HRESULT CWindow::RegisterDragDrop(IDropTarget& dropTarget) {
	AssertValidAsWindow();
	return ::RegisterDragDrop(m_hWnd, &dropTarget);
}

inline bool CWindow::ReleaseCapture() {
	return toBoolean(::ReleaseCapture());
}

inline int CWindow::ReleaseDC(HDC hDC) {
	AssertValidAsWindow();
	return ::ReleaseDC(m_hWnd, hDC);
}

inline HANDLE CWindow::RemoveProp(const TCHAR* lpsz) {
	AssertValidAsWindow();
	return ::RemoveProp(m_hWnd, lpsz);
}

inline HRESULT CWindow::RevokeDragDrop() {
	AssertValidAsWindow();
	return ::RevokeDragDrop(m_hWnd);
}

inline void CWindow::ScreenToClient(RECT& rect) const {
	AssertValidAsWindow();

	POINT	point[2];

	point[0].x = rect.left;
	point[0].y = rect.top;
	point[1].x = rect.right;
	point[1].y = rect.bottom;
	::ScreenToClient(m_hWnd, &point[0]);
	::ScreenToClient(m_hWnd, &point[1]);
	::SetRect(&rect, point[0].x, point[0].y, point[1].x, point[1].y);
}

inline void CWindow::ScrollWindow(int xAmount, int yAmount, LPRECT lpRect /* = 0 */, LPRECT lpClipRect /* = 0 */) {
	AssertValidAsWindow();
	::ScrollWindow(m_hWnd, xAmount, yAmount, lpRect, lpClipRect);
}

inline int CWindow::ScrollWindowEx(
		int dx, int dy, LPRECT lpScrollRect, LPRECT lpClipRect,
		HRGN hrgnUpdate, LPRECT lpUpdateRect, UINT nFlags) {
	AssertValidAsWindow();
	return ::ScrollWindowEx(m_hWnd, dx, dy, lpScrollRect, lpClipRect, hrgnUpdate, lpUpdateRect, nFlags);
}

inline LRESULT CWindow::SendMessage(UINT message, WPARAM wParam /* = 0 */, LPARAM lParam /* = 0L */) {
	AssertValidAsWindow();
	return ::SendMessage(m_hWnd, message, wParam, lParam);
}

inline bool CWindow::SendNotifyMessage(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValidAsWindow();
	return toBoolean(::SendNotifyMessage(m_hWnd, message, wParam, lParam));
}

inline void CWindow::ScreenToClient(POINT& pt) const {
	AssertValidAsWindow();
	::ScreenToClient(m_hWnd, &pt);
}

inline CWindow CWindow::SetActiveWindow() {
	AssertValidAsWindow();
	return CWindow(::SetActiveWindow(m_hWnd));
}

inline CWindow CWindow::SetCapture() {
	AssertValidAsWindow();
	return CWindow(::SetCapture(m_hWnd));
}

inline void CWindow::SetCaretPos(POINT pt) {
	::SetCaretPos(pt.x, pt.y);
}

inline DWORD CWindow::SetClassLong(int nIndex, DWORD dwNewLong) {
	AssertValidAsWindow();
	return ::SetClassLong(m_hWnd, nIndex, dwNewLong);
}

#ifdef _WIN64
inline ULONG_PTR CWindow::SetClassLongPtr(int nIndex, ULONG_PTR dwNewLong) {
	AssertValidAsWindow();
	return ::SetClassLongPtr(nIndex, dwNewLong);
}
#endif /* _WIN64 */

inline CWindow CWindow::SetClipboardViewer() {
	AssertValidAsWindow();
	return CWindow(::SetClipboardViewer(m_hWnd));
}

inline int CWindow::SetDlgCtrlID(int nID) {
	AssertValidAsWindow();
	return static_cast<int>(SetWindowLong(GWL_ID, nID));
}

inline CWindow CWindow::SetFocus() {
	AssertValidAsWindow();
	return CWindow(::SetFocus(m_hWnd));
}

inline void CWindow::SetFont(HFONT hFont, bool bRedraw /* = true */) {
	SendMessage(WM_SETFONT, reinterpret_cast<WPARAM>(hFont), MAKELPARAM(bRedraw, 0));
}

inline bool CWindow::SetForegroundWindow() {
	AssertValidAsWindow();
	return toBoolean(::SetForegroundWindow(m_hWnd));
}

inline int CWindow::SetHotKey(WORD nVirtualKeyCode, WORD nModifiers) {
	AssertValidAsWindow();
	return static_cast<int>(SendMessage(WM_SETHOTKEY, MAKEWPARAM(nVirtualKeyCode, nModifiers)));
}

inline HICON CWindow::SetIcon(HICON hIcon, bool bBigIcon /* = true */) {
	return reinterpret_cast<HICON>(SendMessage(WM_SETICON, bBigIcon ? ICON_BIG : ICON_SMALL));
}

inline bool CWindow::SetMenu(HMENU hMenu) {
	AssertValidAsWindow();
	return toBoolean(::SetMenu(m_hWnd, hMenu));
}

inline CWindow CWindow::SetParent(HWND hwndNewParent) {
	AssertValidAsWindow();
	return CWindow(::SetParent(m_hWnd, hwndNewParent));
}

inline bool CWindow::SetProp(const TCHAR* lpsz, HANDLE hData) {
	AssertValidAsWindow();
	return toBoolean(::SetProp(m_hWnd, lpsz, hData));
}

inline void CWindow::SetRedraw(bool bRedraw /* = true */) {
	SendMessage(WM_SETREDRAW, bRedraw);
}

inline bool CWindow::SetScrollInfo(int nBar, const SCROLLINFO& scrollInfo, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	return toBoolean(::SetScrollInfo(m_hWnd, nBar, &scrollInfo, bRedraw));
}

inline int CWindow::SetScrollPos(int nBar, int nPos, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	return ::SetScrollPos(m_hWnd, nBar, nPos, bRedraw);
}

inline void CWindow::SetScrollRange(int nBar, int nMinPos, int nMaxPos, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	::SetScrollRange(m_hWnd, nBar, nMinPos, nMaxPos, bRedraw);
}

inline timerid_t CWindow::SetTimer(
		timerid_t nIDEvent, UINT nElapse, void (CALLBACK* lpfnTimer)(HWND, UINT, timerid_t, DWORD) /* = 0 */) {
	AssertValidAsWindow();
	return ::SetTimer(m_hWnd, nIDEvent, nElapse, lpfnTimer);
}

inline bool CWindow::SetWindowContextHelpId(DWORD dwContextHelpId) {
	AssertValidAsWindow();
	return toBoolean(::SetWindowContextHelpId(m_hWnd, dwContextHelpId));
}

inline LONG CWindow::SetWindowLong(int nIndex, LONG dwNewLong) {
	AssertValidAsWindow();
	return ::SetWindowLong(m_hWnd, nIndex, dwNewLong);
}

#ifdef _WIN64
inline LONG_PTR CWindow::SetWindowLongPtr(int nIndex, LONG_PTR dwNewLong) {
	AssertValidAsWindow();
	return ::SetWindowLongPtr(m_hWnd, nIndex, dwNewLong);
}
#endif /* _WIN64 */

inline bool CWindow::SetWindowPlacement(const WINDOWPLACEMENT& wndpl) {
	AssertValidAsWindow();
	return toBoolean(::SetWindowPlacement(m_hWnd, &wndpl));
}

inline bool CWindow::SetWindowPos(HWND hwndInsertAfter, int x, int y, int cx, int cy, UINT nFlags) {
	AssertValidAsWindow();
	return toBoolean(::SetWindowPos(m_hWnd, hwndInsertAfter, x, y, cx, cy, nFlags));
}

inline bool CWindow::SetWindowPos(HWND hwndInsertAfter, const RECT& rect, UINT nFlags) {
	AssertValidAsWindow();
	return toBoolean(
		::SetWindowPos(m_hWnd, hwndInsertAfter, rect.left, rect.top,
			rect.right - rect.left, rect.bottom - rect.top, nFlags));
}

inline int CWindow::SetWindowRgn(const HRGN hRgn, bool bRedraw /* = true */) {
	AssertValidAsWindow();
	return ::SetWindowRgn(m_hWnd, hRgn, bRedraw);
}

inline void CWindow::SetWindowText(const TCHAR* lpszText) {
	AssertValidAsWindow();
	::SetWindowText(m_hWnd, lpszText);
}

inline void CWindow::ShowCaret() {
	AssertValidAsWindow();
	::ShowCaret(m_hWnd);
}

inline void CWindow::ShowOwnedPopups(bool bShow /* = true */) {
	AssertValidAsWindow();
	::ShowOwnedPopups(m_hWnd, bShow);
}

inline void CWindow::ShowScrollBar(int nBar, bool bShow /* = true */) {
	AssertValidAsWindow();
	::ShowScrollBar(m_hWnd, nBar, bShow);
}

inline bool CWindow::ShowWindow(UINT nCmdShow) {
	AssertValidAsWindow();
	return toBoolean(::ShowWindow(m_hWnd, nCmdShow));
}

inline void CWindow::UnlockWindowUpdate() {
	AssertValidAsWindow();
	::LockWindowUpdate(0);
}

inline void CWindow::UpdateWindow() {
	AssertValidAsWindow();
	::UpdateWindow(m_hWnd);
}

inline void CWindow::ValidateRect(const RECT* lpRect) {
	AssertValidAsWindow();
	::ValidateRect(m_hWnd, lpRect);
}

inline void CWindow::ValidateRgn(HRGN hRgn) {
	AssertValidAsWindow();
	::ValidateRgn(m_hWnd, hRgn);
}

inline CWindow CWindow::WindowFromPoint(const POINT& pt) {
	return CWindow(::WindowFromPoint(pt));
}

inline bool CWindow::WinHelp(const TCHAR* lpszHelp, UINT nCommand /* = HELP_CONTEXT */, DWORD dwData /* = 0 */) {
	return toBoolean(::WinHelp(m_hWnd, lpszHelp, nCommand, dwData));
}

inline void CWindow::OnActivate(UINT nState, HWND hwndPrevious, bool bMinimized) {
}

inline void CWindow::OnClose() {
	DestroyWindow();
}

inline bool CWindow::OnCommand(WORD wID, WORD wNotifyCode, HWND hwndCtrl) {
	return false;
}

inline bool CWindow::OnContextMenu(HWND hWnd, POINT pt) {
	return false;
}

inline void CWindow::OnDrawItem(UINT idCtrl, LPDRAWITEMSTRUCT lpDrawItem) {
}

inline void CWindow::OnDestroy() {
}

inline void CWindow::OnKillFocus(HWND hwndNew) {
}

inline void CWindow::OnLButtonDown(UINT nFlags, POINT pt) {
}

inline void CWindow::OnLButtonDblClk(UINT nFlags, POINT pt) {
}

inline void CWindow::OnLButtonUp(UINT nFlags, POINT pt) {
}

inline void CWindow::OnMeasureItem(UINT idCtrl, LPMEASUREITEMSTRUCT lpMI) {
}

inline void CWindow::OnMouseMove(UINT nFlags, POINT pt) {
}

inline bool CWindow::OnNotify(int idCtrl, LPNMHDR lpNMHDR) {
	return false;
}

inline bool CWindow::OnSetCursor(HWND hWnd, UINT nHitTest, UINT message) {
	return false;
}

inline void CWindow::OnSetFocus(HWND hwndOld) {
}

inline void CWindow::OnShowWindow(bool bShow, UINT nStatus) {
}

inline void CWindow::OnSize(UINT nType, int cx, int cy) {
}

inline void CWindow::OnSysColorChange() {
}

inline void CWindow::OnTimer(UINT nIDEvent) {
}


template<class Window> inline bool CSubclassable<Window>::Attach(HWND hWnd, bool bSubclass /* = false */) {
	if(bSubclass)	return Window::Attach(hWnd) ? SubclassWindow() : false;
	else			return Window::Attach(hWnd);
}

template<class Window> inline LRESULT CSubclassable<Window>::DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValidAsWindow();
	return (m_pfnOldProc != 0) ?
		::CallWindowProc(m_pfnOldProc, m_hWnd, message, wParam, lParam)
		: ::DefWindowProc(m_hWnd, message, wParam, lParam);
}

template<class Window> inline HWND CSubclassable<Window>::Detach() {
	if(m_bAttached)
		UnsubclassWindow();
	return Window::Detach();
}

template<class Window> inline LRESULT CSubclassable<Window>::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	if(message == WM_NCDESTROY) {
		UnsubclassWindow();
		return 0L;
	}
	return Window::DispatchEvent(message, wParam, lParam);
}

template<class Window> inline void CSubclassable<Window>::OnDestroy() {
	UnsubclassWindow();
}

template<class Window> inline bool CSubclassable<Window>::SubclassWindow() {
	AssertValidAsWindow();
	if(IsSubclassed())
		return false;
#ifdef _WIN64
	else if(m_pfnOldProc = reinterpret_cast<WNDPROC>(GetWindowLongPtr(GWLP_WNDPROC))) {
		SetWindowLongPtr(GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(WndProc));
		SetWindowLongPtr(GWLP_USERDATA, reinterpret_cast<LONG_PTR>(this));
#else
	else if(m_pfnOldProc = reinterpret_cast<WNDPROC>(GetWindowLong(GWL_WNDPROC))) {
		SetWindowLong(GWL_WNDPROC, reinterpret_cast<long>(WndProc));
		SetWindowLong(GWL_USERDATA, reinterpret_cast<long>(this));
#endif /* _WIN64 */
		return true;
	} else
		return false;
}

template<class Window> inline bool CSubclassable<Window>::UnsubclassWindow() {
	AssertValidAsWindow();
	if(!IsSubclassed())
		return false;
#ifdef _WIN64
	SetWindowLongPtr(GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(m_pfnOldProc));
	SetWindowLongPtr(GWLP_USERDATA, 0);
#else
	SetWindowLong(GWL_WNDPROC, reinterpret_cast<long>(m_pfnOldProc));
	SetWindowLong(GWL_USERDATA, 0L);
#endif /* _WIN64 */
	m_pfnOldProc = 0;
	return true;
}

template<class Window>
inline LRESULT CALLBACK CSubclassable<Window>::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
#ifdef _WIN64
	if(CSubclassable<Window>* pWindow = reinterpret_cast<CSubclassable<Window>*>(::GetWindowLongPtr(hWnd, GWLP_USERDATA)))
#else
	if(CSubclassable<Window>* pWindow = reinterpret_cast<CSubclassable<Window>*>(::GetWindowLong(hWnd, GWL_USERDATA)))
#endif /* _WIN64 */
		return pWindow->DispatchEvent(message, wParam, lParam);
	return false;
}


template<class Window>
inline bool CLayered<Window>::AnimateWindow(DWORD dwTime, DWORD dwFlags, bool bCatchError /* = true */) {
	AssertValidAsWindow();
	if(m_hUser32Dll == 0)
		m_hUser32Dll = ::LoadLibrary(_T("User32.dll"));
	if(m_hUser32Dll != 0 && m_pfnAnimateWindow == 0)
		m_pfnAnimateWindow =
			reinterpret_cast<AW>(::GetProcAddress(m_hUser32Dll, "AnimateWindow"));
	if((m_pfnAnimateWindow != 0) ?
			toBoolean((*m_pfnAnimateWindow)(m_hWnd, dwTime, dwFlags)) : false)
		return true;
	else if(bCatchError) {
		if(toBoolean(dwFlags & AW_HIDE))
			return ShowWindow(SW_HIDE);
		else
			return ShowWindow(toBoolean(dwFlags & AW_ACTIVATE) ? SW_SHOW : SW_SHOWNA);
	}
	return false;
}

template<class Window>
inline bool CLayered<Window>::SetLayeredWindowAttributes(COLORREF clrKey, BYTE bAlpha, DWORD dwFlags) {
	AssertValidAsWindow();
	if(m_hUser32Dll == 0)
		m_hUser32Dll = ::LoadLibrary(_T("User32.dll"));
	if(m_hUser32Dll != 0 && m_pfnSetLayeredWindowAttributes == 0)
		m_pfnSetLayeredWindowAttributes =
			reinterpret_cast<SLWA>(::GetProcAddress(m_hUser32Dll, "SetLayeredWindowAttributes"));
	return (m_pfnSetLayeredWindowAttributes != 0) ?
		toBoolean((*m_pfnSetLayeredWindowAttributes)(m_hWnd, clrKey, bAlpha, dwFlags)) : false;
}

template<class Window>
inline bool CLayered<Window>::UpdateLayeredWindow(HDC hDest, POINT* pptDest,
		SIZE* pSize, HDC hSrc, POINT* pptSrc, COLORREF clrKey, BLENDFUNCTION* pBlend, DWORD dwFlags) {
	AssertValidAsWindow();
	if(m_hUser32Dll == 0)
		m_hUser32Dll = ::LoadLibrary(_T("User32.dll"));
	if(m_hUser32Dll != 0 && m_pfnUpdateLayeredWindow == 0)
		m_pfnUpdateLayeredWindow =
			reinterpret_cast<ULW>(::GetProcAddress(m_hUser32Dll, "UpdateLayeredWindow"));
	return (m_pfnUpdateLayeredWindow != 0) ?
		toBoolean((*m_pfnUpdateLayeredWindow)(m_hWnd, hDest, pptDest, pSize, hSrc, pptSrc, clrKey, pBlend, dwFlags)) : false;
}

template<class Control>
inline bool	CStandardControl<Control>::Create(HWND hwndParent, const RECT& rect /* = DefaultWindowRect() */,
		const TCHAR* lpszWindowName /* = 0 */, int id /* = 0 */,
		DWORD dwStyle /* = DefaultStyle */, DWORD dwExStyle /* = 0UL */) {
	return CWindow::Create(Control::GetClassName(), hwndParent,
		rect, lpszWindowName, dwStyle, dwExStyle, reinterpret_cast<HMENU>(id));
}


template<class Control, class BaseWindow> inline CCustomControl<Control, BaseWindow>::~CCustomControl() {
	// prevent to be called as this by WndProc
	if(IsWindow())
#ifdef _WIN64
		SetWindowLongPtr(GWLP_USERDATA, 0);
#else
		SetWindowLong(GWL_USERDATA, 0L);
#endif /* _WIN64 */
}

template<class Control, class BaseWindow>
inline bool CCustomControl<Control, BaseWindow>::Create(HWND hwndParent, const RECT& rect /* = DefaultWindowRect() */,
		const TCHAR* lpszWindowName /* = 0 */, DWORD dwStyle /* = 0UL */, DWORD dwExStyle /* = 0UL */) {
	BrushHandleOrColor	bgColor;
	CursorHandleOrId	cursor;
	AutoZeroCB<WNDCLASSEX>	wc, dummy;

	wc.hInstance = ::GetModuleHandle(0);	// default value
	wc.lpfnWndProc = CCustomControl<Control, BaseWindow>::WndProc;
	Control::GetClass(wc.lpszClassName, wc.hInstance, wc.style,
		bgColor, cursor, wc.hIcon, wc.hIconSm, wc.cbClsExtra, wc.cbWndExtra);
	wc.hbrBackground = bgColor.m_hBrush;
	wc.hCursor = cursor.m_hCursor;
	if(::GetClassInfoEx(wc.hInstance, wc.lpszClassName, &dummy) == 0)
		::RegisterClassEx(&wc);
	if(CWindow::Create(wc.lpszClassName, hwndParent, rect, lpszWindowName, dwStyle, dwExStyle, 0, this)) {
		assert(m_bAttached);	// attached by WndProc method
		m_hWnd = Detach();
		return true;
	} else
		return false;
}

template<class Control, class BaseWindow>
inline LRESULT CALLBACK CCustomControl<Control, BaseWindow>::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	typedef CCustomControl<Control, BaseWindow>	Window;
	if(message == WM_CREATE) {
		Window*	pWindow = reinterpret_cast<Window*>(
			reinterpret_cast<CREATESTRUCT*>(lParam)->lpCreateParams);
		assert(pWindow != 0);
		pWindow->Attach(hWnd);	// this attachment will be revoked at Create method
#ifdef _WIN64
		pWindow->SetWindowLongPtr(GWLP_USERDATA, reinterpret_cast<LONG_PTR>(pWindow));
#else
		pWindow->SetWindowLong(GWL_USERDATA, reinterpret_cast<long>(pWindow));
#endif /* _WIN64 */
		const LRESULT	res = pWindow->DispatchEvent(message, wParam, lParam);
		return res;
	} else {
#ifdef _WIN64
		Window*	pWindow = reinterpret_cast<Window*>(::GetWindowLongPtr(hWnd, GWLP_USERDATA));
#else
		Window*	pWindow = reinterpret_cast<Window*>(::GetWindowLong(hWnd, GWL_USERDATA));
#endif /* _WIN64 */
		if(message == WM_PAINT && pWindow != 0) {
			GDI::CPaintDC	dc(*pWindow);
			pWindow->OnPaint(dc);
			return true;
		} else
			return (pWindow != 0) ?
				pWindow->DispatchEvent(message, wParam, lParam) : ::DefWindowProc(hWnd, message, wParam, lParam);
	}
}

#undef UNCONST_THIS

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _M_WINDOW_H_ */

/* [EOF] */